use std::{
    process::{self, Command, Stdio},
    io::{BufReader, prelude::*},
    os::unix::net::{UnixStream, UnixListener},
    sync::RwLock,
    fs, thread
};
use rust_utils::config::Config;
use glob::glob;
use rust_utils::logging::LogLevel;
use crate::{client, mcserver, config, LOG};
use lazy_static::lazy_static;

mod instances;
use instances::Servers;

lazy_static! {
    static ref SERVERS: RwLock<Servers> = RwLock::new(Servers::new());
}

pub fn init() {
    LOG.line_basic("Starting up Minecraft Server Manager background process...", true);
    fs::remove_dir_all(&*mcserver::TMP_DIR).unwrap_or_default();
    fs::create_dir(&*mcserver::TMP_DIR).expect("Unable to create directiory!");
    let listener = UnixListener::bind(&*client::SOCKET).unwrap();
    LOG.line_basic("Startup complete!", true);

    // if Ctrl-C is pressed
    ctrlc::set_handler(move || {
        let mut servers = SERVERS.write().unwrap();
        LOG.line_basic("Shutting down!", true);
        servers.cleanup_dead();
        servers.shutdown_all();
        process::exit(0);
    }).expect("Error setting Ctrl-C handler");

    // clean up all dead servers in a background thread
    thread::spawn(move || {
        // sometimes Symeon, infinite loops can actually be useful!
        loop {
            SERVERS.write().unwrap().cleanup_dead();
        }
    });

    for request in listener.incoming() {
        if let Ok(stream) = request {
            thread::spawn(move || exec_req(stream));
        }
    }
}

fn exec_req(stream: UnixStream) {
    let mut out_stream = stream.try_clone().unwrap();
    let buffer = BufReader::new(&stream);
    let encoded: Vec<u8> = buffer.bytes().map(|r| r.unwrap_or(0)).collect();
    let request: client::ClientRequest = bincode::deserialize(&encoded[..]).expect("Error parsing request!");
    let cmd_t = request.cmd.clone();
    let cmd = cmd_t.as_str();
    let name = match request.args.get(0) {
        Some(s) => s,
        None => {
            report_error(&mut out_stream, "Server name is missing!");
            return;
        }
    };

    match cmd {
        // is this server running?
        "ir" => send_val(&mut out_stream, &SERVERS.read().unwrap().is_running(name)),

        // set up a server
        "sf" | "sv" => {
            if let Err(error) = setup_server(request, cmd == "sf") {
                report_error(&mut out_stream, &error);
            }
            else {
                report_success(&mut out_stream);
            }
        },

        // deletes a server's profile and directory
        "rms" => {
            if SERVERS.read().unwrap().is_running(name) {
                report_error(&mut out_stream, "Running servers cannot be deleted!");
                return;
            }
            
            let mut config = config::GlobalConfig::load();
            config.remove_profile(name);
            LOG.line_basic(format!("Deleted \"{}\" Minecraft server", name), true);
            report_success(&mut out_stream);
        },

        // launch a server
        "l" => {
            if SERVERS.read().unwrap().is_running(name) {
                report_error(&mut out_stream, format!("{} is already running!", name).as_str());
                return;
            }

            let config = config::GlobalConfig::load();
            let profile = match config.get_profile(&name) {
                Some(p) => p,
                None => {
                    let msg = format!("Server launch profile \"{}\" does not exist!", name);
                    report_error(&mut out_stream, &msg);
                    return;
                }
            };

            SERVERS.write().unwrap().add_server(mcserver::ServerInstance::launch(profile));
            report_success(&mut out_stream);
        },

        // send the console of a specified server back to the client
        "rqc" => {
            let numlines = match request.args.get(1) {
                Some(string) => {
                    match string.parse::<usize>() {
                        Ok(num) => num,
                        Err(error) => {
                            let msg = format!("Unable to parse number: {}", error);
                            report_error(&mut out_stream, &msg);
                            return;
                        }
                    }
                },

                None => {
                    report_error(&mut out_stream, "How many lines do you have?");
                    return;
                }
            };

            let console: String = match SERVERS.read().unwrap().get_console(name) {
                Ok(string) => string,

                Err(error) => {
                    report_error(&mut out_stream, &error);
                    return;
                }
            };

            let mut lines = console.lines();
            let cur_numlines = console.lines().count();

            if numlines > 0 {
                for _ in 0..numlines {
                    lines.next();
                }
            }

            let data = if cur_numlines == numlines {
                client::ConsoleLines::empty()
            }
            else {
                client::ConsoleLines{ lines: lines.map(|s| s.to_string()).collect() }
            };

            send_val(&mut out_stream, &data);
        },

        // execute a command on a specific server
        "xcmd" => {
            let cmd = match request.args.get(1) {
                Some(s) => s,
                None => {
                    report_error(&mut out_stream, "Command is missing!");
                    return;
                }
            };

            if let Err(error) = SERVERS.write().unwrap().exec_command(name, cmd) {
                report_error(&mut out_stream, &error);
            }
            else {
                report_success(&mut out_stream);
            }
        },

        // restart/shut down a specified server
        "rs" | "sds" => {
            let result = if cmd == "sds" {
                SERVERS.write().unwrap().shut_down_server(name);
                Ok(())
            }
            else {
                LOG.line_basic(format!("Restarting \"{}\" Minecraft server", name), true);
                SERVERS.write().unwrap().restart_server(name)
            };

            if let Err(error) = result {
                report_error(&mut out_stream, &error);
            }
            else {
                report_success(&mut out_stream);
            }
        },

        _ => report_error(&mut out_stream, "Invalid request!")
    }
}

fn setup_server(request: client::ClientRequest, is_forge: bool) -> Result<(), String> {
    let name = &request.args[0];
    let eula_res = request.args.get(1).ok_or("Unable to parse request!".to_string())?;
    let java = request.args.get(2).ok_or("Unable to parse request!".to_string())?;
    let version = request.args.get(3).ok_or("Unable to parse request!".to_string())?;
    let directory = format!("{}/{}", config::GlobalConfig::load().default_dir, name);
    let mut java_ops = mcserver::JavaOptions::default();
    java_ops.launch_command = java.clone();
    let new_profile = mcserver::ServerProfile::new(&name, java_ops, mcserver::Version::from_str(&version), &directory, is_forge, false);
    LOG.line_basic("Setting up new server", true);
    LOG.line_basic(format!("Name: {}", new_profile.name), true);
    
    if is_forge {
        LOG.line_basic(format!("Version: {} with Forge", new_profile.version), true);
    }
    else {
        LOG.line_basic(format!("Version: {}", new_profile.version), true);
    }

    LOG.line_basic(format!("Directory: {}", new_profile.directory), true);
    let profile = new_profile.clone();
    let threads = new_profile.setup_server(eula_res == "true");
    threads.0.join().unwrap().unwrap();

    // get the name of the forge installer JAR and run it
    if is_forge {
        threads.1.ok_or("I swear forge was supposed to be downloading in the background!".to_string())?.join().unwrap().unwrap();
        let pattern = format!("forge-{}-*-installer.jar", profile.version);
        let mut glob = glob(&pattern).expect("Failed to read glob pattern");
        let name = glob.nth(0).unwrap().unwrap().into_os_string().into_string().unwrap();
        let mut forge_setup_cmd = Command::new(profile.java_ops.launch_command);
        let file = fs::File::create("forge_install.log").expect("Unable to create file!");
        let file2 = fs::File::open("forge_install.log").expect("Unable to open file!");
        forge_setup_cmd.arg("-jar").arg(&name).arg("--installServer");
        LOG.line_basic("Installing Forge...", true);
        let proc = forge_setup_cmd.stdout(Stdio::from(file)).stderr(Stdio::from(file2)).spawn().expect("Failed to launch Forge installer!");
        proc.wait_with_output().unwrap();
        fs::remove_file(&name).unwrap_or(());
        fs::remove_file(format!("{}.log", name)).unwrap_or(());
        fs::remove_file("run.sh").unwrap_or(());
        fs::remove_file("run.bat").unwrap_or(());
        fs::remove_file("user_jvm_args.txt").unwrap_or(());
        LOG.line_basic("Forge setup is complete!", true)
    }

    LOG.line_basic(format!("Finished setting up \"{}\" Minecraft server", name), true);
    Ok(())
}

fn report_error(stream: &mut UnixStream, msg: &str) {
    let error = client::ServerError::from(msg.to_string());
    let encoded = bincode::serialize(&error).unwrap();
    stream.write_all(&encoded[..]).expect("Unable to write to socket!");
}

fn report_success(stream: &mut UnixStream) {
    let encoded = bincode::serialize(&()).unwrap();
    stream.write_all(&encoded[..]).expect("Unable to write to socket!");
}

fn send_val<V: serde::Serialize + for <'de> serde::Deserialize<'de> + ?Sized>(stream: &mut UnixStream, val: &V) {
    let encoded = bincode::serialize(val).unwrap();
    if let Err(why) = stream.write_all(&encoded[..]) {
        LOG.line(LogLevel::Warn, format!("Unable to write to socket: {}", why), false);
        LOG.line(LogLevel::Warn, "A console viewer may have crashed or been killed", false);
    };
}