use super::*;

pub struct Servers { servers: Vec<mcserver::ServerInstance> }

impl Servers {
    pub fn new() -> Servers {
        Servers {
            servers: vec![]
        }
    }

    pub fn is_running(&self, name: &str) -> bool {
        for server in &self.servers {
            if server.profile.name.as_str() == name {
                return true;
            }
        }
        false
    }

    pub fn get_console(&self, name: &str) -> Result<String, String> {
        let console_str = match fs::read_to_string(format!("{}/{}.console", &*mcserver::TMP_DIR, name)) {
            Ok(string) => string,
            Err(_) => return Err(format!("Unable to find {} Minecraft server! Is it running?", name))
        };

        Ok(console_str)
    }

    pub fn add_server(&mut self, instance: mcserver::ServerInstance) {
        self.servers.push(instance);
    }

    pub fn restart_server(&mut self, name: &str) -> Result<(), String> {
        let instance = match self.get_instance(name) {
            Some(i) => i,
            None => return Err(format!("Unable to find \"{}\" Minecraft server! Is it running?", name))
        };

        let profile = instance.profile.clone();
        self.servers.retain(|server| server.profile.name.as_str() != name);
        let new_instance = mcserver::ServerInstance::launch(profile);
        self.servers.push(new_instance);
        Ok(())
    }

    pub fn shut_down_server(&mut self, name: &str) {
        self.servers.retain(|server| server.profile.name.as_str() != name);
    }

    pub fn shutdown_all(&mut self) {
        self.servers.clear();
    }

    // drops all the handles to the dead server processes
    pub fn cleanup_dead(&mut self) {
        self.servers.retain(|server| !server.is_dead());
    }

    pub fn exec_command(&mut self, name: &str, cmd: &str) -> Result<(), String> {
        let instance = match self.get_instance_mut(name) {
            Some(i) => i,
            None => return Err(format!("Unable to find \"{}\" Minecraft server! Is it running?", name))
        };

        let stdin = instance.process.stdin.as_mut().unwrap();
        writeln!(stdin,"{}", cmd).expect("Unable to execute command!");
        Ok(())
    }

    // mutable ref to a server instance
    fn get_instance_mut(&mut self, name: &str) -> Option<&mut mcserver::ServerInstance> {
        if self.servers.is_empty() {
            return None;
        }

        for server in &mut self.servers {
            if server.profile.name.as_str() == name {
                return Some(server);
            }
        }

        None
    }

    // immutable ref to a server instance
    fn get_instance(&mut self, name: &str) -> Option<&mcserver::ServerInstance> {
        if self.servers.is_empty() {
            return None;
        }

        for server in &self.servers {
            if server.profile.name.as_str() == name {
                return Some(server);
            }
        }

        None
    }
}