use crate::mcserver::*;
use std::{env, fs};
use serde_derive::{Deserialize, Serialize};
use rust_utils::config::Config;

#[derive(Deserialize, Serialize)]
pub struct GlobalConfig {
    pub default_dir: String,
    pub profiles: Vec<ServerProfile>
}

impl GlobalConfig {
    pub fn get_profile(&self, name: &str) -> Option<ServerProfile> {
        for profile in &self.profiles {
            if &profile.name == name {
                return Some(profile.clone());
            }
        }
        None
    }

    pub fn add_profile(&mut self, profile: ServerProfile) {
        self.profiles.push(profile);
        self.save().expect("Unable to save config!");
    }

    pub fn remove_profile(&mut self, name: &str) {
        let profile = match self.get_profile(name) {
            Some(p) => p,
            None => return
        };

        fs::remove_dir_all(profile.directory).unwrap_or(());
        self.profiles.retain(|profile| profile.name.as_str() != name);
        self.save().expect("Unable to save config!");
    }
}

impl Default for GlobalConfig {
    fn default() -> GlobalConfig {
        GlobalConfig {
            default_dir: format!("{}", env::var("HOME").expect("Where the hell is your home folder?")),
            profiles: Vec::new()
        }
    }
}

impl Config for GlobalConfig {
    const FILE_NAME: &'static str = "config.toml";
    fn get_save_dir() -> String {
        format!("{}/.config/serverman/",env::var("HOME").expect("Where the hell is your home folder?"))
    }
}