use serde_derive::{Deserialize, Serialize};
use std::{
    mem,
    os::unix::net::UnixStream,
    net::Shutdown,
    io::{prelude::*, BufReader},
    fmt
};
use lazy_static::lazy_static;
use crate::mcserver;

lazy_static! {
    pub static ref SOCKET: String = format!("{}/socket", *mcserver::TMP_DIR);
}

#[derive(Serialize, Deserialize)]
pub struct ConsoleLines {
    pub lines: Vec<String>
}

impl ConsoleLines {
    pub fn empty() -> ConsoleLines {
        ConsoleLines {
            lines: vec![]
        }
    }

    pub fn len(&self) -> usize {
        self.lines.len()
    }
}

#[derive(Serialize, Deserialize, Clone)]
pub struct ClientRequest {
    pub cmd: String,
    pub args: Vec<String>
}

impl ClientRequest {
    fn new(cmd: &str, args: &[&str]) -> ClientRequest {
        let new_args: Vec<String> = args.to_vec().iter().map(|s| s.to_string()).collect();
        ClientRequest {
            cmd: cmd.to_string(),
            args: new_args
        }
    }

    pub fn setup_server(name: &str, eula: bool, java: &str, version: &str, forge: bool) -> ClientRequest {
        let eula_str = eula.to_string();
        Self::new(if forge { "sf" } else { "sv" }, &[name, &eula_str, java, version])
    }
    
    pub fn is_running(name: &str) -> ClientRequest { Self::new("ir", &[name]) }
    pub fn remove_server(name: &str) -> ClientRequest { Self::new("rms", &[name]) }
    pub fn launch_server(name: &str) -> ClientRequest { Self::new("l", &[name]) }
    pub fn restart_server(name: &str) -> ClientRequest { Self::new("rs", &[name]) }
    pub fn stop_server(name: &str) -> ClientRequest { Self::new("sds", &[name]) }
    pub fn run_command(name: &str, cmd: &str) -> ClientRequest { Self::new("xcmd", &[name, cmd]) }
    pub fn get_console(name: &str, num_lines: usize) -> ClientRequest { Self::new("rqc", &[name, num_lines.to_string().as_str()]) }

    pub fn send<T>(self) -> ServerResponse<T>
        where T: serde::Serialize + for <'de> serde::Deserialize<'de>
    {
        let encoded = bincode::serialize(&self).unwrap();
        let mut response = ServerResponse {
            result: ServerResult::Success,
            body: None
        };

        let mut socket = match UnixStream::connect(&*SOCKET) {
            Ok(s) => s,
            Err(error) => {
                response.result = ServerResult::Fail(ServerError::from(format!("Unable to connect to socket: {}", error)));
                return response;
            }
        };

        socket.write_all(&encoded[..]).expect("Unable to write to socket!");
        socket.shutdown(Shutdown::Write).unwrap();
        let buffer = BufReader::new(&socket);
        let encoded: Vec<u8> = buffer.bytes().map(|r| r.unwrap_or(0)).collect();

        if mem::size_of::<T>() == 0 {
            if encoded.len() > 1 {
                match bincode::deserialize::<ServerError>(&encoded[..]) {
                    Ok(se) => response.result = ServerResult::Fail(se),
                    Err(error) => response.result = ServerResult::Fail(ServerError::from(format!("Unable to parse response body: {}", error)))
                };
            }
        }
        else {
            match bincode::deserialize::<T>(&encoded[..]) {
                Ok(body) => response.body = Some(body),
                Err(_) => {
                    match bincode::deserialize::<ServerError>(&encoded[..]) {
                        Ok(se) => response.result = ServerResult::Fail(se),
                        Err(error) => response.result = ServerResult::Fail(ServerError::from(format!("Unable to parse response body: {}", error)))
                    }
                }
            };
        }   
        response
    }

    pub fn send_empty(self) -> ServerResponse<()> {
        self.send()
    }
}

#[must_use]
pub enum ServerResult {
    Success,
    Fail(ServerError)
}

#[derive(Serialize, Deserialize)]
pub struct ServerError {
    pub why: String
}

impl From<String> for ServerError {
    fn from(string: String) -> ServerError {
        ServerError {
            why: string
        }
    }
}

impl fmt::Display for ServerError {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f, "{}", self.why)
    }
}

pub struct ServerResponse<T: serde::Serialize + for <'de> serde::Deserialize<'de>> {
    pub result: ServerResult,
    pub body: Option<T>
}