use crate::{client::*, config, tui::console};
use rust_utils::config::Config;
use std::{io, env};
use cursive::{Cursive, CursiveExt};
use colored::Colorize;

pub fn parse_args(args: Vec<String>) {
    if args.len() == 1 {
        println!("Please specify an option!");
        return;
    }

    let cmd = args[1].as_str();
    match cmd {
        "setup" => setup_server(false),
        "setup-forge" => setup_server(true),
        "launch" => launch_server(args.get(2)),
        "restart" | "stop" => shutdown_server(args.get(2), cmd == "restart"),
        "console" => {
            if let None = args.get(2) {
                println!("Please specify a server name!");
                return;
            }
            let mut root = Cursive::new();
            console::show(args.get(2).unwrap(), &mut root, true);
            root.run()
        },
        "remove" => remove_server(args.get(2)),
        "config" => get_server_cfg(args.get(2), args.get(3), args.get(4), args.get(5)),
        "servers" => {
            let config = config::GlobalConfig::load();
            println!("{}\n", "Known Minecraft servers:".bright_white());

            for server in config.profiles {
                println!("{}", server);
            }
        },

        "version" | "-v" => {
            println!("Minecraft Server Manager");
            println!("Version {}", env!("CARGO_PKG_VERSION"));
        },

        "help" | "-h" => {
            println!("{}","Minecraft Server Manager Help:".bright_white());
            println!("{}", "Subcommands:".bright_white());
            println!("    {} set up a new Minecraft server","setup:".bright_white());
            println!("    {} set up a new server with Minecraft Forge installed","setup-forge:".bright_white());
            println!("    {} Launch a specified server", "launch <server name>:".bright_white());
            println!("    {} Restart a specified server", "restart <name>:".bright_white());
            println!("    {} Show the console of a specified server", "console <name>:".bright_white());
            println!("    {} Deletes a specified server. Please use this carefully because it cannot be undone!", "remove <name>:".bright_white());
            println!("    {} Get or set the configuration (server.properties) of a specified server", "config <name> [option] [key] [new value]:".bright_white());
            println!("    {} List all known Minecraft servers", "servers:".bright_white());
            println!("    {} show this message again", "help:".bright_white());
        },
        _ => println!("Invalid option!")
    }
}

fn get_server_cfg(name: Option<&String>, subcmd: Option<&String>, key: Option<&String>, new_val: Option<&String>) {
    if let None = name {
        println!("Please specify a server name!");
        return;
    }

    if name.unwrap() == "keys" {
        println!("{}\n\
        allow-flight\n\
        allow-nether\n\
        broadcast-console-to-ops\n\
        broadcast-rcon-to-ops\n\
        difficulty\n\
        enable-command-block\n\
        enable-jmx-monitoring\n\
        enable-query\n\
        enable-rcon\n\
        enable-status\n\
        enforce-whitelist\n\
        entity-broadcast-range-percentage\n\
        force-gamemode\n\
        function-permission-level\n\
        gamemode\n\
        generate-structures\n\
        generator-settings\n\
        hardcore\n\
        level-name\n\
        level-seed\n\
        level-type\n\
        max-build-height\n\
        max-players\n\
        max-tick-time\n\
        max-world-size\n\
        motd\n\
        network-compression-threshold\n\
        online-mode\n\
        op-permission-level\n\
        player-idle-timeout\n\
        prevent-proxy-connections\n\
        pvp\n\
        query.port\n\
        rate-limit\n\
        rcon.password\n\
        rcon.port\n\
        resource-pack\n\
        resource-pack-sha1\n\
        server-ip\n\
        server-port\n\
        snooper-enabled\n\
        spawn-animals\n\
        spawn-monsters\n\
        spawn-npcs\n\
        spawn-protection\n\
        sync-chunk-writes\n\
        text-filtering-config\n\
        use-native-transport\n\
        view-distance\n\
        white-list", "List of all server configuration (server.properties) keys:".bright_white());
        return;
    }
    else if name.unwrap() == "help" {
        println!("{}", "Minecraft server configuration help:".bright_white());
        println!("{}", "Subcommands:".bright_white());
        println!("    {} sets a config key to a new value", "<name> set <key name> <new value>:".bright_white());
        println!("    {} get a config key value", "<name> get <key name>:".bright_white());
        println!("    {} list all config keys", "keys:".bright_white());
        println!("    {} show this message again", "help:".bright_white());
        return;
    }

    let config = config::GlobalConfig::load();

    let profile = match config.get_profile(name.unwrap()) {
        Some(server) => server,
        None => {
            println!("Error retrieving the configuration of \"{}\" Minecraft server: Server does not exist!", name.unwrap());
            return;
        }
    };

    env::set_current_dir(&profile.directory).expect("Where is the directory for the current server?");
    let mut properties = profile.get_properties();
    if let Some(s) = subcmd {
        if let None = key {
            println!("Please specify a config key!");
            return;
        }

        if s.as_str() == "get" {
            let val = properties.get_value(key.unwrap());

            if let None = val {
                println!("Config key does not exist!");
                return;
            }

            let (label, raw_val) = val.unwrap();
            let form_val = match key.unwrap().as_str() {
                "entity-broadcast-range-percentage" => format!("{}%", raw_val),
                "spawn-protection" | "view-distance" => format!("{} Chunks", raw_val),
                _ => raw_val.to_string()
            };

            println!("{} {}", label.bright_white(), form_val);
        }
        else if s.as_str() == "set" {
            if let None = new_val {
                println!("Please specify a new value for the config key!")
            }
            if let Err(error) = properties.set_value(key.unwrap(), new_val.unwrap(), profile.version) {
                println!("{}", error);
            };

            ClientRequest::run_command(name.unwrap(), "reload").send_empty();
        }

        else if s.as_str() == "help" {

        }
        else {
            println!("Invalid option!");
            return;
        }
    }
    else {
        println!("{}{}{}", "Configuration for \"".bright_white(), name.unwrap().bright_white(), "\" Minecraft server:".bright_white());
        println!("{}", properties);
    }
}

fn remove_server(name: Option<&String>) {
    if let None = name {
        println!("Please specify a server name!");
        return;
    }

    let response = ClientRequest::remove_server(name.unwrap()).send_empty();

    match response.result {
        ServerResult::Success => println!("Removed \"{}\" Minecraft server", name.unwrap()),
        ServerResult::Fail(error) => println!("Error removing \"{}\" Minecraft server: {}", name.unwrap(), error)
    }
}

fn launch_server(name: Option<&String>) {
    // because of this, any calls to Option::unwrap() should not panic
    if let None = name {
        println!("Please specify a server name!");
        return;
    }

    let response = ClientRequest::launch_server(name.unwrap()).send_empty();

    match response.result {
        ServerResult::Success => println!("Successfully launched \"{}\" Minecraft server", name.unwrap()),
        ServerResult::Fail(error) => println!("Error launching \"{}\" Minecraft server!\n{}", name.unwrap(), error)
    }
}

fn shutdown_server(name: Option<&String>, restart: bool) {
    if let None = name {
        println!("Please specify a server name!");
        return;
    }

    let response = if restart {
        ClientRequest::restart_server(name.unwrap()).send_empty()
    }
    else {
        ClientRequest::stop_server(name.unwrap()).send_empty()
    };

    match response.result {
        ServerResult::Success if restart => println!("Successfully restarted \"{}\" Minecraft server", name.unwrap()),
        ServerResult::Success => println!("Successfully shut down \"{}\" Minecraft server", name.unwrap()),
        ServerResult::Fail(error) => println!("Unable to {} \"{}\" Minecraft server: {}", if restart { "restart" } else { "shut down" }, name.unwrap(), error)
    }
}

fn setup_server(is_forge: bool) {
    let mut eula_res = String::new();
    let mut java = String::new();
    let mut name = String::new();
    let mut version = String::new();

    println!("Do you agree to Mojang's EULA? https://account.mojang.com/documents/minecraft_eula (Please read it)(Y/n)");
    io::stdin().read_line(&mut eula_res).unwrap();

    println!("Name:");
    io::stdin().read_line(&mut name).unwrap();

    println!("Java launch command:");
    io::stdin().read_line(&mut java).unwrap();

    println!("Version:");
    io::stdin().read_line(&mut version).unwrap();

    name.retain(|c| c != '\n');
    java.retain(|c| c != '\n');
    version.retain(|c| c != '\n');
    eula_res.retain(|c| c != '\n');
    let eula = eula_res.to_lowercase().as_str() == "y" || eula_res.is_empty();
    let response = ClientRequest::setup_server(&name, eula, &java, &version, is_forge).send_empty();

    match response.result {
        ServerResult::Success => println!("Successfully set up \"{}\" Minecraft server", name),
        ServerResult::Fail(error) => println!("Error setting up \"{}\" Minecraft server: {}", name, error)
    }
}