use serde_test::Token;

#[macro_use]
mod utils;

use utils::assert_all_tokens;

#[test]
fn test_node() {
    assert_all_tokens(
        "node",
        // &[node_tokens! {
        //     name: "node",
        //     values: [],
        //     properties: {},
        //     children: []
        // }],
        &[&[
            Token::Struct {
                name: "Node",
                len: 4,
            },
            Token::Str("name"),
            Token::Str("node"),
            Token::Str("values"),
            Token::Seq { len: Some(0) },
            Token::SeqEnd,
            Token::Str("properties"),
            Token::Map { len: Some(0) },
            Token::MapEnd,
            Token::Str("children"),
            Token::Seq { len: Some(0) },
            Token::SeqEnd,
            Token::StructEnd,
        ]],
    );
}

#[test]
fn test_node_value_key() {
    assert_all_tokens(
        "node 1 key=true",
        // &[node_tokens! {
        //     name: "node",
        //     values: [1],
        //     properties: { "key" => true },
        // }],
        &[&[
            Token::Struct {
                name: "Node",
                len: 4,
            },
            Token::Str("name"),
            Token::Str("node"),
            Token::Str("values"),
            Token::Seq { len: Some(1) },
            Token::I64(1),
            Token::SeqEnd,
            Token::Str("properties"),
            Token::Map { len: Some(1) },
            Token::Str("key"),
            Token::Bool(true),
            Token::MapEnd,
            Token::Str("children"),
            Token::Seq { len: Some(0) },
            Token::SeqEnd,
            Token::StructEnd,
        ]],
    );
}

#[test]
fn test_simple_children() {
    assert_all_tokens(
        "parent 1 { child 2; }",
        // &[node_tokens! {
        //     name: "parent",
        //     values: [1],
        //     children: [
        //         { name: "child", values: [2] }
        //     ]
        // }],
        &[&[
            Token::Struct {
                name: "Node",
                len: 4,
            },
            Token::Str("name"),
            Token::Str("parent"),
            Token::Str("values"),
            Token::Seq { len: Some(1) },
            Token::I64(1),
            Token::SeqEnd,
            Token::Str("properties"),
            Token::Map { len: Some(0) },
            Token::MapEnd,
            Token::Str("children"),
            Token::Seq { len: Some(1) },
            // child
            Token::Struct {
                name: "Node",
                len: 4,
            },
            Token::Str("name"),
            Token::Str("child"),
            Token::Str("values"),
            Token::Seq { len: Some(1) },
            Token::I64(2),
            Token::SeqEnd,
            Token::Str("properties"),
            Token::Map { len: Some(0) },
            Token::MapEnd,
            Token::Str("children"),
            Token::Seq { len: Some(0) },
            Token::SeqEnd,
            Token::StructEnd,
            // end child
            Token::SeqEnd,
            Token::StructEnd,
        ]],
    );
}

#[test]
#[ignore]
fn test_example() {
    let document = r#"
author "Alex Monad" email="alex@example.com" active=true

contents {
  section "First section" {
    paragraph "This is the first paragraph"
    paragraph "This is the second paragraph"
  }
}

// unicode! comments!
π 3.14159
"#;
    assert_all_tokens(
        document,
        &[
            node_tokens! {
                name: "author",
                values: ["Alex Monad"],
                properties: {
                    "email" => "alex@example.com",
                    "active" => true,
                },
            },
            &[
                // contents
                Token::Struct {
                    name: "Node",
                    len: 4,
                },
                Token::Str("name"),
                Token::Str("contents"),
                Token::Str("values"),
                Token::Seq { len: None },
                Token::SeqEnd,
                Token::Str("properties"),
                Token::Map { len: None },
                Token::MapEnd,
                Token::Str("children"),
                Token::Seq { len: Some(1) },
                // section
                Token::Struct {
                    name: "Node",
                    len: 4,
                },
                Token::Str("name"),
                Token::Str("section"),
                Token::Str("values"),
                Token::Seq { len: Some(1) },
                Token::Str("First section"),
                Token::SeqEnd,
                Token::Str("properties"),
                Token::Map { len: None },
                Token::MapEnd,
                Token::Str("children"),
                Token::Seq { len: Some(2) },
                Token::Struct {
                    name: "Node",
                    len: 4,
                },
                Token::Str("name"),
                Token::Str("contents"),
                Token::Str("values"),
                Token::Seq { len: None },
                Token::SeqEnd,
                Token::Str("properties"),
                Token::Map { len: None },
                Token::MapEnd,
                Token::Struct {
                    name: "Node",
                    len: 4,
                },
                Token::Str("name"),
                Token::Str("contents"),
                Token::Str("values"),
                Token::Seq { len: None },
                Token::SeqEnd,
                Token::Str("properties"),
                Token::Map { len: None },
                Token::MapEnd,
                Token::SeqEnd,
                Token::StructEnd, // section
                Token::SeqEnd,
                Token::StructEnd, //contents
            ],
        ],
    );
}
