use serde::{ser, de};
use serde_derive::{Serialize, Deserialize};

use crate::error::{Error, Result};
use crate::types::{Read, Write};
use de::Visitor;

/// A variable-length integer which is simply truncated by skipping high zero bytes.
///
/// A BOLT structure may have at most one of these, and it must come at the end.
#[cfg_attr(test, derive(PartialEq, Debug))]
#[derive(Serialize, Deserialize)]
pub struct HighZeroBytesDroppedVarInt<T>(pub T);

impl HighZeroBytesDroppedVarInt<()> {
	pub(crate) const NAME: &'static str = "HighZeroBytesDroppedVarInt";
}

macro_rules! impl_writeable_primitive {
	($val_type:ty, $len: expr) => {
		impl HighZeroBytesDroppedVarInt<$val_type> {
			/// Write
			#[inline]
			pub fn write<W: Write>(&self, writer: &mut W) -> $crate::Result<()> {
				// Skip any full leading 0 bytes when writing (in BE):
				writer.write_all(&self.0.to_be_bytes()[(self.0.leading_zeros()/8) as usize..$len])
			}

		}

		impl HighZeroBytesDroppedVarInt<$val_type> {
			/// Read
			#[inline]
			pub fn read<R: Read>(reader: &mut R) -> $crate::Result<HighZeroBytesDroppedVarInt<$val_type>> {
				// We need to accept short reads (read_len == 0) as "EOF" and handle them as simply
				// the high bytes being dropped. To do so, we start reading into the middle of buf
				// and then convert the appropriate number of bytes with extra high bytes out of
				// buf.
				let mut buf = [0; $len*2];
				let mut read_len = reader.read(&mut buf[$len..])?;
				let mut total_read_len = read_len;
				while read_len != 0 && total_read_len != $len {
					read_len = reader.read(&mut buf[($len + total_read_len)..])?;
					total_read_len += read_len;
				}
				if total_read_len == 0 || buf[$len] != 0 {
					let first_byte = $len - ($len - total_read_len);
					let mut bytes = [0; $len];
					bytes.copy_from_slice(&buf[first_byte..first_byte + $len]);
					Ok(HighZeroBytesDroppedVarInt(<$val_type>::from_be_bytes(bytes)))
				} else {
					// If the encoding had extra zero bytes, return a failure even though we know
					// what they meant (as the TLV test vectors require this)
					Err($crate::Error::InvalidVarInt)
				}
			}
		}
	}
}

impl_writeable_primitive!(u64, 8);
impl_writeable_primitive!(u32, 4);
impl_writeable_primitive!(u16, 2);

pub(crate) struct HighZeroBytesDroppedVarIntSerializer<'a, W: Write> {
	pub(crate) writer: &'a mut W,
}

impl<'a, W: Write> ser::Serializer for HighZeroBytesDroppedVarIntSerializer<'a, W> {
	type Ok = ();
	type Error = Error;
	type SerializeSeq = Self;
	type SerializeTuple = Self;
	type SerializeTupleStruct = Self;
	type SerializeTupleVariant = Self;
	type SerializeMap = Self;
	type SerializeStruct = Self;
	type SerializeStructVariant = Self;

	fn serialize_bool(self, _v: bool) -> Result<Self::Ok> {
		unimplemented!()
	}

	fn serialize_i8(self, _v: i8) -> Result<Self::Ok> {
		unimplemented!()
	}

	fn serialize_i16(self, _v: i16) -> Result<Self::Ok> {
		unimplemented!()
	}

	fn serialize_i32(self, _v: i32) -> Result<Self::Ok> {
		unimplemented!()
	}

	fn serialize_i64(self, _v: i64) -> Result<Self::Ok> {
		unimplemented!()
	}

	fn serialize_u8(self, _v: u8) -> Result<Self::Ok> {
		unimplemented!()
	}

	fn serialize_u16(self, v: u16) -> Result<Self::Ok> {
		HighZeroBytesDroppedVarInt(v).write(self.writer)
	}

	fn serialize_u32(self, v: u32) -> Result<Self::Ok> {
		HighZeroBytesDroppedVarInt(v).write(self.writer)
	}

	fn serialize_u64(self, v: u64) -> Result<Self::Ok> {
		HighZeroBytesDroppedVarInt(v).write(self.writer)
	}

	fn serialize_f32(self, _v: f32) -> Result<Self::Ok> {
		unimplemented!()
	}

	fn serialize_f64(self, _v: f64) -> Result<Self::Ok> {
		unimplemented!()
	}

	fn serialize_char(self, _v: char) -> Result<Self::Ok> {
		unimplemented!()
	}

	fn serialize_str(self, _v: &str) -> Result<Self::Ok> {
		unimplemented!()
	}

	fn serialize_bytes(self, _v: &[u8]) -> Result<Self::Ok> {
		unimplemented!()
	}

	fn serialize_none(self) -> Result<Self::Ok> {
		unimplemented!()
	}

	fn serialize_some<T: ?Sized>(self, _value: &T) -> Result<Self::Ok> where T: ser::Serialize {
		unimplemented!()
	}

	fn serialize_unit(self) -> Result<Self::Ok> {
		unimplemented!()
	}

	fn serialize_unit_struct(self, _name: &'static str) -> Result<Self::Ok> {
		unimplemented!()
	}

	fn serialize_unit_variant(self, _name: &'static str, _variant_index: u32, _variant: &'static str) -> Result<Self::Ok> {
		unimplemented!()
	}

	fn serialize_newtype_struct<T: ?Sized>(self, _name: &'static str, _value: &T) -> Result<Self::Ok> where T: ser::Serialize {
		unimplemented!()
	}

	fn serialize_newtype_variant<T: ?Sized>(self, _name: &'static str, _variant_index: u32, _variant: &'static str, _value: &T) -> Result<Self::Ok> where T: ser::Serialize {
		unimplemented!()
	}

	fn serialize_seq(self, _len: Option<usize>) -> Result<Self::SerializeSeq> {
		unimplemented!()
	}

	fn serialize_tuple(self, _len: usize) -> Result<Self::SerializeTuple> {
		unimplemented!()
	}

	fn serialize_tuple_struct(self, _name: &'static str, _len: usize) -> Result<Self::SerializeTupleStruct> {
		unimplemented!()
	}

	fn serialize_tuple_variant(self, _name: &'static str, _variant_index: u32, _variant: &'static str, _len: usize) -> Result<Self::SerializeTupleVariant> {
		unimplemented!()
	}

	fn serialize_map(self, _len: Option<usize>) -> Result<Self::SerializeMap> {
		unimplemented!()
	}

	fn serialize_struct(self, _name: &'static str, _len: usize) -> Result<Self::SerializeStruct> {
		unimplemented!()
	}

	fn serialize_struct_variant(self, _name: &'static str, _variant_index: u32, _variant: &'static str, _len: usize) -> Result<Self::SerializeStructVariant> {
		unimplemented!()
	}
}

impl<'a, W: Write> ser::SerializeSeq for HighZeroBytesDroppedVarIntSerializer<'a, W> {
	type Ok = ();
	type Error = Error;

	fn serialize_element<T>(&mut self, _value: &T) -> Result<()>
		where T: ?Sized + ser::Serialize,
	{
		unimplemented!();
	}

	fn end(self) -> Result<()> {
		Ok(())
	}
}

impl<'a, W: Write> ser::SerializeTuple for HighZeroBytesDroppedVarIntSerializer<'a, W> {
	type Ok = ();
	type Error = Error;

	fn serialize_element<T>(&mut self, _value: &T) -> Result<()>
		where T: ?Sized + ser::Serialize,
	{
		unimplemented!();
	}

	fn end(self) -> Result<()> {
		unimplemented!();
	}
}

impl<'a, W: Write> ser::SerializeTupleStruct for HighZeroBytesDroppedVarIntSerializer<'a, W> {
	type Ok = ();
	type Error = Error;

	fn serialize_field<T>(&mut self, _value: &T) -> Result<()>
		where T: ?Sized + ser::Serialize,
	{
		unimplemented!();
	}

	fn end(self) -> Result<()> {
		unimplemented!();
	}
}

impl<'a, W: Write> ser::SerializeTupleVariant for HighZeroBytesDroppedVarIntSerializer<'a, W> {
	type Ok = ();
	type Error = Error;

	fn serialize_field<T>(&mut self, _value: &T) -> Result<()>
		where T: ?Sized + ser::Serialize,
	{
		unimplemented!();
	}

	fn end(self) -> Result<()> {
		unimplemented!();
	}
}

impl<'a, W: Write> ser::SerializeMap for HighZeroBytesDroppedVarIntSerializer<'a, W> {
	type Ok = ();
	type Error = Error;

	fn serialize_key<T>(&mut self, _key: &T) -> Result<()>
		where T: ?Sized + ser::Serialize,
	{
		unimplemented!();
	}

	fn serialize_value<T>(&mut self, _value: &T) -> Result<()>
		where T: ?Sized + ser::Serialize,
	{
		unimplemented!();
	}

	fn end(self) -> Result<()> {
		unimplemented!();
	}
}

impl<'a, W: Write> ser::SerializeStruct for HighZeroBytesDroppedVarIntSerializer<'a, W> {
	type Ok = ();
	type Error = Error;

	fn serialize_field<T>(&mut self, __key: &'static str, _value: &T) -> Result<()>
		where T: ?Sized + ser::Serialize,
	{
		unimplemented!();
	}

	fn end(self) -> Result<()> {
		unimplemented!();
	}
}

impl<'a, W: Write> ser::SerializeStructVariant for HighZeroBytesDroppedVarIntSerializer<'a, W> {
	type Ok = ();
	type Error = Error;

	fn serialize_field<T>(&mut self, _key: &'static str, _value: &T) -> Result<()>
		where T: ?Sized + ser::Serialize,
	{
		unimplemented!();
	}

	fn end(self) -> Result<()> {
		unimplemented!();
	}
}

pub(crate) struct HighZeroBytesDroppedVarIntDeserializer<'a, R: Read> {
	pub(crate) reader: &'a mut R,
}

impl<'de, 'a, R: Read> de::Deserializer<'de> for HighZeroBytesDroppedVarIntDeserializer<'a, R> {
	type Error = Error;

	fn deserialize_any<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_bool<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_i8<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_i16<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_i32<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_i64<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_u8<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_u16<V>(self, visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		let v = HighZeroBytesDroppedVarInt::<u16>::read(self.reader)?;
		visitor.visit_u16(v.0)
	}

	fn deserialize_u32<V>(self, visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		let v = HighZeroBytesDroppedVarInt::<u32>::read(self.reader)?;
		visitor.visit_u32(v.0)
	}

	fn deserialize_u64<V>(self, visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		let v = HighZeroBytesDroppedVarInt::<u64>::read(self.reader)?;
		visitor.visit_u64(v.0)
	}

	fn deserialize_f32<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_f64<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_char<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_str<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_string<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_bytes<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_byte_buf<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_option<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_unit<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_unit_struct<V>(self, _name: &'static str, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_newtype_struct<V>(self, _name: &'static str, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_seq<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_tuple<V>(self, _len: usize, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_tuple_struct<V>(self, _name: &'static str, _len: usize, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_map<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_struct<V>(self, _name: &'static str, _fields: &'static [&'static str], _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_enum<V>(self, _name: &'static str, _variants: &'static [&'static str], _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_identifier<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}

	fn deserialize_ignored_any<V>(self, _visitor: V) -> Result<V::Value> where V: Visitor<'de> {
		unimplemented!()
	}
}

#[cfg(test)]
mod tests {
	use hex::{decode, encode};

	use crate::Error;
	use super::HighZeroBytesDroppedVarInt;

	#[test]
	fn test_tu32() {
		let v = HighZeroBytesDroppedVarInt(0x1234u32);
		let mut encoded = Vec::new();
		v.write(&mut &mut encoded).unwrap();
		assert_eq!("1234", encode(&encoded));
		let dv = HighZeroBytesDroppedVarInt::<u32>::read(&mut encoded).unwrap();
		assert_eq!(dv, v);
	}

	#[test]
	fn test_tu32_non_standard() {
		let mut encoded = decode("123400").unwrap();
		assert_eq!(HighZeroBytesDroppedVarInt::<u32>::read(&mut encoded).unwrap().0, 0x123400u32);
		let mut encoded = decode("001234").unwrap();
		assert_eq!(HighZeroBytesDroppedVarInt::<u32>::read(&mut encoded), Err(Error::InvalidVarInt));
	}
}
