use crate::error::{Error, Result};

/// A no-std replacement for the std::io Write trait
pub trait Write {
	/// The type of error returned when a write operation fails.
	type Error: Into<Error>;

	/// Attempts to write an entire buffer into this write.
	fn write_all(&mut self, buf: &[u8]) -> Result<()>;
}

/// A no-std replacement for the std::io Read trait
pub trait Read {
	/// The type of error returned when a write operation fails.
	type Error: Into<Error>;

	/// Attempts to read into the buffer, returning the number of bytes read
	fn read(&mut self, dest: &mut [u8]) -> Result<usize>;
}

/// A BOLT compatible variable-size compact integer representation
pub struct BigSize(pub u64);

impl BigSize {
	/// Write
	#[inline]
	pub fn write<W: Write>(&self, writer: &mut W) -> Result<()> {
		match self.0 {
			0..=0xFC => {
				writer.write_all(&[self.0 as u8])
			},
			0xFD..=0xFFFF => {
				writer.write_all(&[0xFDu8])?;
				writer.write_all(&(self.0 as u16).to_be_bytes())
			},
			0x10000..=0xFFFFFFFF => {
				writer.write_all(&[0xFEu8])?;
				writer.write_all(&(self.0 as u32).to_be_bytes())
			},
			_ => {
				writer.write_all(&[0xFFu8])?;
				writer.write_all(&(self.0 as u64).to_be_bytes())
			},
		}
	}
}
