use alloc::vec::Vec;

use serde::de::{self, Deserialize, DeserializeSeed, Visitor};

use crate::error::{Error, Result};
use crate::types::Read;

impl Read for Vec<u8> {
    type Error = Error;

    fn read(&mut self, dest: &mut [u8]) -> Result<usize> {
        if dest.len() >= self.len() {
            dest[0..self.len()].copy_from_slice(self);
            let n = self.len();
            self.drain(0..self.len());
            Ok(n)
        } else {
            dest.copy_from_slice(&self[0..dest.len()]);
            self.drain(0..dest.len());
            Ok(dest.len())
        }
    }
}


/// BOLT deserializer
pub struct Deserializer<'de, R: Read> {
    reader: &'de mut R,
}

impl<'de, R: Read> Deserializer<'de, R> {
    /// Deserialize from reader
    pub fn from_reader(reader: &'de mut R) -> Self {
        Deserializer { reader }
    }
}

/// Deserialize from Vec
pub fn from_vec<'a, T>(s: &'a mut Vec<u8>) -> Result<T>
where
    T: Deserialize<'a>,
{
    let mut deserializer = Deserializer::from_reader(s);
    let t = T::deserialize(&mut deserializer)?;
    if deserializer.reader.is_empty() {
        Ok(t)
    } else {
        Err(Error::TrailingBytes)
    }
}

impl<'de, R: Read> Deserializer<'de, R> {
}

impl<'de, 'a, R: Read> de::Deserializer<'de> for &'a mut Deserializer<'de, R> {
    type Error = Error;

    fn deserialize_any<V>(self, _visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        unimplemented!()
    }

    fn deserialize_bool<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        let result = self.read_bool()?;
        visitor.visit_bool(result)
    }

    fn deserialize_i8<V>(self, _visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        unimplemented!()
    }

    fn deserialize_i16<V>(self, _visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        unimplemented!()
    }

    fn deserialize_i32<V>(self, _visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        unimplemented!()
    }

    fn deserialize_i64<V>(self, _visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        unimplemented!()
    }

    fn deserialize_u8<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        let mut result = [0u8; 1];
        let n = self.reader.read(&mut result)?;
        if n < result.len() {
            return Err(Error::Eof)
        }
        visitor.visit_u8(result[0])
    }

    fn deserialize_u16<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        let mut result = [0u8; 2];
        let n = self.reader.read(&mut result)?;
        if n < result.len() {
            return Err(Error::Eof)
        }
        visitor.visit_u16(u16::from_be_bytes(result))
    }

    fn deserialize_u32<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        let mut result = [0u8; 4];
        let n = self.reader.read(&mut result)?;
        if n < result.len() {
            return Err(Error::Eof)
        }
        visitor.visit_u32(u32::from_be_bytes(result))
    }

    fn deserialize_u64<V>(self, _visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        unimplemented!()
    }

    fn deserialize_f32<V>(self, _visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        unimplemented!()
    }

    fn deserialize_f64<V>(self, _visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        unimplemented!()
    }

    fn deserialize_char<V>(self, _visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        unimplemented!()
    }

    fn deserialize_str<V>(self, _visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        unimplemented!()
    }

    fn deserialize_string<V>(self, _visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        unimplemented!()
    }

    fn deserialize_bytes<V>(self, _visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        unimplemented!()
    }

    fn deserialize_byte_buf<V>(self, _visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        unimplemented!()
    }

    fn deserialize_option<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        let result = self.read_bool()?;

        if result {
            visitor.visit_some(self)
        } else {
            visitor.visit_none()
        }
    }

    fn deserialize_unit<V>(self, _visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        unimplemented!()
    }

    fn deserialize_unit_struct<V>(
        self,
        _name: &'static str,
        visitor: V,
    ) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.deserialize_unit(visitor)
    }

    fn deserialize_newtype_struct<V>(
        self,
        _name: &'static str,
        visitor: V,
    ) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        visitor.visit_newtype_struct(self)
    }

    fn deserialize_seq<V>(mut self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        let mut result = [0u8; 2];
        let n = self.reader.read(&mut result)?;
        if n < result.len() {
            return Err(Error::Eof)
        }
        let size = u16::from_be_bytes(result);

        visitor.visit_seq(SeqDeser::new(&mut self, size))
    }

    fn deserialize_tuple<V>(self, _len: usize, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.deserialize_seq(visitor)
    }

    fn deserialize_tuple_struct<V>(
        self,
        _name: &'static str,
        _len: usize,
        visitor: V,
    ) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.deserialize_seq(visitor)
    }

    fn deserialize_map<V>(self, _visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        unimplemented!()
    }

    fn deserialize_struct<V>(
        mut self,
        _name: &'static str,
        fields: &'static [&'static str],
        visitor: V,
    ) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        let fields_vec = Vec::from(fields);
        visitor.visit_seq(StructDeser::new(&mut self, fields_vec))
    }

    fn deserialize_enum<V>(
        self,
        _name: &'static str,
        _variants: &'static [&'static str],
        _visitor: V,
    ) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        unimplemented!()
    }

    fn deserialize_identifier<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.deserialize_str(visitor)
    }

    fn deserialize_ignored_any<V>(self, visitor: V) -> Result<V::Value>
    where
        V: Visitor<'de>,
    {
        self.deserialize_any(visitor)
    }
}


struct StructDeser<'a, 'de: 'a, R: Read> {
    de: &'a mut Deserializer<'de, R>,
    fields: Vec<&'static str>,
}

impl<'a, 'de, R: Read> StructDeser<'a, 'de, R> {
    fn new(de: &'a mut Deserializer<'de, R>, fields: Vec<&'static str>) -> Self {
        StructDeser { de, fields }
    }
}

impl<'a, 'de, R: Read> de::SeqAccess<'de> for StructDeser<'a, 'de, R> {
    type Error = Error;

    fn next_element_seed<T>(&mut self, seed: T) -> Result<Option<T::Value>> where T: DeserializeSeed<'de> {
        let next_field_opt = self.fields.pop();
        if next_field_opt.is_some() {
            seed.deserialize(&mut *self.de).map(Some)
        } else {
            Ok(None)
        }
    }
}

struct SeqDeser<'a, 'de: 'a, R: Read> {
    de: &'a mut Deserializer<'de, R>,
    size: u16,
}


impl<'a, 'de, R: Read> SeqDeser<'a, 'de, R> {
    fn new(de: &'a mut Deserializer<'de, R>, size: u16) -> Self {
        SeqDeser { de, size }
    }
}

impl<'a, 'de, R: Read> de::SeqAccess<'de> for SeqDeser<'a, 'de, R> {
    type Error = Error;

    fn next_element_seed<T>(&mut self, seed: T) -> Result<Option<T::Value>> where T: DeserializeSeed<'de> {
        if self.size > 0 {
            self.size -= 1;
            seed.deserialize(&mut *self.de).map(Some)
        } else {
            Ok(None)
        }
    }
}

impl<'de, R: Read> Deserializer<'de, R> {
    fn read_bool(&mut self) -> Result<bool> {
        let result = self.read_u8()?;
        if result > 1 {
            return Err(Error::ExpectedBoolean);
        }
        Ok(result > 0)
    }

    fn read_u8(&mut self) -> Result<u8> {
        let mut result = [0u8];
        let n = self.reader.read(&mut result)?;
        if n < 1 {
            return Err(Error::Eof)
        }
        Ok(result[0])
    }
}
