#![allow(unused_parens)]

use std::fmt::Formatter;

use serde::Deserialize;
use serde::Deserializer;
use serde::de::DeserializeSeed;
use serde::de::EnumAccess;
use serde::de::MapAccess;
use serde::de::SeqAccess;
use serde::de::VariantAccess;
use serde::de::Visitor;

/// Something that may be of the kind of `T`.
///
/// If a value doesn't have the kind of `T`, this will *discard* the value and
/// provide an `MayBe::IsNot`. Nevertheless, if the value does have the kind of
/// `T` but cannot be deserialized by `T`, this will also fail to deserialize,
/// making this wrapper "shallow".
///
/// For example, `struct`s have the same kind as a JSON object, `{}`, If the
/// contents of said JSON object don't match what the `struct` is expecting,
/// like if the `struct Foo { bar: u32 }` is being matched, as `MayBe<Foo>`,
/// against `{"bar": ""}`, then that is an error. However, matching the same
/// `MayBe<Foo>` against a JSON array, `[]` would not lead to an error, but
/// to an `MayBe::IsNot`.
///
/// Can only be used with self-describing formats, like JSON.
#[derive(Copy, Debug, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub enum MayBe<T> {
    Is(T),
    IsNot,
}

// FIXME decide on Default
// This enum is analogous to Result, *not* Option.
//
//impl<T> Default for MayBe<T> {
//    fn default() -> Self {
//        MayBe::IsNot
//    }
//}
//impl<T: Default> Default for MayBe<T> {
//    fn default() -> Self {
//        MayBe::Is(T::default())
//    }
//}

impl<T> MayBe<T> {
    /// Returns whether the `MayBe` contains a `T`.
    pub fn is(&self) -> bool {
        matches!(self, MayBe::Is(_))
    }

    /// Returns whether the `MayBe` does not contain a `T`.
    pub fn is_not(&self) -> bool {
        !self.is()
    }

    /// Converts this `MayBe<T>` into an `Option<T>`.
    pub fn into_opt(self) -> Option<T> {
        self.into()
    }

    /// Converts this `&MayBe<T>` into an `Option<&T>`.
    pub fn as_opt(&self) -> Option<&T> {
        match self {
            MayBe::Is(t) => Some(t),
            MayBe::IsNot => None,
        }
    }

    /// Converts this `&mut MayBe<T>` into an `Option<&mut T>`.
    pub fn as_mut_opt(&mut self) -> Option<&mut T> {
        match self {
            MayBe::Is(t) => Some(t),
            MayBe::IsNot => None,
        }
    }
}

impl<T: Clone> Clone for MayBe<T> {
    fn clone(&self) -> Self {
        match self {
            MayBe::Is(t) => MayBe::Is(t.clone()),
            MayBe::IsNot => MayBe::IsNot,
        }
    }

    fn clone_from(&mut self, source: &Self) {
        match (self, source) {
            (MayBe::Is(a), MayBe::Is(b)) => a.clone_from(b),
            (MayBe::IsNot, MayBe::IsNot) => (),
            (a, b) => *a = b.clone(),
        }
    }
}

impl<T> From<MayBe<T>> for Option<T> {
    fn from(thing: MayBe<T>) -> Option<T> {
        match thing {
            MayBe::Is(t) => Some(t),
            MayBe::IsNot => None,
        }
    }
}

impl<T> From<Option<T>> for MayBe<T> {
    fn from(thing: Option<T>) -> MayBe<T> {
        match thing {
            Some(t) => MayBe::Is(t),
            None => MayBe::IsNot,
        }
    }
}

impl<T> From<T> for MayBe<T> {
    fn from(thing: T) -> MayBe<T> {
        MayBe::Is(thing)
    }
}

// actual deserialize

struct MayBeDer<'a, D>(&'a mut bool, D);

struct MayBeVis<'a, V>(&'a mut bool, V);

// acts as Deserializer, MapAccess, SeqAccess and EnumAccess, depending on T.
struct DeepDer<'a, T>(&'a mut bool, T);

#[derive(Deserialize)]
#[serde(untagged)]
enum MayBeHelper<T> {
    #[serde(deserialize_with="may_be")]
    #[serde(bound = "T: Deserialize<'de>")]
    Value(MayBe<T>),
}

fn may_be<'de, D: Deserializer<'de>, T: Deserialize<'de>>(
    d: D,
) -> Result<MayBe<T>, D::Error> {
    let mut depth: bool = false;
    match T::deserialize(MayBeDer(&mut depth, d)) {
        Ok(t) => Ok(MayBe::Is(t)),
        Err(_) if !depth => Ok(MayBe::IsNot),
        Err(e) => Err(e),
    }
}

impl<'de, T: Deserialize<'de>> Deserialize<'de> for MayBe<T> {
    fn deserialize<D: Deserializer<'de>>(d: D) -> Result<Self, D::Error> {
        MayBeHelper::<T>::deserialize(d).map(|MayBeHelper::Value(v): _| v)
    }
}

impl<'a, 'de: 'a, V: Visitor<'de>> Visitor<'de> for MayBeVis<'a, V> {
    type Value = V::Value;

    fn expecting(&self, formatter: &mut Formatter<'_>) -> std::fmt::Result {
        let MayBeVis(_depth, inner) = self;
        inner.expecting(formatter)
    }

    fn visit_bool<E: serde::de::Error>(
        self,
        v: bool,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_bool(v)
    }

    fn visit_i8<E: serde::de::Error>(
        self,
        v: i8,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_i8(v)
    }

    fn visit_i16<E: serde::de::Error>(
        self,
        v: i16,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_i16(v)
    }

    fn visit_i32<E: serde::de::Error>(
        self,
        v: i32,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_i32(v)
    }

    fn visit_i64<E: serde::de::Error>(
        self,
        v: i64,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_i64(v)
    }

    fn visit_i128<E: serde::de::Error>(
        self,
        v: i128,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_i128(v)
    }

    fn visit_u8<E: serde::de::Error>(
        self,
        v: u8,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_u8(v)
    }

    fn visit_u16<E: serde::de::Error>(
        self,
        v: u16,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_u16(v)
    }

    fn visit_u32<E: serde::de::Error>(
        self,
        v: u32,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_u32(v)
    }

    fn visit_u64<E: serde::de::Error>(
        self,
        v: u64,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_u64(v)
    }

    fn visit_u128<E: serde::de::Error>(
        self,
        v: u128,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_u128(v)
    }

    fn visit_f32<E: serde::de::Error>(
        self,
        v: f32,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_f32(v)
    }

    fn visit_f64<E: serde::de::Error>(
        self,
        v: f64,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_f64(v)
    }

    fn visit_char<E: serde::de::Error>(
        self,
        v: char,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_char(v)
    }

    fn visit_str<E: serde::de::Error>(
        self,
        v: &str,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_str(v)
    }

    fn visit_borrowed_str<E: serde::de::Error>(
        self,
        v: &'de str,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_borrowed_str(v)
    }

    fn visit_string<E: serde::de::Error>(
        self,
        v: String,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_string(v)
    }

    fn visit_bytes<E: serde::de::Error>(
        self,
        v: &[u8],
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_bytes(v)
    }

    fn visit_borrowed_bytes<E: serde::de::Error>(
        self,
        v: &'de [u8],
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_borrowed_bytes(v)
    }

    fn visit_byte_buf<E: serde::de::Error>(
        self,
        v: Vec<u8>,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_byte_buf(v)
    }

    fn visit_none<E: serde::de::Error>(
        self,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_none()
    }

    fn visit_some<D: Deserializer<'de>>(
        self,
        deserializer: D,
    ) -> Result<Self::Value, D::Error> {
        let MayBeVis(depth, inner) = self;
        inner.visit_some(DeepDer(depth, deserializer))
    }

    fn visit_unit<E: serde::de::Error>(
        self,
    ) -> Result<Self::Value, E> {
        let MayBeVis(_depth, inner) = self;
        inner.visit_unit()
    }

    fn visit_newtype_struct<D: Deserializer<'de>>(
        self,
        deserializer: D,
    ) -> Result<Self::Value, D::Error> {
        let MayBeVis(depth, inner) = self;
        inner.visit_newtype_struct(DeepDer(depth, deserializer))
    }

    fn visit_seq<A: SeqAccess<'de>>(
        self,
        seq: A,
    ) -> Result<Self::Value, A::Error> {
        let MayBeVis(depth, inner) = self;
        inner.visit_seq(DeepDer(depth, seq))
    }

    fn visit_map<A: MapAccess<'de>>(
        self,
        map: A,
    ) -> Result<Self::Value, A::Error> {
        let MayBeVis(depth, inner) = self;
        inner.visit_map(DeepDer(depth, map))
    }

    fn visit_enum<A: EnumAccess<'de>>(
        self,
        data: A,
    ) -> Result<Self::Value, A::Error> {
        let MayBeVis(depth, inner) = self;
        inner.visit_enum(DeepDer(depth, data))
    }
}

impl<'a, 'de: 'a, D: Deserializer<'de>> Deserializer<'de> for MayBeDer<'a, D> {
    type Error = D::Error;

    fn deserialize_any<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_any(MayBeVis(depth, visitor))
    }

    fn deserialize_bool<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_bool(MayBeVis(depth, visitor))
    }

    fn deserialize_i8<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_i8(MayBeVis(depth, visitor))
    }

    fn deserialize_i16<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_i16(MayBeVis(depth, visitor))
    }

    fn deserialize_i32<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_i32(MayBeVis(depth, visitor))
    }

    fn deserialize_i64<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_i64(MayBeVis(depth, visitor))
    }

    fn deserialize_u8<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_u8(MayBeVis(depth, visitor))
    }

    fn deserialize_u16<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_u16(MayBeVis(depth, visitor))
    }

    fn deserialize_u32<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_u32(MayBeVis(depth, visitor))
    }

    fn deserialize_u64<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_u64(MayBeVis(depth, visitor))
    }

    fn deserialize_f32<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_f32(MayBeVis(depth, visitor))
    }

    fn deserialize_f64<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_f64(MayBeVis(depth, visitor))
    }

    fn deserialize_char<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_char(MayBeVis(depth, visitor))
    }

    fn deserialize_str<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_str(MayBeVis(depth, visitor))
    }

    fn deserialize_string<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_string(MayBeVis(depth, visitor))
    }

    fn deserialize_bytes<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_bytes(MayBeVis(depth, visitor))
    }

    fn deserialize_byte_buf<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_byte_buf(MayBeVis(depth, visitor))
    }

    fn deserialize_option<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_option(MayBeVis(depth, visitor))
    }

    fn deserialize_unit<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_unit(MayBeVis(depth, visitor))
    }

    fn deserialize_unit_struct<V: Visitor<'de>>(
        self, 
        name: &'static str, 
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_unit_struct(name, MayBeVis(depth, visitor))
    }

    fn deserialize_newtype_struct<V: Visitor<'de>>(
        self, 
        name: &'static str, 
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_newtype_struct(name, MayBeVis(depth, visitor))
    }

    fn deserialize_seq<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_seq(MayBeVis(depth, visitor))
    }

    fn deserialize_tuple<V: Visitor<'de>>(
        self, 
        len: usize, 
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_tuple(len, MayBeVis(depth, visitor))
    }

    fn deserialize_tuple_struct<V: Visitor<'de>>(
        self, 
        name: &'static str, 
        len: usize, 
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_tuple_struct(name, len, MayBeVis(depth, visitor))
    }

    fn deserialize_map<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_map(MayBeVis(depth, visitor))
    }

    fn deserialize_struct<V: Visitor<'de>>(
        self, 
        name: &'static str, 
        fields: &'static [&'static str], 
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_struct(name, fields, MayBeVis(depth, visitor))
    }

    fn deserialize_enum<V: Visitor<'de>>(
        self, 
        name: &'static str, 
        variants: &'static [&'static str], 
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_enum(name, variants, MayBeVis(depth, visitor))
    }

    fn deserialize_identifier<V: Visitor<'de>>(
        self, 
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_identifier(MayBeVis(depth, visitor))
    }

    fn deserialize_ignored_any<V: Visitor<'de>>(
        self, 
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_ignored_any(MayBeVis(depth, visitor))
    }

    fn deserialize_i128<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_i128(MayBeVis(depth, visitor))
    }

    fn deserialize_u128<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let MayBeDer(depth, inner) = self;
        inner.deserialize_u128(MayBeVis(depth, visitor))
    }

    fn is_human_readable(&self) -> bool {
        let MayBeDer(_depth, inner) = self;
        inner.is_human_readable()
    }
}

impl<'a, 'de: 'a, A: VariantAccess<'de>> VariantAccess<'de> for DeepDer<'a, A> {
    type Error = A::Error;

    fn unit_variant(self) -> Result<(), Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.unit_variant()
    }

    fn newtype_variant_seed<T: DeserializeSeed<'de>>(
        self,
        seed: T,
    ) -> Result<T::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.newtype_variant_seed(seed)
    }

    fn tuple_variant<V: Visitor<'de>>(
        self, 
        len: usize, 
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.tuple_variant(len, visitor)
    }

    fn struct_variant<V: Visitor<'de>>(
        self, 
        fields: &'static [&'static str], 
        visitor: V
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.struct_variant(fields, visitor)
    }

    fn newtype_variant<T: Deserialize<'de>>(
        self,
    ) -> Result<T, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.newtype_variant()
    }
}

impl<'a, 'de: 'a, A: EnumAccess<'de>> EnumAccess<'de> for DeepDer<'a, A> {
    type Error = A::Error;

    type Variant = DeepDer<'a, A::Variant>;

    fn variant_seed<V: DeserializeSeed<'de>>(
        self, 
        seed: V,
    ) -> Result<(V::Value, Self::Variant), Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.variant_seed(seed).map(move |(a, b)| {
            (a, DeepDer(depth, b))
        })
    }

    fn variant<V: Deserialize<'de>>(
        self,
    ) -> Result<(V, Self::Variant), Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.variant().map(move |(a, b)| {
            (a, DeepDer(depth, b))
        })
    }
}

impl<'a, 'de: 'a, A: SeqAccess<'de>> SeqAccess<'de> for DeepDer<'a, A> {
    type Error = A::Error;

    fn next_element_seed<T: DeserializeSeed<'de>>(
        &mut self, 
        seed: T,
    ) -> Result<Option<T::Value>, Self::Error> {
        let DeepDer(ref mut depth, inner) = self;
        **depth = true;
        inner.next_element_seed(seed)
    }

    fn next_element<T: Deserialize<'de>>(
        &mut self, 
    ) -> Result<Option<T>, Self::Error> {
        let DeepDer(ref mut depth, inner) = self;
        **depth = true;
        inner.next_element()
    }

    fn size_hint(&self) -> Option<usize> {
        let DeepDer(_depth, inner) = self;
        inner.size_hint()
    }
}

impl<'a, 'de: 'a, A: MapAccess<'de>> MapAccess<'de> for DeepDer<'a, A> {
    type Error = A::Error;

    fn next_key_seed<K: DeserializeSeed<'de>>(
        &mut self, 
        seed: K,
    ) -> Result<Option<K::Value>, Self::Error> {
        let DeepDer(ref mut depth, inner) = self;
        **depth = true;
        inner.next_key_seed(seed)
    }

    fn next_value_seed<V: DeserializeSeed<'de>>(
        &mut self,
        seed: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(ref mut depth, inner) = self;
        **depth = true;
        inner.next_value_seed(seed)
    }

    fn next_entry_seed<K: DeserializeSeed<'de>, V: DeserializeSeed<'de>>(
        &mut self, 
        kseed: K, 
        vseed: V,
    ) -> Result<Option<(K::Value, V::Value)>, Self::Error> {
        let DeepDer(ref mut depth, inner) = self;
        **depth = true;
        inner.next_entry_seed(kseed, vseed)
    }

    fn next_key<K: Deserialize<'de>>(
        &mut self,
    ) -> Result<Option<K>, Self::Error> {
        let DeepDer(ref mut depth, inner) = self;
        **depth = true;
        inner.next_key()
    }

    fn next_value<V: Deserialize<'de>>(
        &mut self,
    ) -> Result<V, Self::Error> {
        let DeepDer(ref mut depth, inner) = self;
        **depth = true;
        inner.next_value()
    }

    fn next_entry<K: Deserialize<'de>, V: Deserialize<'de>>(
        &mut self,
    ) -> Result<Option<(K, V)>, Self::Error> {
        let DeepDer(ref mut depth, inner) = self;
        **depth = true;
        inner.next_entry()
    }

    fn size_hint(&self) -> Option<usize> {
        let DeepDer(_depth, inner) = self;
        inner.size_hint()
    }
}

impl<'a, 'de: 'a, D: Deserializer<'de>> Deserializer<'de> for DeepDer<'a, D> {
    type Error = D::Error;

    fn deserialize_any<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_any((visitor))
    }

    fn deserialize_bool<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_bool((visitor))
    }

    fn deserialize_i8<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_i8((visitor))
    }

    fn deserialize_i16<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_i16((visitor))
    }

    fn deserialize_i32<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_i32((visitor))
    }

    fn deserialize_i64<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_i64((visitor))
    }

    fn deserialize_u8<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_u8((visitor))
    }

    fn deserialize_u16<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_u16((visitor))
    }

    fn deserialize_u32<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_u32((visitor))
    }

    fn deserialize_u64<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_u64((visitor))
    }

    fn deserialize_f32<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_f32((visitor))
    }

    fn deserialize_f64<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_f64((visitor))
    }

    fn deserialize_char<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_char((visitor))
    }

    fn deserialize_str<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_str((visitor))
    }

    fn deserialize_string<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_string((visitor))
    }

    fn deserialize_bytes<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_bytes((visitor))
    }

    fn deserialize_byte_buf<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_byte_buf((visitor))
    }

    fn deserialize_option<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_option((visitor))
    }

    fn deserialize_unit<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_unit((visitor))
    }

    fn deserialize_unit_struct<V: Visitor<'de>>(
        self, 
        name: &'static str, 
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_unit_struct(name, (visitor))
    }

    fn deserialize_newtype_struct<V: Visitor<'de>>(
        self, 
        name: &'static str, 
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_newtype_struct(name, (visitor))
    }

    fn deserialize_seq<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_seq((visitor))
    }

    fn deserialize_tuple<V: Visitor<'de>>(
        self, 
        len: usize, 
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_tuple(len, (visitor))
    }

    fn deserialize_tuple_struct<V: Visitor<'de>>(
        self, 
        name: &'static str, 
        len: usize, 
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_tuple_struct(name, len, (visitor))
    }

    fn deserialize_map<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_map((visitor))
    }

    fn deserialize_struct<V: Visitor<'de>>(
        self, 
        name: &'static str, 
        fields: &'static [&'static str], 
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_struct(name, fields, (visitor))
    }

    fn deserialize_enum<V: Visitor<'de>>(
        self, 
        name: &'static str, 
        variants: &'static [&'static str], 
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_enum(name, variants, (visitor))
    }

    fn deserialize_identifier<V: Visitor<'de>>(
        self, 
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_identifier((visitor))
    }

    fn deserialize_ignored_any<V: Visitor<'de>>(
        self, 
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_ignored_any((visitor))
    }

    fn deserialize_i128<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_i128((visitor))
    }

    fn deserialize_u128<V: Visitor<'de>>(
        self,
        visitor: V,
    ) -> Result<V::Value, Self::Error> {
        let DeepDer(depth, inner) = self;
        *depth = true;
        inner.deserialize_u128((visitor))
    }

    fn is_human_readable(&self) -> bool {
        let DeepDer(_depth, inner) = self;
        inner.is_human_readable()
    }
}
