//! X25519 key-pair (public-key and private-key).

use core::ops::DerefMut;

use crate::random::global_rng;

use self::{
    private_key::{ReceiverPrivateKey, SenderPrivateKey},
    public_key::{ReceiverPublicKey, SenderPublicKey},
};
use crypto_box::{PublicKey, SecretKey};

pub(in crate::key) mod private_key;
pub(in crate::key) mod public_key;

/// X25519 Key-pair generated by sender.
#[derive(Clone, Debug)]
pub struct SenderKeyPair {
    sender_private_key: SenderPrivateKey,
    sender_public_key: SenderPublicKey,
}

impl SenderKeyPair {
    /// Generates a key-pair for message sender.
    pub fn generate() -> Self {
        let (private_key, public_key) = gen_key_pair();
        let sender_private_key = SenderPrivateKey::from(private_key);
        let sender_public_key = SenderPublicKey::from(public_key);
        Self {
            sender_private_key,
            sender_public_key,
        }
    }

    /// Ref to private key.
    pub fn private_key(&self) -> &SenderPrivateKey {
        &self.sender_private_key
    }

    /// Ref to public key.
    pub fn public_key(&self) -> &SenderPublicKey {
        &self.sender_public_key
    }
}

/// X25519 Key-pair generated by receiver.
#[derive(Clone, Debug)]
pub struct ReceiverKeyPair {
    receiver_private_key: ReceiverPrivateKey,
    receiver_public_key: ReceiverPublicKey,
}

impl ReceiverKeyPair {
    /// Generates a key-pair for message receiver.
    pub fn generate() -> Self {
        let (private_key, public_key) = gen_key_pair();
        let receiver_private_key = ReceiverPrivateKey::from(private_key);
        let receiver_public_key = ReceiverPublicKey::from(public_key);
        Self {
            receiver_private_key,
            receiver_public_key,
        }
    }

    /// Ref to private key.
    pub fn private_key(&self) -> &ReceiverPrivateKey {
        &self.receiver_private_key
    }

    /// Ref to public key.
    pub fn public_key(&self) -> &ReceiverPublicKey {
        &self.receiver_public_key
    }
}

fn gen_key_pair() -> (SecretKey, PublicKey) {
    let mut rng = global_rng().lock();

    let secret_key = SecretKey::generate(rng.deref_mut());
    let public_key = secret_key.public_key();

    (secret_key, public_key)
}
