// Copyright 2018 MaidSafe.net limited.
//
// This SAFE Network Software is licensed to you under The General Public License (GPL), version 3.
// Unless required by applicable law or agreed to in writing, the SAFE Network Software distributed
// under the GPL Licence is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied. Please review the Licences for the specific language governing
// permissions and limitations relating to use of the SAFE Network Software.

use futures::future::join_all;

use super::{
    small_encryptor::SmallEncryptor, utils, SelfEncryptionError, Storage, MAX_CHUNK_SIZE,
    MIN_CHUNK_SIZE,
};
use crate::data_map::{ChunkDetails, DataMap};
use std::convert::From;
pub const MIN: usize = 3 * MIN_CHUNK_SIZE;
pub const MAX: usize = 3 * MAX_CHUNK_SIZE;

// An encryptor for data which will be split into exactly three chunks (i.e. size is between
// `3 * MIN_CHUNK_SIZE` and `3 * MAX_CHUNK_SIZE` inclusive).  Only `close()` will actually cause
// chunks to be stored.  Until then, data is held internally in `buffer`.
pub struct MediumEncryptor<S: Storage + Send + Sync + Clone> {
    pub storage: S,
    pub buffer: Vec<u8>,
    original_chunks: Option<Vec<ChunkDetails>>,
}

impl<S> MediumEncryptor<S>
where
    S: Storage + 'static + Send + Sync + Clone,
{
    // Constructor for use with pre-existing `DataMap::Chunks` where there are exactly three chunks.
    // Retrieves the chunks from storage and decrypts them to its internal `buffer`.
    #[allow(clippy::new_ret_no_self)]
    pub async fn new(
        storage: S,
        chunks: Vec<ChunkDetails>,
    ) -> Result<MediumEncryptor<S>, SelfEncryptionError> {
        debug_assert_eq!(chunks.len(), 3);
        debug_assert!(MIN <= chunks.iter().fold(0, |acc, chunk| acc + chunk.source_size));
        debug_assert!(chunks.iter().fold(0, |acc, chunk| acc + chunk.source_size) <= MAX);

        let mut data = Vec::with_capacity(chunks.len());
        let mut get_futures = Vec::new();
        for (index, chunk) in chunks.iter().enumerate() {
            let pad_key_iv = utils::get_pad_key_and_iv(index, &chunks);
            let mut storage = storage.clone();
            get_futures.push(async move {
                let chunk = storage.get(&chunk.hash).await?;
                let decrypted_chunk = utils::decrypt_chunk(&chunk, pad_key_iv)?;
                Ok::<_, SelfEncryptionError>(decrypted_chunk)
            });
        }
        let results = join_all(get_futures.into_iter()).await;
        for chunk in results {
            data.push(chunk?);
        }

        let init = Vec::with_capacity(MAX);
        let buffer = data.into_iter().fold(init, |mut buffer, data| {
            buffer.extend(data);
            buffer
        });

        Ok(MediumEncryptor {
            storage,
            buffer,
            original_chunks: Some(chunks),
        })
    }

    // Simply appends to internal buffer assuming the size limit is not exceeded.  No chunks are
    // generated by this call.
    pub async fn write(mut self, data: &[u8]) -> Result<Self, SelfEncryptionError> {
        debug_assert!(data.len() + self.len() <= MAX);
        self.original_chunks = None;
        self.buffer.extend_from_slice(data);
        Ok(self)
    }

    // This finalises the encryptor - it should not be used again after this call.  Exactly three
    // chunks will be generated and stored by calling this unless the encryptor didn't receive any
    // `write()` calls.
    pub async fn close(mut self) -> Result<(DataMap, S), SelfEncryptionError> {
        if let Some(chunks) = self.original_chunks {
            return Ok((DataMap::Chunks(chunks), self.storage));
        }

        // let mut chunk_storage_futures;
        let mut chunk_details;

        {
            // Third the contents, with the extra single or two bytes in the last chunk.
            let chunk_contents = vec![
                &self.buffer[..(self.buffer.len() / 3)],
                &self.buffer[(self.buffer.len() / 3)..(2 * (self.buffer.len() / 3))],
                &self.buffer[(2 * (self.buffer.len() / 3))..],
            ];
            // Note the pre-encryption hashes and sizes.
            chunk_details = vec![];
            for (index, contents) in chunk_contents.iter().enumerate() {
                chunk_details.push(ChunkDetails {
                    chunk_num: index,
                    hash: vec![],
                    pre_hash: self.storage.generate_address(contents).await?,
                    source_size: contents.len(),
                });
            }
            // Encrypt the chunks and note the post-encryption hashes
            let partial_details = chunk_details.clone();
            // chunk_storage_futures = Vec::with_capacity(chunk_contents.len());
            // FIXME: rust-nightly requires this to be mutable while rust-stable does not
            #[allow(unused)]
            for (index, (contents, mut details)) in chunk_contents
                .iter()
                .zip(chunk_details.iter_mut())
                .enumerate()
            {
                let pad_key_iv = utils::get_pad_key_and_iv(index, &partial_details);
                let encrypted_contents = utils::encrypt_chunk(contents, pad_key_iv)?;

                let hash = self.storage.generate_address(&encrypted_contents).await?;
                details.hash = hash.to_vec();
                self.storage.put(hash.to_vec(), encrypted_contents).await?
                // let mut storage = self.storage.clone();
                // chunk_storage_futures.push(async move { storage.put(hash.to_vec(), encrypted_contents).await });
            }
        }
        // let results = join_all(chunk_storage_futures.into_iter()).await;
        // for result in results {
        //     result?;
        // }

        Ok((DataMap::Chunks(chunk_details), self.storage))
    }

    pub fn len(&self) -> usize {
        self.buffer.len()
    }

    pub fn is_empty(&self) -> bool {
        self.buffer.is_empty()
    }
}

impl<S: Storage + Send + Sync + Clone> From<SmallEncryptor<S>> for MediumEncryptor<S> {
    fn from(small_encryptor: SmallEncryptor<S>) -> MediumEncryptor<S> {
        MediumEncryptor {
            storage: small_encryptor.storage,
            buffer: small_encryptor.buffer,
            original_chunks: None,
        }
    }
}

#[cfg(test)]
mod tests {
    use super::{
        super::{
            small_encryptor::{self, SmallEncryptor},
            utils, MAX_CHUNK_SIZE,
        },
        *,
    };
    use crate::{
        data_map::DataMap,
        self_encryptor::SelfEncryptor,
        test_helpers::{new_test_rng, random_bytes, Blob, SimpleStorage},
    };
    use rand::Rng;

    #[test]
    fn consts() {
        assert_eq!(MIN, small_encryptor::MAX + 1);
    }

    // Writes all of `data` to a new encryptor in a single call, then closes and reads back via
    // a `SelfEncryptor`.
    async fn basic_write_and_close(data: &[u8]) -> Result<(), SelfEncryptionError> {
        let (data_map, storage) = {
            let storage = SimpleStorage::new();
            let mut encryptor = SmallEncryptor::new(storage, vec![])
                .await
                .map(MediumEncryptor::from)?;
            assert_eq!(encryptor.len(), 0);
            assert!(encryptor.is_empty());
            encryptor = encryptor.write(data).await?;
            assert_eq!(encryptor.len(), data.len());
            assert!(!encryptor.is_empty());
            encryptor.close().await?
        };

        match data_map {
            DataMap::Chunks(ref chunks) => assert_eq!(chunks.len(), 3),
            _ => panic!("Wrong DataMap type returned."),
        }

        let self_encryptor = SelfEncryptor::new(storage, data_map)?;
        let fetched = self_encryptor.read(0, data.len()).await?;
        assert_eq!(Blob(&fetched), Blob(data));
        Ok(())
    }

    // Splits `data` into several pieces, then for each piece:
    //  * constructs a new encryptor from existing chunk details (except for the first piece)
    //  * writes the piece
    //  * closes and reads back the full data via a `SelfEncryptor`.
    async fn multiple_writes_then_close<T: Rng>(
        rng: &mut T,
        data: &[u8],
    ) -> Result<(), SelfEncryptionError> {
        let mut storage = SimpleStorage::new();
        let mut existing_data = vec![];
        let data_pieces = utils::make_random_pieces(rng, data, MIN);
        let mut current_chunks = vec![];
        for data in data_pieces {
            let data_map = {
                let mut encryptor = if current_chunks.is_empty() {
                    SmallEncryptor::new(storage, vec![])
                        .await
                        .map(MediumEncryptor::from)?
                } else {
                    MediumEncryptor::new(storage, current_chunks).await?
                };
                encryptor = encryptor.write(data).await?;
                existing_data.extend_from_slice(data);
                assert_eq!(encryptor.len(), existing_data.len());

                let (data_map, storage2) = encryptor.close().await?;
                storage = storage2;
                data_map
            };

            match data_map {
                DataMap::Chunks(ref chunks) => {
                    assert_eq!(chunks.len(), 3);
                    current_chunks = chunks.clone()
                }
                _ => panic!("Wrong DataMap type returned."),
            }

            let self_encryptor = SelfEncryptor::new(storage, data_map)?;
            assert_eq!(self_encryptor.len().await, existing_data.len());
            let fetched = self_encryptor.read(0, existing_data.len()).await?;
            assert_eq!(fetched, existing_data);
            storage = self_encryptor.into_storage().await;
        }
        assert_eq!(Blob(&existing_data[..]), Blob(data));
        Ok(())
    }

    #[tokio::test]
    async fn all_unit() -> Result<(), SelfEncryptionError> {
        let mut rng = new_test_rng()?;
        let data = random_bytes(&mut rng, MAX);

        basic_write_and_close(&data[..MIN]).await?;
        basic_write_and_close(&data[..MAX_CHUNK_SIZE]).await?;
        basic_write_and_close(&data[..(MAX_CHUNK_SIZE * 2)]).await?;
        basic_write_and_close(&data).await?;

        multiple_writes_then_close(&mut rng, &data[..(MIN * 2)]).await?;
        multiple_writes_then_close(&mut rng, &data[..MAX_CHUNK_SIZE]).await?;
        multiple_writes_then_close(&mut rng, &data[..(MAX_CHUNK_SIZE * 2)]).await?;
        multiple_writes_then_close(&mut rng, &data).await?;
        Ok(())
    }
}
