use ansi_term::Colour::Yellow;

use crate::cli::*;
use crate::core::extract::{Extract, Params};
use crate::parser::txt;

impl InputCli for ExtractParser<'_> {}
impl InputPrint for ExtractParser<'_> {}
impl OutputCli for ExtractParser<'_> {}

pub(in crate::cli) struct ExtractParser<'a> {
    matches: &'a ArgMatches<'a>,
    input_dir: Option<PathBuf>,
    params: Params,
}

impl<'a> ExtractParser<'a> {
    pub(in crate::cli) fn new(matches: &'a ArgMatches<'a>) -> Self {
        Self {
            matches,
            input_dir: None,
            params: Params::None,
        }
    }

    pub(in crate::cli) fn extract(&mut self) {
        let input_fmt = self.parse_input_fmt(self.matches);
        let datatype = self.parse_datatype(self.matches);
        let output_fmt = self.parse_output_fmt(self.matches);
        let outdir = self.parse_output(self.matches);
        let task_desc = "Sequence extraction";
        let files = if self.is_input_wcard() {
            self.parse_input_wcard(self.matches)
        } else {
            let dir = self.parse_dir_input(self.matches);
            self.input_dir = Some(PathBuf::from(dir));
            self.get_files(dir, &input_fmt)
        };
        self.print_input_multi(
            &self.input_dir,
            task_desc,
            files.len(),
            &input_fmt,
            &datatype,
        );
        self.check_output_dir_exist(&outdir);
        log::info!("{}", Yellow.paint("Params"));
        self.parse_params();
        let extract = Extract::new(&self.params, &input_fmt, &datatype);
        extract.extract_sequences(&files, &outdir, &output_fmt);
    }

    fn parse_params(&mut self) {
        match self.matches {
            m if m.is_present("regex") => {
                let re = self.parse_regex();
                log::info!("{:18}: {}\n", "Regex", re);
                self.params = Params::Regex(re);
            }
            m if m.is_present("id") => {
                let ids = self.parse_id();
                log::info!("{:18}: {:?}\n", "IDs", ids);
                self.params = Params::Id(ids);
            }
            m if m.is_present("file") => {
                let ids = self.parse_file();
                log::info!(
                    "{:18}: {}\n",
                    "File",
                    self.matches
                        .value_of("file")
                        .expect("Failed parsing file path")
                );
                self.params = Params::Id(ids);
            }
            _ => unreachable!("Unknown parameters!"),
        }
    }

    fn parse_regex(&self) -> String {
        let re = self
            .matches
            .value_of("regex")
            .expect("Failed parsing regex string");
        String::from(re)
    }

    fn parse_file(&self) -> Vec<String> {
        let file = PathBuf::from(
            self.matches
                .value_of("file")
                .expect("Failed parsing file path"),
        );
        assert!(file.is_file(), "File does not exist: {}", file.display());
        txt::parse_text_file(&file)
    }

    fn parse_id(&self) -> Vec<String> {
        self.matches
            .values_of("id")
            .expect("Failed parsing IDs input")
            .map(String::from)
            .collect()
    }

    fn is_input_wcard(&self) -> bool {
        self.matches.is_present("wildcard")
    }
}
