use pulldown_cmark::Options;
use serde::Deserialize;

use crate::options::Opt;


/// Information about a specific page; Its title, description, language tag, author, and if list is enabled
#[derive(Deserialize, Debug)]
pub struct Page {
    /// The title of the page
    #[serde(default)]
    pub title: Option<String>,
    /// The description of the page, used for the description metatag 
    #[serde(default)]
    pub description: Option<String>,
    /// The HTML language tag of the page's contents
    #[serde(default)]
    pub language: Option<String>,
    /// The page's author, used for the author metatag
    #[serde(default)]
    pub author: Option<String>,
    /// Whether the page should be included in the list
    #[serde(default)]
    pub list: Option<String>,
}

/// The global settings, used for all pages.
#[derive(Deserialize, Debug)]
pub struct Settings {
    /// The default author to use if not set.
    #[serde(default = "default_author")]
    pub default_author: String,
}

#[doc(hidden)]
fn default_author() -> String {
    "Sedum".to_string()
}

impl Default for Settings {
    fn default() -> Settings {
        Settings {
            default_author: ("Sedum").to_string(),
        }
    }
}

/// Settings that remain constant througout the generation. These are kept in a struct to avoid re-generating them for every page.
pub struct Constants {
    /// The HTML generated by sedum::io::list_files
    pub list_html: String,
    /// The number of pages in the list. If this is 0, then the list element will not be inserted, to prevent strange formatting
    pub list_count: i64,
    /// The command line options. Handled by sedum::options
    pub opt: Opt,
    /// The head.include, if it exists
    pub head_include: String,
    /// The body.include, if it exists
    pub body_include: String,
    /// The options for pulldown_cmark
    pub pulldown_cmark_options: Options,
    /// The global settings. See sedum::structs::Settings
    pub global_settings: Settings,
}

#[doc(hidden)]
pub struct PageUnwrapped {
    pub lang_string: String,
    pub title_string: String,
    pub description_string: String,
    pub author_string: String,
    pub timestamp_string: String,
}