use clap::{crate_description, crate_name, crate_version, App, Arg};
use serde::Deserialize;
use shell_words;
use std::collections::HashMap;
use std::env;
use std::fs;
use std::io::Read;
use std::process::{exit, Command, Stdio};
use std::string::String;
use toml;

#[derive(Deserialize)]
struct Config {
    sedo: ConfigSedo,
}

#[derive(Deserialize)]
struct ConfigSedo {
    default_env: String,
    environments: Vec<ConfigSedoEnvironment>,
    op_path: Option<String>,
}

#[derive(Deserialize)]
struct ConfigSedoEnvironment {
    name: String,
    sign_in_address: String,
    items: Vec<ConfigSedoEnvironmentItem>,
}

#[derive(Deserialize)]
struct ConfigSedoEnvironmentItem {
    vault_id: String,
    item_id: String,
}

#[derive(Deserialize, Debug)]
struct VaultItem {
    details: VaultItemDetails,
}

#[derive(Deserialize, Debug)]
struct VaultItemDetails {
    // fields
    // notesPlain
    // passwordHistory
    sections: Vec<VaultItemSection>,
}

#[derive(Deserialize, Debug)]
struct VaultItemSection {
    fields: Option<Vec<VaultItemSectionField>>,
    name: String,
    title: String,
}

#[derive(Deserialize, Debug)]
struct VaultItemSectionField {
    k: String, // kind: string, concealed, etc
    n: String,
    t: String,         // title
    v: Option<String>, // value
}

impl VaultItemSection {
    /// Create a new HashMap to be used as environment variables
    fn as_env(&self) -> HashMap<String, String> {
        match &self.fields {
            None => return HashMap::new(),
            Some(fields) => {
                let mut env = HashMap::with_capacity(fields.len());
                for field in fields {
                    if let Some(v) = &field.v {
                        env.insert(field.t.clone(), v.clone());
                    } else {
                        env.insert(field.t.clone(), String::new());
                    }
                }
                return env;
            }
        }
    }
}

/// Run the command in a `$SHELL` subshell.
/// This allows for the command to use the injected environment variables.
///
/// # Example
///
/// Shell usage:
///
/// ```bash
/// sedo -c 'echo $SEDO'
/// ```
fn run_cmd(
    env_name: &str,
    command_string: String,
    cmd_env: HashMap<String, String>,
    interactive: bool,
    login: bool,
) -> std::process::ExitStatus {
    let shell = env::var("SHELL").expect("SHELL not set for current user");
    let mut cmd = Command::new(shell);
    if interactive {
        cmd.arg("-i");
    }
    if login {
        cmd.arg("-l");
    }
    cmd.env("SEDO", "1")
        .env("SEDO_ENV", env_name)
        .envs(cmd_env)
        .arg("-c")
        .arg(command_string)
        .spawn()
        .expect("failed to run command")
        .wait()
        .expect("failed to wait for command")
}

fn signin(sign_in_address: &str, op_path: Option<&str>) -> Result<String, String> {
    let mut op_signin = match op_path {
        Some(val) => Command::new(val),
        None => Command::new("op"),
    };

    op_signin.args(&["signin", sign_in_address, "--raw"]);
    op_signin.stdout(Stdio::piped());

    let mut child = op_signin.spawn().expect("failed to start op");

    // wait for op to prompt for the password
    match child.try_wait() {
        Ok(Some(status)) => {
            let msg = format!("op signin exited early with status: {}", status);
            return Err(msg);
        }
        Ok(None) => {
            // This will wait for the user to input the password
            let status = child.wait().expect("ERROR: did not prompt for password");

            match status.code() {
                Some(code) => match code {
                    0 => {
                        let mut session_token = String::new();
                        child
                            .stdout
                            .expect("failed to get op signin stdout")
                            .read_to_string(&mut session_token)
                            .expect("failed to read output");
                        let token = session_token.trim_end_matches("\n");
                        return Ok(String::from(token));
                    }
                    _ => {
                        return Err("op signin failed".to_string());
                    }
                },
                None => Err("op terminated by signal".to_string()),
            }
        }
        Err(e) => {
            let msg = format!("error attempting to wait for op signin: {}", e);
            return Err(msg.to_string());
        }
    }
}

fn fetch_env(
    session_token: &String,
    item_id: &String,
    vault_id: &String,
    op_path: Option<&str>,
) -> HashMap<String, String> {
    let mut cmd_get_item = match op_path {
        Some(val) => Command::new(val),
        None => Command::new("op"),
    };

    cmd_get_item.args(&[
        "get",
        "item",
        item_id,
        "--session",
        session_token,
        "--vault",
        vault_id,
        "--format",
        "json",
    ]);
    cmd_get_item.stdout(Stdio::piped());

    let child = cmd_get_item.spawn().expect("failed to start op");
    let output = child
        .wait_with_output()
        .expect("failed to wait for op get item");

    match output.status.code() {
        None => panic!("op get item terminated by signal"),
        Some(0) => {
            let item = String::from_utf8(output.stdout).expect("failed to read item stdout");
            let vault_item: VaultItem = serde_json::from_str(&item).expect("failed to parse json");
            let mut cmd_env = HashMap::new();
            for section in vault_item.details.sections {
                if "environment" == section.title.to_lowercase() {
                    cmd_env.extend(section.as_env())
                }
            }
            return cmd_env;
        }
        Some(code) => panic!("op get item exited with status code: {}", code),
    }
}

fn build_env(env_cfg: &ConfigSedoEnvironment, op_path: Option<&str>) -> HashMap<String, String> {
    let session_token = signin(&env_cfg.sign_in_address, op_path).unwrap_or_else(|msg| {
        eprintln!("ERROR: {}", msg);
        exit(1);
    });

    let mut cmd_env: HashMap<String, String> = HashMap::new();
    for item in env_cfg.items.iter() {
        let item_env = fetch_env(&session_token, &item.item_id, &item.vault_id, op_path);
        cmd_env.extend(item_env);
    }
    cmd_env
}

fn read_config(filename: &str) -> Config {
    let msg = format!("Failed to read config: {}", filename);
    let contents = fs::read_to_string(filename).expect(&msg);
    let config: Config = toml::from_str(&contents).unwrap();
    config
}

fn main() {
    let matches = App::new(crate_name!())
        .version(crate_version!())
        .about(crate_description!())
        .arg(
            Arg::with_name("config")
                .long("config")
                .value_name("FILE")
                .help("The config file to use")
                .takes_value(true),
        )
        .arg(
            Arg::with_name("command_string")
                .short("c")
                .help("The command string to execute")
                .conflicts_with("command")
                .takes_value(true)
                .required(true),
        )
        .arg(
            Arg::with_name("environment")
                .short("e")
                .help("The sedo environment to use")
                .takes_value(true),
        )
        .arg(
            Arg::with_name("interactive")
                .short("i")
                .long("interactive")
                .help("Pass the -i flag to the subshell for the command"),
        )
        .arg(
            Arg::with_name("login")
                .short("l")
                .long("login")
                .help("Pass the -l flag to the subshell for the command"),
        )
        .arg(
            Arg::with_name("command")
                .help("The command to run")
                .conflicts_with("command_string")
                .multiple(true)
                .required(true)
                .index(1),
        )
        .get_matches();

    // check for command XOR command_string
    let cmd: String = if let Some(command) = matches.values_of("command") {
        shell_words::join::<Vec<&str>, &str>(command.collect())
    } else if let Some(command_string) = matches.value_of("command_string") {
        String::from(command_string)
    } else {
        panic!("command or command_string must be set");
    };

    let cfg = match matches.value_of("config") {
        Some(cfg_file_path) => read_config(cfg_file_path),
        _ => match env::var("HOME") {
            Ok(val) => {
                let file = format!("{}/.config/sedo/conf.toml", val);
                read_config(&file)
            }
            Err(_) => panic!("HOME not set for current user"),
        },
    };

    let sedo_env = matches
        .value_of("environment")
        .or_else(|| Some(&cfg.sedo.default_env))
        .unwrap();

    let env_cfg = cfg
        .sedo
        .environments
        .iter()
        .find(|env| sedo_env == env.name)
        .unwrap_or_else(|| {
            let msg = format!("Could not find environment '{}' in config", sedo_env);
            panic!(msg);
        });

    let cmd_env = build_env(env_cfg, cfg.sedo.op_path.as_deref());

    let interactive = matches.is_present("interactive");
    let login = matches.is_present("login");
    run_cmd(&sedo_env, cmd, cmd_env, interactive, login);
}
