#![deny(missing_docs)]
//! Drop-in replacement for the Debug derive macro that hides fields marked as sensitive.
//!
//! # Example
//! The following code snippet
//! ```rust
//! # extern crate securefmt;
//! # use securefmt::Debug;
//!
//! #[derive(Debug)]
//! struct SensitiveData {
//!     id: u8,
//!     #[sensitive]
//!     secret: u8
//! }
//!
//! fn main() {
//!     println!("{:?}", SensitiveData { id: 1, secret: 42 })
//! }
//! ```
//! will print:
//! ```output
//! SensitiveData { id: 1, secret: <redacted> }
//! ```
//!
//! If the 'debug_mode' feature is active, the same code will print:
//! ```output
//! WARN - WARNING: securefmt debug_mode feature is active. Sensitive data may be leaked. It is strongly recommended to disable debug_mode in production releases.
//! SensitiveData { id: 1, secret: 42 }
//! ```

// use features::CompileTimeFeatureDiscriminator;
use proc_macro::TokenStream;
use proc_macro_error::{abort, proc_macro_error};
use syn::{parse_macro_input, DeriveInput};
// use crate::implementation::SynDebugImplGenerator;

#[macro_use]
mod errors;
mod formatting_data;
mod implementation;
mod derive;

extern crate proc_macro;

/// # Usage
/// ```rust
/// # extern crate securefmt;
/// # use securefmt::Debug;
/// #[derive(Debug)]
/// struct SensitiveData {
///     id: u8,
///     #[sensitive]
///     secret: u8
/// }
/// ```
#[proc_macro_derive(Debug, attributes(sensitive))]
#[proc_macro_error]
pub fn derive_secure_debug(input: TokenStream) -> TokenStream {
    let derive_input = parse_macro_input!(input as DeriveInput);
    let derive_result = derive::debug(
        &derive_input.ident,
        &derive_input.data);

    match derive_result {
        Ok(tokens) => tokens.into(),
        Err(e) => abort!(e.call_site, e.message),
    }
}
