/**********************************************************************
 * Copyright (c) 2015 Gregory Maxwell                                 *
 * Distributed under the MIT software license, see the accompanying   *
 * file COPYING or http://www.opensource.org/licenses/mit-license.php.*
 **********************************************************************/

#ifndef SECP256K1_MODULE_RANGEPROOF_TESTS
#define SECP256K1_MODULE_RANGEPROOF_TESTS

#include <string.h>

#include "group.h"
#include "scalar.h"
#include "testrand.h"
#include "util.h"

#include "include/secp256k1_rangeproof.h"

static void test_pedersen_api(const rustsecp256k1zkp_v0_5_0_context *none, const rustsecp256k1zkp_v0_5_0_context *sign, const rustsecp256k1zkp_v0_5_0_context *vrfy, const int32_t *ecount) {
    rustsecp256k1zkp_v0_5_0_pedersen_commitment commit;
    const rustsecp256k1zkp_v0_5_0_pedersen_commitment *commit_ptr = &commit;
    unsigned char blind[32];
    unsigned char blind_out[32];
    const unsigned char *blind_ptr = blind;
    unsigned char *blind_out_ptr = blind_out;
    uint64_t val = rustsecp256k1zkp_v0_5_0_testrand32();

    rustsecp256k1zkp_v0_5_0_testrand256(blind);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_commit(none, &commit, blind, val, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
    CHECK(*ecount == 1);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_commit(vrfy, &commit, blind, val, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
    CHECK(*ecount == 2);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_commit(sign, &commit, blind, val, rustsecp256k1zkp_v0_5_0_generator_h) != 0);
    CHECK(*ecount == 2);

    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_commit(sign, NULL, blind, val, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
    CHECK(*ecount == 3);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_commit(sign, &commit, NULL, val, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
    CHECK(*ecount == 4);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_commit(sign, &commit, blind, val, NULL) == 0);
    CHECK(*ecount == 5);

    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_blind_sum(none, blind_out, &blind_ptr, 1, 1) != 0);
    CHECK(*ecount == 5);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_blind_sum(none, NULL, &blind_ptr, 1, 1) == 0);
    CHECK(*ecount == 6);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_blind_sum(none, blind_out, NULL, 1, 1) == 0);
    CHECK(*ecount == 7);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_blind_sum(none, blind_out, &blind_ptr, 0, 1) == 0);
    CHECK(*ecount == 8);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_blind_sum(none, blind_out, &blind_ptr, 0, 0) != 0);
    CHECK(*ecount == 8);

    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_commit(sign, &commit, blind, val, rustsecp256k1zkp_v0_5_0_generator_h) != 0);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_verify_tally(none, &commit_ptr, 1, &commit_ptr, 1) != 0);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_verify_tally(none, NULL, 0, &commit_ptr, 1) == 0);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_verify_tally(none, &commit_ptr, 1, NULL, 0) == 0);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_verify_tally(none, NULL, 0, NULL, 0) != 0);
    CHECK(*ecount == 8);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_verify_tally(none, NULL, 1, &commit_ptr, 1) == 0);
    CHECK(*ecount == 9);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_verify_tally(none, &commit_ptr, 1, NULL, 1) == 0);
    CHECK(*ecount == 10);

    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_blind_generator_blind_sum(none, &val, &blind_ptr, &blind_out_ptr, 1, 0) != 0);
    CHECK(*ecount == 10);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_blind_generator_blind_sum(none, &val, &blind_ptr, &blind_out_ptr, 1, 1) == 0);
    CHECK(*ecount == 11);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_blind_generator_blind_sum(none, &val, &blind_ptr, &blind_out_ptr, 0, 0) == 0);
    CHECK(*ecount == 12);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_blind_generator_blind_sum(none, NULL, &blind_ptr, &blind_out_ptr, 1, 0) == 0);
    CHECK(*ecount == 13);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_blind_generator_blind_sum(none, &val, NULL, &blind_out_ptr, 1, 0) == 0);
    CHECK(*ecount == 14);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_blind_generator_blind_sum(none, &val, &blind_ptr, NULL, 1, 0) == 0);
    CHECK(*ecount == 15);
}

static void test_rangeproof_api(const rustsecp256k1zkp_v0_5_0_context *none, const rustsecp256k1zkp_v0_5_0_context *sign, const rustsecp256k1zkp_v0_5_0_context *vrfy, const rustsecp256k1zkp_v0_5_0_context *both, const int32_t *ecount) {
    unsigned char proof[5134];
    unsigned char blind[32];
    rustsecp256k1zkp_v0_5_0_pedersen_commitment commit;
    uint64_t vmin = rustsecp256k1zkp_v0_5_0_testrand32();
    uint64_t val = vmin + rustsecp256k1zkp_v0_5_0_testrand32();
    size_t len = sizeof(proof);
    /* we'll switch to dylan thomas for this one */
    const unsigned char message[68] = "My tears are like the quiet drift / Of petals from some magic rose;";
    size_t mlen = sizeof(message);
    const unsigned char ext_commit[72] = "And all my grief flows from the rift / Of unremembered skies and snows.";
    size_t ext_commit_len = sizeof(ext_commit);

    rustsecp256k1zkp_v0_5_0_testrand256(blind);
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_commit(ctx, &commit, blind, val, rustsecp256k1zkp_v0_5_0_generator_h));

    CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(none, proof, &len, vmin, &commit, blind, commit.data, 0, 0, val, message, mlen, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
    CHECK(*ecount == 1);
    CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(sign, proof, &len, vmin, &commit, blind, commit.data, 0, 0, val, message, mlen, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
    CHECK(*ecount == 2);
    CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(vrfy, proof, &len, vmin, &commit, blind, commit.data, 0, 0, val, message, mlen, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
    CHECK(*ecount == 3);
    CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(both, proof, &len, vmin, &commit, blind, commit.data, 0, 0, val, message, mlen, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) != 0);
    CHECK(*ecount == 3);

    CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(both, NULL, &len, vmin, &commit, blind, commit.data, 0, 0, val, message, mlen, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
    CHECK(*ecount == 4);
    CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(both, proof, NULL, vmin, &commit, blind, commit.data, 0, 0, val, message, mlen, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
    CHECK(*ecount == 5);
    CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(both, proof, &len, vmin, NULL, blind, commit.data, 0, 0, val, message, mlen, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
    CHECK(*ecount == 6);
    CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(both, proof, &len, vmin, &commit, NULL, commit.data, 0, 0, val, message, mlen, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
    CHECK(*ecount == 7);
    CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(both, proof, &len, vmin, &commit, blind, NULL, 0, 0, val, message, mlen, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
    CHECK(*ecount == 8);
    CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(both, proof, &len, vmin, &commit, blind, commit.data, 0, 0, vmin - 1, message, mlen, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
    CHECK(*ecount == 8);
    CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(both, proof, &len, vmin, &commit, blind, commit.data, 0, 0, val, NULL, mlen, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
    CHECK(*ecount == 9);
    CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(both, proof, &len, vmin, &commit, blind, commit.data, 0, 0, val, NULL, 0, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) != 0);
    CHECK(*ecount == 9);
    CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(both, proof, &len, vmin, &commit, blind, commit.data, 0, 0, val, NULL, 0, NULL, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
    CHECK(*ecount == 10);
    CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(both, proof, &len, vmin, &commit, blind, commit.data, 0, 0, val, NULL, 0, NULL, 0, rustsecp256k1zkp_v0_5_0_generator_h) != 0);
    CHECK(*ecount == 10);
    CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(both, proof, &len, vmin, &commit, blind, commit.data, 0, 0, val, NULL, 0, NULL, 0, NULL) == 0);
    CHECK(*ecount == 11);

    CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(both, proof, &len, vmin, &commit, blind, commit.data, 0, 0, val, message, mlen, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) != 0);
    {
        int exp;
        int mantissa;
        uint64_t min_value;
        uint64_t max_value;
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_info(none, &exp, &mantissa, &min_value, &max_value, proof, len) != 0);
        CHECK(exp == 0);
        CHECK(((uint64_t) 1 << mantissa) > val - vmin);
        CHECK(((uint64_t) 1 << (mantissa - 1)) <= val - vmin);
        CHECK(min_value == vmin);
        CHECK(max_value >= val);

        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_info(none, NULL, &mantissa, &min_value, &max_value, proof, len) == 0);
        CHECK(*ecount == 12);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_info(none, &exp, NULL, &min_value, &max_value, proof, len) == 0);
        CHECK(*ecount == 13);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_info(none, &exp, &mantissa, NULL, &max_value, proof, len) == 0);
        CHECK(*ecount == 14);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_info(none, &exp, &mantissa, &min_value, NULL, proof, len) == 0);
        CHECK(*ecount == 15);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_info(none, &exp, &mantissa, &min_value, &max_value, NULL, len) == 0);
        CHECK(*ecount == 16);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_info(none, &exp, &mantissa, &min_value, &max_value, proof, 0) == 0);
        CHECK(*ecount == 16);
    }
    {
        uint64_t min_value;
        uint64_t max_value;
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_verify(none, &min_value, &max_value, &commit, proof, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 17);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_verify(sign, &min_value, &max_value, &commit, proof, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 18);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_verify(vrfy, &min_value, &max_value, &commit, proof, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) != 0);
        CHECK(*ecount == 18);

        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_verify(vrfy, NULL, &max_value, &commit, proof, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 19);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_verify(vrfy, &min_value, NULL, &commit, proof, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 20);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_verify(vrfy, &min_value, &max_value, NULL, proof, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 21);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_verify(vrfy, &min_value, &max_value, &commit, NULL, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 22);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_verify(vrfy, &min_value, &max_value, &commit, proof, 0, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 22);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_verify(vrfy, &min_value, &max_value, &commit, proof, len, NULL, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 23);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_verify(vrfy, &min_value, &max_value, &commit, proof, len, NULL, 0, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 23);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_verify(vrfy, &min_value, &max_value, &commit, proof, len, NULL, 0, NULL) == 0);
        CHECK(*ecount == 24);
    }
    {
        unsigned char blind_out[32];
        unsigned char message_out[68];
        uint64_t value_out;
        uint64_t min_value;
        uint64_t max_value;
        size_t message_len = sizeof(message_out);

        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(none, blind_out, &value_out, message_out, &message_len, commit.data, &min_value, &max_value, &commit, proof, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 25);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(sign, blind_out, &value_out, message_out, &message_len, commit.data, &min_value, &max_value, &commit, proof, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 26);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(vrfy, blind_out, &value_out, message_out, &message_len, commit.data, &min_value, &max_value, &commit, proof, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 27);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(both, blind_out, &value_out, message_out, &message_len, commit.data, &min_value, &max_value, &commit, proof, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) != 0);
        CHECK(*ecount == 27);

        CHECK(min_value == vmin);
        CHECK(max_value >= val);
        CHECK(value_out == val);
        CHECK(message_len == sizeof(message_out));
        CHECK(memcmp(message, message_out, sizeof(message_out)) == 0);

        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(both, NULL, &value_out, message_out, &message_len, commit.data, &min_value, &max_value, &commit, proof, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) != 0);
        CHECK(*ecount == 27);  /* blindout may be NULL */
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(both, blind_out, NULL, message_out, &message_len, commit.data, &min_value, &max_value, &commit, proof, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) != 0);
        CHECK(*ecount == 27);  /* valueout may be NULL */
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(both, blind_out, &value_out, NULL, &message_len, commit.data, &min_value, &max_value, &commit, proof, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 28);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(both, blind_out, &value_out, NULL, 0, commit.data, &min_value, &max_value, &commit, proof, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) != 0);
        CHECK(*ecount == 28);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(both, blind_out, &value_out, NULL, 0, NULL, &min_value, &max_value, &commit, proof, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 29);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(both, blind_out, &value_out, NULL, 0, commit.data, NULL, &max_value, &commit, proof, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 30);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(both, blind_out, &value_out, NULL, 0, commit.data, &min_value, NULL, &commit, proof, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 31);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(both, blind_out, &value_out, NULL, 0, commit.data, &min_value, &max_value, NULL, proof, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 32);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(both, blind_out, &value_out, NULL, 0, commit.data, &min_value, &max_value, &commit, NULL, len, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 33);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(both, blind_out, &value_out, NULL, 0, commit.data, &min_value, &max_value, &commit, proof, 0, ext_commit, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 33);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(both, blind_out, &value_out, NULL, 0, commit.data, &min_value, &max_value, &commit, proof, len, NULL, ext_commit_len, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 34);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(both, blind_out, &value_out, NULL, 0, commit.data, &min_value, &max_value, &commit, proof, len, NULL, 0, rustsecp256k1zkp_v0_5_0_generator_h) == 0);
        CHECK(*ecount == 34);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(both, blind_out, &value_out, NULL, 0, commit.data, &min_value, &max_value, &commit, proof, len, NULL, 0, NULL) == 0);
        CHECK(*ecount == 35);
    }
}

static void test_api(void) {
    rustsecp256k1zkp_v0_5_0_context *none = rustsecp256k1zkp_v0_5_0_context_create(SECP256K1_CONTEXT_NONE);
    rustsecp256k1zkp_v0_5_0_context *sign = rustsecp256k1zkp_v0_5_0_context_create(SECP256K1_CONTEXT_SIGN);
    rustsecp256k1zkp_v0_5_0_context *vrfy = rustsecp256k1zkp_v0_5_0_context_create(SECP256K1_CONTEXT_VERIFY);
    rustsecp256k1zkp_v0_5_0_context *both = rustsecp256k1zkp_v0_5_0_context_create(SECP256K1_CONTEXT_SIGN | SECP256K1_CONTEXT_VERIFY);
    int32_t ecount;
    int i;

    rustsecp256k1zkp_v0_5_0_context_set_error_callback(none, counting_illegal_callback_fn, &ecount);
    rustsecp256k1zkp_v0_5_0_context_set_error_callback(sign, counting_illegal_callback_fn, &ecount);
    rustsecp256k1zkp_v0_5_0_context_set_error_callback(vrfy, counting_illegal_callback_fn, &ecount);
    rustsecp256k1zkp_v0_5_0_context_set_error_callback(both, counting_illegal_callback_fn, &ecount);
    rustsecp256k1zkp_v0_5_0_context_set_illegal_callback(none, counting_illegal_callback_fn, &ecount);
    rustsecp256k1zkp_v0_5_0_context_set_illegal_callback(sign, counting_illegal_callback_fn, &ecount);
    rustsecp256k1zkp_v0_5_0_context_set_illegal_callback(vrfy, counting_illegal_callback_fn, &ecount);
    rustsecp256k1zkp_v0_5_0_context_set_illegal_callback(both, counting_illegal_callback_fn, &ecount);

    for (i = 0; i < count; i++) {
        ecount = 0;
        test_pedersen_api(none, sign, vrfy, &ecount);
        ecount = 0;
        test_rangeproof_api(none, sign, vrfy, both, &ecount);
    }

    rustsecp256k1zkp_v0_5_0_context_destroy(none);
    rustsecp256k1zkp_v0_5_0_context_destroy(sign);
    rustsecp256k1zkp_v0_5_0_context_destroy(vrfy);
    rustsecp256k1zkp_v0_5_0_context_destroy(both);
}

static void test_pedersen(void) {
    rustsecp256k1zkp_v0_5_0_pedersen_commitment commits[19];
    const rustsecp256k1zkp_v0_5_0_pedersen_commitment *cptr[19];
    unsigned char blinds[32*19];
    const unsigned char *bptr[19];
    rustsecp256k1zkp_v0_5_0_scalar s;
    uint64_t values[19];
    int64_t totalv;
    int i;
    int inputs;
    int outputs;
    int total;
    inputs = (rustsecp256k1zkp_v0_5_0_testrand32() & 7) + 1;
    outputs = (rustsecp256k1zkp_v0_5_0_testrand32() & 7) + 2;
    total = inputs + outputs;
    for (i = 0; i < 19; i++) {
        cptr[i] = &commits[i];
        bptr[i] = &blinds[i * 32];
    }
    totalv = 0;
    for (i = 0; i < inputs; i++) {
        values[i] = rustsecp256k1zkp_v0_5_0_testrandi64(0, INT64_MAX - totalv);
        totalv += values[i];
    }
    for (i = 0; i < outputs - 1; i++) {
        values[i + inputs] = rustsecp256k1zkp_v0_5_0_testrandi64(0, totalv);
        totalv -= values[i + inputs];
    }
    values[total - 1] = totalv;

    for (i = 0; i < total - 1; i++) {
        random_scalar_order(&s);
        rustsecp256k1zkp_v0_5_0_scalar_get_b32(&blinds[i * 32], &s);
    }
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_blind_sum(ctx, &blinds[(total - 1) * 32], bptr, total - 1, inputs));
    for (i = 0; i < total; i++) {
        CHECK(rustsecp256k1zkp_v0_5_0_pedersen_commit(ctx, &commits[i], &blinds[i * 32], values[i], rustsecp256k1zkp_v0_5_0_generator_h));
    }
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_verify_tally(ctx, cptr, inputs, &cptr[inputs], outputs));
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_verify_tally(ctx, &cptr[inputs], outputs, cptr, inputs));
    if (inputs > 0 && values[0] > 0) {
        CHECK(!rustsecp256k1zkp_v0_5_0_pedersen_verify_tally(ctx, cptr, inputs - 1, &cptr[inputs], outputs));
    }
    random_scalar_order(&s);
    for (i = 0; i < 4; i++) {
        rustsecp256k1zkp_v0_5_0_scalar_get_b32(&blinds[i * 32], &s);
    }
    values[0] = INT64_MAX;
    values[1] = 0;
    values[2] = 1;
    for (i = 0; i < 3; i++) {
        CHECK(rustsecp256k1zkp_v0_5_0_pedersen_commit(ctx, &commits[i], &blinds[i * 32], values[i], rustsecp256k1zkp_v0_5_0_generator_h));
    }
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_verify_tally(ctx, &cptr[0], 1, &cptr[0], 1));
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_verify_tally(ctx, &cptr[1], 1, &cptr[1], 1));
}

static void test_borromean(void) {
    unsigned char e0[32];
    rustsecp256k1zkp_v0_5_0_scalar s[64];
    rustsecp256k1zkp_v0_5_0_gej pubs[64];
    rustsecp256k1zkp_v0_5_0_scalar k[8];
    rustsecp256k1zkp_v0_5_0_scalar sec[8];
    rustsecp256k1zkp_v0_5_0_ge ge;
    rustsecp256k1zkp_v0_5_0_scalar one;
    unsigned char m[32];
    size_t rsizes[8];
    size_t secidx[8];
    size_t nrings;
    size_t i;
    size_t j;
    int c;
    rustsecp256k1zkp_v0_5_0_testrand256_test(m);
    nrings = 1 + (rustsecp256k1zkp_v0_5_0_testrand32()&7);
    c = 0;
    rustsecp256k1zkp_v0_5_0_scalar_set_int(&one, 1);
    if (rustsecp256k1zkp_v0_5_0_testrand32()&1) {
        rustsecp256k1zkp_v0_5_0_scalar_negate(&one, &one);
    }
    for (i = 0; i < nrings; i++) {
        rsizes[i] = 1 + (rustsecp256k1zkp_v0_5_0_testrand32()&7);
        secidx[i] = rustsecp256k1zkp_v0_5_0_testrand32() % rsizes[i];
        random_scalar_order(&sec[i]);
        random_scalar_order(&k[i]);
        if(rustsecp256k1zkp_v0_5_0_testrand32()&7) {
            sec[i] = one;
        }
        if(rustsecp256k1zkp_v0_5_0_testrand32()&7) {
            k[i] = one;
        }
        for (j = 0; j < rsizes[i]; j++) {
            random_scalar_order(&s[c + j]);
            if(rustsecp256k1zkp_v0_5_0_testrand32()&7) {
                s[i] = one;
            }
            if (j == secidx[i]) {
                rustsecp256k1zkp_v0_5_0_ecmult_gen(&ctx->ecmult_gen_ctx, &pubs[c + j], &sec[i]);
            } else {
                random_group_element_test(&ge);
                random_group_element_jacobian_test(&pubs[c + j],&ge);
            }
        }
        c += rsizes[i];
    }
    CHECK(rustsecp256k1zkp_v0_5_0_borromean_sign(&ctx->ecmult_ctx, &ctx->ecmult_gen_ctx, e0, s, pubs, k, sec, rsizes, secidx, nrings, m, 32));
    CHECK(rustsecp256k1zkp_v0_5_0_borromean_verify(&ctx->ecmult_ctx, NULL, e0, s, pubs, rsizes, nrings, m, 32));
    i = rustsecp256k1zkp_v0_5_0_testrand32() % c;
    rustsecp256k1zkp_v0_5_0_scalar_negate(&s[i],&s[i]);
    CHECK(!rustsecp256k1zkp_v0_5_0_borromean_verify(&ctx->ecmult_ctx, NULL, e0, s, pubs, rsizes, nrings, m, 32));
    rustsecp256k1zkp_v0_5_0_scalar_negate(&s[i],&s[i]);
    rustsecp256k1zkp_v0_5_0_scalar_set_int(&one, 1);
    for(j = 0; j < 4; j++) {
        i = rustsecp256k1zkp_v0_5_0_testrand32() % c;
        if (rustsecp256k1zkp_v0_5_0_testrand32() & 1) {
            rustsecp256k1zkp_v0_5_0_gej_double_var(&pubs[i],&pubs[i], NULL);
        } else {
            rustsecp256k1zkp_v0_5_0_scalar_add(&s[i],&s[i],&one);
        }
        CHECK(!rustsecp256k1zkp_v0_5_0_borromean_verify(&ctx->ecmult_ctx, NULL, e0, s, pubs, rsizes, nrings, m, 32));
    }
}

static void test_rangeproof(void) {
    const uint64_t testvs[11] = {0, 1, 5, 11, 65535, 65537, INT32_MAX, UINT32_MAX, INT64_MAX - 1, INT64_MAX, UINT64_MAX};
    rustsecp256k1zkp_v0_5_0_pedersen_commitment commit;
    rustsecp256k1zkp_v0_5_0_pedersen_commitment commit2;
    unsigned char proof[5134 + 1]; /* One additional byte to test if trailing bytes are rejected */
    unsigned char blind[32];
    unsigned char blindout[32];
    unsigned char message[4096];
    size_t mlen;
    uint64_t v;
    uint64_t vout;
    uint64_t vmin;
    uint64_t minv;
    uint64_t maxv;
    size_t len;
    size_t i;
    size_t j;
    size_t k;
    /* Short message is a Simone de Beauvoir quote */
    const unsigned char message_short[120] = "When I see my own likeness in the depths of someone else's consciousness,  I always experience a moment of panic.";
    /* Long message is 0xA5 with a bunch of this quote in the middle */
    unsigned char message_long[3968];
    memset(message_long, 0xa5, sizeof(message_long));
    for (i = 1200; i < 3600; i += 120) {
        memcpy(&message_long[i], message_short, sizeof(message_short));
    }

    rustsecp256k1zkp_v0_5_0_testrand256(blind);
    for (i = 0; i < 11; i++) {
        v = testvs[i];
        CHECK(rustsecp256k1zkp_v0_5_0_pedersen_commit(ctx, &commit, blind, v, rustsecp256k1zkp_v0_5_0_generator_h));
        for (vmin = 0; vmin < (i<9 && i > 0 ? 2 : 1); vmin++) {
            const unsigned char *input_message = NULL;
            size_t input_message_len = 0;
            /* vmin is always either 0 or 1; if it is 1, then we have no room for a message.
             * If it's 0, we use "minimum encoding" and only have room for a small message when
             * `testvs[i]` is >= 4; for a large message when it's >= 2^32. */
            if (vmin == 0 && i > 2) {
                input_message = message_short;
                input_message_len = sizeof(message_short);
            }
            if (vmin == 0 && i > 7) {
                input_message = message_long;
                input_message_len = sizeof(message_long);
            }
            len = 5134;
            CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(ctx, proof, &len, vmin, &commit, blind, commit.data, 0, 0, v, input_message, input_message_len, NULL, 0, rustsecp256k1zkp_v0_5_0_generator_h));
            CHECK(len <= 5134);
            mlen = 4096;
            CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(ctx, blindout, &vout, message, &mlen, commit.data, &minv, &maxv, &commit, proof, len, NULL, 0, rustsecp256k1zkp_v0_5_0_generator_h));
            if (input_message != NULL) {
                CHECK(memcmp(message, input_message, input_message_len) == 0);
            }
            for (j = input_message_len; j < mlen; j++) {
                CHECK(message[j] == 0);
            }
            CHECK(mlen <= 4096);
            CHECK(memcmp(blindout, blind, 32) == 0);
            CHECK(vout == v);
            CHECK(minv <= v);
            CHECK(maxv >= v);
            len = 5134;
            CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(ctx, proof, &len, v, &commit, blind, commit.data, -1, 64, v, NULL, 0, NULL, 0, rustsecp256k1zkp_v0_5_0_generator_h));
            CHECK(len <= 73);
            CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(ctx, blindout, &vout, NULL, NULL, commit.data, &minv, &maxv, &commit, proof, len, NULL, 0, rustsecp256k1zkp_v0_5_0_generator_h));
            CHECK(memcmp(blindout, blind, 32) == 0);
            CHECK(vout == v);
            CHECK(minv == v);
            CHECK(maxv == v);

            /* Check with a committed message */
            len = 5134;
            CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(ctx, proof, &len, v, &commit, blind, commit.data, -1, 64, v, NULL, 0, message_short, sizeof(message_short), rustsecp256k1zkp_v0_5_0_generator_h));
            CHECK(len <= 73);
            CHECK(!rustsecp256k1zkp_v0_5_0_rangeproof_rewind(ctx, blindout, &vout, NULL, NULL, commit.data, &minv, &maxv, &commit, proof, len, NULL, 0, rustsecp256k1zkp_v0_5_0_generator_h));
            CHECK(!rustsecp256k1zkp_v0_5_0_rangeproof_rewind(ctx, blindout, &vout, NULL, NULL, commit.data, &minv, &maxv, &commit, proof, len, message_long, sizeof(message_long), rustsecp256k1zkp_v0_5_0_generator_h));
            CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(ctx, blindout, &vout, NULL, NULL, commit.data, &minv, &maxv, &commit, proof, len, message_short, sizeof(message_short), rustsecp256k1zkp_v0_5_0_generator_h));
            CHECK(memcmp(blindout, blind, 32) == 0);
            CHECK(vout == v);
            CHECK(minv == v);
            CHECK(maxv == v);
        }
    }
    rustsecp256k1zkp_v0_5_0_testrand256(blind);
    v = INT64_MAX - 1;
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_commit(ctx, &commit, blind, v, rustsecp256k1zkp_v0_5_0_generator_h));
    for (i = 0; i < 19; i++) {
        len = 5134;
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(ctx, proof, &len, 0, &commit, blind, commit.data, i, 0, v, NULL, 0, NULL, 0, rustsecp256k1zkp_v0_5_0_generator_h));
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_verify(ctx, &minv, &maxv, &commit, proof, len, NULL, 0, rustsecp256k1zkp_v0_5_0_generator_h));
        CHECK(len <= 5134);
        CHECK(minv <= v);
        CHECK(maxv >= v);
        /* Make sure it fails when validating with a committed message */
        CHECK(!rustsecp256k1zkp_v0_5_0_rangeproof_verify(ctx, &minv, &maxv, &commit, proof, len, message_short, sizeof(message_short), rustsecp256k1zkp_v0_5_0_generator_h));
    }
    rustsecp256k1zkp_v0_5_0_testrand256(blind);
    {
        /*Malleability test.*/
        v = rustsecp256k1zkp_v0_5_0_testrandi64(0, 255);
        CHECK(rustsecp256k1zkp_v0_5_0_pedersen_commit(ctx, &commit, blind, v, rustsecp256k1zkp_v0_5_0_generator_h));
        len = 5134;
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(ctx, proof, &len, 0, &commit, blind, commit.data, 0, 3, v, NULL, 0, NULL, 0, rustsecp256k1zkp_v0_5_0_generator_h));
        CHECK(len <= 5134);
        /* Test if trailing bytes are rejected. */
        proof[len] = v;
        CHECK(!rustsecp256k1zkp_v0_5_0_rangeproof_verify(ctx, &minv, &maxv, &commit, proof, len + 1, NULL, 0, rustsecp256k1zkp_v0_5_0_generator_h));
        for (i = 0; i < len*8; i++) {
            proof[i >> 3] ^= 1 << (i & 7);
            CHECK(!rustsecp256k1zkp_v0_5_0_rangeproof_verify(ctx, &minv, &maxv, &commit, proof, len, NULL, 0, rustsecp256k1zkp_v0_5_0_generator_h));
            proof[i >> 3] ^= 1 << (i & 7);
        }
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_verify(ctx, &minv, &maxv, &commit, proof, len, NULL, 0, rustsecp256k1zkp_v0_5_0_generator_h));
        CHECK(minv <= v);
        CHECK(maxv >= v);
    }
    memcpy(&commit2, &commit, sizeof(commit));
    for (i = 0; i < (size_t) count; i++) {
        int exp;
        int min_bits;
        v = rustsecp256k1zkp_v0_5_0_testrandi64(0, UINT64_MAX >> (rustsecp256k1zkp_v0_5_0_testrand32()&63));
        vmin = 0;
        if ((v < INT64_MAX) && (rustsecp256k1zkp_v0_5_0_testrand32()&1)) {
            vmin = rustsecp256k1zkp_v0_5_0_testrandi64(0, v);
        }
        rustsecp256k1zkp_v0_5_0_testrand256(blind);
        CHECK(rustsecp256k1zkp_v0_5_0_pedersen_commit(ctx, &commit, blind, v, rustsecp256k1zkp_v0_5_0_generator_h));
        len = 5134;
        exp = (int)rustsecp256k1zkp_v0_5_0_testrandi64(0,18)-(int)rustsecp256k1zkp_v0_5_0_testrandi64(0,18);
        if (exp < 0) {
            exp = -exp;
        }
        min_bits = (int)rustsecp256k1zkp_v0_5_0_testrandi64(0,64)-(int)rustsecp256k1zkp_v0_5_0_testrandi64(0,64);
        if (min_bits < 0) {
            min_bits = -min_bits;
        }
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_sign(ctx, proof, &len, vmin, &commit, blind, commit.data, exp, min_bits, v, NULL, 0, NULL, 0, rustsecp256k1zkp_v0_5_0_generator_h));
        CHECK(len <= 5134);
        mlen = 4096;
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(ctx, blindout, &vout, message, &mlen, commit.data, &minv, &maxv, &commit, proof, len, NULL, 0, rustsecp256k1zkp_v0_5_0_generator_h));
        for (j = 0; j < mlen; j++) {
            CHECK(message[j] == 0);
        }
        CHECK(mlen <= 4096);
        CHECK(memcmp(blindout, blind, 32) == 0);

        CHECK(minv <= v);
        CHECK(maxv >= v);
        CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_rewind(ctx, blindout, &vout, NULL, NULL, commit.data, &minv, &maxv, &commit, proof, len, NULL, 0, rustsecp256k1zkp_v0_5_0_generator_h));
        memcpy(&commit2, &commit, sizeof(commit));
    }
    for (j = 0; j < 3; j++) {
        for (i = 0; i < 96; i++) {
            rustsecp256k1zkp_v0_5_0_testrand256(&proof[i * 32]);
        }
        for (k = 0; k < 128; k += 3) {
            len = k;
            CHECK(!rustsecp256k1zkp_v0_5_0_rangeproof_verify(ctx, &minv, &maxv, &commit2, proof, len, NULL, 0, rustsecp256k1zkp_v0_5_0_generator_h));
        }
        len = rustsecp256k1zkp_v0_5_0_testrandi64(0, 3072);
        CHECK(!rustsecp256k1zkp_v0_5_0_rangeproof_verify(ctx, &minv, &maxv, &commit2, proof, len, NULL, 0, rustsecp256k1zkp_v0_5_0_generator_h));
    }
}

#define MAX_N_GENS	30
void test_multiple_generators(void) {
    const size_t n_inputs = (rustsecp256k1zkp_v0_5_0_testrand32() % (MAX_N_GENS / 2)) + 1;
    const size_t n_outputs = (rustsecp256k1zkp_v0_5_0_testrand32() % (MAX_N_GENS / 2)) + 1;
    const size_t n_generators = n_inputs + n_outputs;
    unsigned char *generator_blind[MAX_N_GENS];
    unsigned char *pedersen_blind[MAX_N_GENS];
    rustsecp256k1zkp_v0_5_0_generator generator[MAX_N_GENS];
    rustsecp256k1zkp_v0_5_0_pedersen_commitment commit[MAX_N_GENS];
    const rustsecp256k1zkp_v0_5_0_pedersen_commitment *commit_ptr[MAX_N_GENS];
    size_t i;
    int64_t total_value;
    uint64_t value[MAX_N_GENS];

    rustsecp256k1zkp_v0_5_0_scalar s;

    unsigned char generator_seed[32];
    random_scalar_order(&s);
    rustsecp256k1zkp_v0_5_0_scalar_get_b32(generator_seed, &s);
    /* Create all the needed generators */
    for (i = 0; i < n_generators; i++) {
        generator_blind[i] = (unsigned char*) malloc(32);
        pedersen_blind[i] = (unsigned char*) malloc(32);

        random_scalar_order(&s);
        rustsecp256k1zkp_v0_5_0_scalar_get_b32(generator_blind[i], &s);
        random_scalar_order(&s);
        rustsecp256k1zkp_v0_5_0_scalar_get_b32(pedersen_blind[i], &s);

        CHECK(rustsecp256k1zkp_v0_5_0_generator_generate_blinded(ctx, &generator[i], generator_seed, generator_blind[i]));

        commit_ptr[i] = &commit[i];
    }

    /* Compute all the values -- can be positive or negative */
    total_value = 0;
    for (i = 0; i < n_outputs; i++) {
        value[n_inputs + i] = rustsecp256k1zkp_v0_5_0_testrandi64(0, INT64_MAX - total_value);
        total_value += value[n_inputs + i];
    }
    for (i = 0; i < n_inputs - 1; i++) {
        value[i] = rustsecp256k1zkp_v0_5_0_testrandi64(0, total_value);
        total_value -= value[i];
    }
    value[i] = total_value;

    /* Correct for blinding factors and do the commitments */
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_blind_generator_blind_sum(ctx, value, (const unsigned char * const *) generator_blind, pedersen_blind, n_generators, n_inputs));
    for (i = 0; i < n_generators; i++) {
        CHECK(rustsecp256k1zkp_v0_5_0_pedersen_commit(ctx, &commit[i], pedersen_blind[i], value[i], &generator[i]));
    }

    /* Verify */
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_verify_tally(ctx, &commit_ptr[0], n_inputs, &commit_ptr[n_inputs], n_outputs));

    /* Cleanup */
    for (i = 0; i < n_generators; i++) {
        free(generator_blind[i]);
        free(pedersen_blind[i]);
    }
}

void test_rangeproof_fixed_vectors(void) {
    const unsigned char vector_1[] = {
        0x62, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x56, 0x02, 0x2a, 0x5c, 0x42, 0x0e, 0x1d,
        0x51, 0xe1, 0xb7, 0xf3, 0x69, 0x04, 0xb5, 0xbb, 0x9b, 0x41, 0x66, 0x14, 0xf3, 0x64, 0x42, 0x26,
        0xe3, 0xa7, 0x6a, 0x06, 0xbb, 0xa8, 0x5a, 0x49, 0x6f, 0x19, 0x76, 0xfb, 0xe5, 0x75, 0x77, 0x88,
        0xab, 0xa9, 0x66, 0x44, 0x80, 0xea, 0x29, 0x95, 0x7f, 0xdf, 0x72, 0x4a, 0xaf, 0x02, 0xbe, 0xdd,
        0x5d, 0x15, 0xd8, 0xae, 0xff, 0x74, 0xc9, 0x8c, 0x1a, 0x67, 0x0e, 0xb2, 0x57, 0x22, 0x99, 0xc3,
        0x21, 0x46, 0x6f, 0x15, 0x58, 0x0e, 0xdb, 0xe6, 0x6e, 0xc4, 0x0d, 0xfe, 0x6f, 0x04, 0x6b, 0x0d,
        0x18, 0x3d, 0x78, 0x40, 0x98, 0x56, 0x4e, 0xe4, 0x4a, 0x74, 0x90, 0xa7, 0xac, 0x9c, 0x16, 0xe0,
        0x3e, 0x81, 0xaf, 0x0f, 0xe3, 0x4f, 0x34, 0x99, 0x52, 0xf7, 0xa7, 0xf6, 0xd3, 0x83, 0xa0, 0x17,
        0x4b, 0x2d, 0xa7, 0xd4, 0xfd, 0xf7, 0x84, 0x45, 0xc4, 0x11, 0x71, 0x3d, 0x4a, 0x22, 0x34, 0x09,
        0x9c, 0xa7, 0xe5, 0xc8, 0xba, 0x04, 0xbf, 0xfd, 0x25, 0x11, 0x7d, 0xa4, 0x43, 0x45, 0xc7, 0x62,
        0x9e, 0x7b, 0x80, 0xf6, 0x09, 0xbb, 0x1b, 0x2e, 0xf3, 0xcd, 0x23, 0xe0, 0xed, 0x81, 0x43, 0x42,
        0xbe, 0xc4, 0x9f, 0x58, 0x8a, 0x0d, 0x66, 0x79, 0x09, 0x70, 0x11, 0x68, 0x3d, 0x87, 0x38, 0x1c,
        0x3c, 0x85, 0x52, 0x5b, 0x62, 0xf7, 0x3e, 0x7e, 0x87, 0xa2, 0x99, 0x24, 0xd0, 0x7d, 0x18, 0x63,
        0x56, 0x48, 0xa4, 0x3a, 0xfe, 0x65, 0xfa, 0xa4, 0xd0, 0x67, 0xaa, 0x98, 0x65, 0x4d, 0xe4, 0x22,
        0x75, 0x45, 0x52, 0xe8, 0x41, 0xc7, 0xed, 0x38, 0xeb, 0xf5, 0x02, 0x90, 0xc9, 0x45, 0xa3, 0xb0,
        0x4d, 0x03, 0xd7, 0xab, 0x43, 0xe4, 0x21, 0xfc, 0x83, 0xd6, 0x12, 0x1d, 0x76, 0xb1, 0x3c, 0x67,
        0x63, 0x1f, 0x52, 0x9d, 0xc3, 0x23, 0x5c, 0x4e, 0xa6, 0x8d, 0x01, 0x4a, 0xba, 0x9a, 0xf4, 0x16,
        0x5b, 0x67, 0xc8, 0xe1, 0xd2, 0x42, 0x6d, 0xdf, 0xcd, 0x08, 0x6a, 0x73, 0x41, 0x6a, 0xc2, 0x84,
        0xc6, 0x31, 0xbe, 0x57, 0xcb, 0x0e, 0xde, 0xbf, 0x71, 0xd5, 0x8a, 0xf7, 0x24, 0xb2, 0xa7, 0x89,
        0x96, 0x62, 0x4f, 0xd9, 0xf7, 0xc3, 0xde, 0x4c, 0xab, 0x13, 0x72, 0xb4, 0xb3, 0x35, 0x04, 0x82,
        0xa8, 0x75, 0x1d, 0xde, 0x46, 0xa8, 0x0d, 0xb8, 0x23, 0x44, 0x00, 0x44, 0xfa, 0x53, 0x6c, 0x2d,
        0xce, 0xd3, 0xa6, 0x80, 0xa1, 0x20, 0xca, 0xd1, 0x63, 0xbb, 0xbe, 0x39, 0x5f, 0x9d, 0x27, 0x69,
        0xb3, 0x33, 0x1f, 0xdb, 0xda, 0x67, 0x05, 0x37, 0xbe, 0x65, 0xe9, 0x7e, 0xa9, 0xc3, 0xff, 0x37,
        0x8a, 0xb4, 0x2d, 0xfe, 0xf2, 0x16, 0x85, 0xc7, 0x0f, 0xd9, 0xbe, 0x14, 0xd1, 0x80, 0x14, 0x9f,
        0x58, 0x56, 0x98, 0x41, 0xf6, 0x26, 0xf7, 0xa2, 0x71, 0x66, 0xb4, 0x7a, 0x9c, 0x12, 0x73, 0xd3,
        0xdf, 0x77, 0x2b, 0x49, 0xe5, 0xca, 0x50, 0x57, 0x44, 0x6e, 0x3f, 0x58, 0x56, 0xbc, 0x21, 0x70,
        0x4f, 0xc6, 0xaa, 0x12, 0xff, 0x7c, 0xa7, 0x3d, 0xed, 0x46, 0xc1, 0x40, 0xe6, 0x58, 0x09, 0x2a,
        0xda, 0xb3, 0x76, 0xab, 0x44, 0xb5, 0x4e, 0xb3, 0x12, 0xe0, 0x26, 0x8a, 0x52, 0xac, 0x49, 0x1d,
        0xe7, 0x06, 0x53, 0x3a, 0x01, 0x35, 0x21, 0x2e, 0x86, 0x48, 0xc5, 0x75, 0xc1, 0xa2, 0x7d, 0x22,
        0x53, 0xf6, 0x3f, 0x41, 0xc5, 0xb3, 0x08, 0x7d, 0xa3, 0x67, 0xc0, 0xbb, 0xb6, 0x8d, 0xf0, 0xd3,
        0x01, 0x72, 0xd3, 0x63, 0x82, 0x01, 0x1a, 0xe7, 0x1d, 0x22, 0xfa, 0x95, 0x33, 0xf6, 0xf2, 0xde,
        0xa2, 0x53, 0x86, 0x55, 0x5a, 0xb4, 0x2e, 0x75, 0x75, 0xc6, 0xd5, 0x93, 0x9c, 0x57, 0xa9, 0x1f,
        0xb9, 0x3e, 0xe8, 0x1c, 0xbf, 0xac, 0x1c, 0x54, 0x6f, 0xf5, 0xab, 0x41, 0xee, 0xb3, 0x0e, 0xd0,
        0x76, 0xc4, 0x1a, 0x45, 0xcd, 0xf1, 0xd6, 0xcc, 0xb0, 0x83, 0x70, 0x73, 0xbc, 0x88, 0x74, 0xa0,
        0x5b, 0xe7, 0x98, 0x10, 0x36, 0xbf, 0xec, 0x23, 0x1c, 0xc2, 0xb5, 0xba, 0x4b, 0x9d, 0x7f, 0x8c,
        0x8a, 0xe2, 0xda, 0x18, 0xdd, 0xab, 0x27, 0x8a, 0x15, 0xeb, 0xb0, 0xd4, 0x3a, 0x8b, 0x77, 0x00,
        0xc7, 0xbb, 0xcc, 0xfa, 0xba, 0xa4, 0x6a, 0x17, 0x5c, 0xf8, 0x51, 0x5d, 0x8d, 0x16, 0xcd, 0xa7,
        0x0e, 0x71, 0x97, 0x98, 0x78, 0x5a, 0x41, 0xb3, 0xf0, 0x1f, 0x87, 0x2d, 0x65, 0xcd, 0x29, 0x49,
        0xd2, 0x87, 0x2c, 0x91, 0xa9, 0x5f, 0xcc, 0xa9, 0xd8, 0xbb, 0x53, 0x18, 0xe7, 0xd6, 0xec, 0x65,
        0xa6, 0x45, 0xf6, 0xce, 0xcf, 0x48, 0xf6, 0x1e, 0x3d, 0xd2, 0xcf, 0xcb, 0x3a, 0xcd, 0xbb, 0x92,
        0x29, 0x24, 0x16, 0x7f, 0x8a, 0xa8, 0x5c, 0x0c, 0x45, 0x71, 0x33
    };
    const unsigned char commit_1[] = {
        0x08,
        0xf5, 0x1e, 0x0d, 0xc5, 0x86, 0x78, 0x51, 0xa9, 0x00, 0x00, 0xef, 0x4d, 0xe2, 0x94, 0x60, 0x89,
        0x83, 0x04, 0xb4, 0x0e, 0x90, 0x10, 0x05, 0x1c, 0x7f, 0xd7, 0x33, 0x92, 0x1f, 0xe7, 0x74, 0x59
    };
    uint64_t min_value_1;
    uint64_t max_value_1;
    rustsecp256k1zkp_v0_5_0_pedersen_commitment pc;

    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_commitment_parse(ctx, &pc, commit_1));

    CHECK(rustsecp256k1zkp_v0_5_0_rangeproof_verify(
        ctx,
        &min_value_1, &max_value_1,
        &pc,
        vector_1, sizeof(vector_1),
        NULL, 0,
        rustsecp256k1zkp_v0_5_0_generator_h
    ));
}

void test_pedersen_commitment_fixed_vector(void) {
    const unsigned char two_g[33] = {
        0x09,
        0xc6, 0x04, 0x7f, 0x94, 0x41, 0xed, 0x7d, 0x6d, 0x30, 0x45, 0x40, 0x6e, 0x95, 0xc0, 0x7c, 0xd8,
        0x5c, 0x77, 0x8e, 0x4b, 0x8c, 0xef, 0x3c, 0xa7, 0xab, 0xac, 0x09, 0xb9, 0x5c, 0x70, 0x9e, 0xe5
    };
    unsigned char result[33];
    rustsecp256k1zkp_v0_5_0_pedersen_commitment parse;

    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_commitment_parse(ctx, &parse, two_g));
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_commitment_serialize(ctx, result, &parse));
    CHECK(memcmp(two_g, result, 33) == 0);

    result[0] = 0x08;
    CHECK(rustsecp256k1zkp_v0_5_0_pedersen_commitment_parse(ctx, &parse, result));
    result[0] = 0x0c;
    CHECK(!rustsecp256k1zkp_v0_5_0_pedersen_commitment_parse(ctx, &parse, result));
}

void run_rangeproof_tests(void) {
    int i;
    test_api();
    test_rangeproof_fixed_vectors();
    test_pedersen_commitment_fixed_vector();
    for (i = 0; i < count / 2 + 1; i++) {
        test_pedersen();
    }
    for (i = 0; i < count / 2 + 1; i++) {
        test_borromean();
    }
    test_rangeproof();
    test_multiple_generators();
}

#endif
