use serde::{ser::SerializeTuple, Serialize};
use zvariant_derive::{SerializeDict, TypeDict};

use crate::ResultID;

/// Detailed information of a [`ResultID`].
#[derive(SerializeDict, TypeDict, Debug, Default)]
pub struct ResultMeta {
    id: ResultID,
    name: String,
    description: Option<String>,
    #[zvariant(rename = "clipboardText")]
    clipboard_text: Option<String>,
    icon: Option<zvariant::OwnedValue>,
    gicon: Option<String>,
    #[zvariant(rename = "icon-data")]
    icon_data: Option<IconData>,
}

impl ResultMeta {
    pub fn builder(id: ResultID, name: &str) -> ResultMetaBuilder {
        ResultMetaBuilder::new(id, name)
    }
}

/// A struct wrapping the required information to re-construct an icon with
/// [`gdk-pixbuf`](https://gtk-rs.org/gtk-rs-core/stable/0.14/docs/gdk_pixbuf/index.html).
///
/// You can make use of the `pixbuf` feature that implements `From<gdk_pixbuf::Pixbuf> for IconData`.
#[derive(Debug)]
pub struct IconData {
    pub width: i32,
    pub height: i32,
    pub rowstride: i32,
    pub has_alpha: bool,
    pub bits_per_sample: i32,
    pub n_channels: i32,
    pub data: Vec<u8>,
}

#[cfg(feature = "pixbuf")]
impl From<&gdk_pixbuf::Pixbuf> for IconData {
    fn from(pixbuf: &gdk_pixbuf::Pixbuf) -> Self {
        let data = pixbuf.read_pixel_bytes().unwrap();
        Self {
            width: pixbuf.width(),
            height: pixbuf.height(),
            rowstride: pixbuf.rowstride(),
            has_alpha: pixbuf.has_alpha(),
            bits_per_sample: pixbuf.bits_per_sample(),
            n_channels: pixbuf.n_channels(),
            data: data.to_vec(),
        }
    }
}

impl zvariant::Type for IconData {
    fn signature() -> zvariant::Signature<'static> {
        zvariant::Signature::from_static_str_unchecked("(iiibiiay)")
    }
}

impl Serialize for IconData {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        let mut s = serializer.serialize_tuple(7)?;
        s.serialize_element(&self.width)?;
        s.serialize_element(&self.height)?;
        s.serialize_element(&self.rowstride)?;
        s.serialize_element(&self.has_alpha)?;
        s.serialize_element(&self.bits_per_sample)?;
        s.serialize_element(&self.n_channels)?;
        s.serialize_element(&self.data)?;
        s.end()
    }
}

/// Create an instance of [`ResultMeta`].
pub struct ResultMetaBuilder {
    id: String,
    name: String,
    description: Option<String>,
    clipboard_text: Option<String>,
    gicon: Option<String>,
    icon: Option<zvariant::OwnedValue>,
    icon_data: Option<IconData>,
}

impl ResultMetaBuilder {
    pub fn new(id: ResultID, name: &str) -> Self {
        Self {
            id,
            name: name.to_owned(),
            gicon: None,
            description: None,
            icon: None,
            icon_data: None,
            clipboard_text: None,
        }
    }

    /// Set a short description of the search result.
    pub fn description(mut self, description: &str) -> Self {
        self.description = Some(description.to_owned());
        self
    }

    /// Set a text to be copied to the clipboard when the result is activated.
    pub fn clipboard_text(mut self, clipboard_text: &str) -> Self {
        self.clipboard_text = Some(clipboard_text.to_owned());
        self
    }

    /// Set an icon-name or a URI/path.
    pub fn gicon(mut self, gicon: &str) -> Self {
        self.gicon = Some(gicon.to_owned());
        self
    }

    /// Set an icon serialized with [`gio::Icon::Serialize`](https://gtk-rs.org/gtk-rs-core/stable/0.14/docs/gio/prelude/trait.IconExt.html#tymethod.serialize).
    pub fn icon(mut self, icon: zvariant::OwnedValue) -> Self {
        self.icon = Some(icon);
        self
    }

    /// Set the icon as data. GNOME Shell will re-construct it using GdkPixbuf's API.
    pub fn icon_data(mut self, icon_data: IconData) -> Self {
        self.icon_data = Some(icon_data);
        self
    }

    /// Build an instance of [`ResultMeta`].
    pub fn build(self) -> ResultMeta {
        ResultMeta {
            id: self.id,
            name: self.name,
            gicon: self.gicon,
            description: self.description,
            icon: self.icon,
            icon_data: self.icon_data,
            clipboard_text: self.clipboard_text,
        }
    }
}
