// Copyright 2021 The sdcons Authors. Licensed under Apache-2.0.

// Copyright 2015 The etcd Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

use std::collections::HashMap;
use std::io::Error as IoError;

use log::debug;

use crate::cons::*;
use crate::constant::*;
use crate::error::Error;
use crate::progress::*;
use crate::storage::*;
use crate::types::*;

/// A struct records info generated by `RawNode::advance()` and used in
/// `RawNode::post_advance()`.
pub struct PostReady {
    pub(crate) first_apply_index_id: u64,
    pub(crate) last_apply_index_id: u64,
    pub(crate) chosen_entries: Vec<Entry>,
    pub(crate) msgs: HashMap<u64, Vec<Message>>,
}

/// WriteTask records all info to persistent.
pub struct WriteTask {
    /// Indicates user should persist fields `committed_stats`, `hard_states`,
    /// `member_states`.
    pub should_stable_metas: bool,

    /// Records the committed id of each channel.  Won't persist everytime,
    /// unless the field `should_stable_metas` is setted.
    pub committed_stats: HashMap<u64, u64>,

    /// Records the hard state of each channel. It should be persistted if this
    /// optional field is Some.
    pub hard_states: Option<HashMap<u64, HardState>>,

    /// Records the member states of each channel. It should be persistted if
    /// this optional field is Some.
    pub member_states: Option<HashMap<u64, MemberState>>,

    /// The index and entry logs is to persist.
    pub unstable_indexes: Vec<LogIndex>,
    pub unstable_entries: Vec<Entry>,

    // Records some info need by `RawNode`.  User should invoke `RawNode::post_advance()` when
    // write task is persisted.
    pub post_ready: Option<PostReady>,
}

/// A trait describes a set observer function invoked when a channel role
/// is changed.
///
/// **NOTICE**: only the index channel and the local channel is observed.
pub trait RoleObserver {
    fn start_leading(&self, channel_id: u64, term: u64);
    fn stop_leading(&self, channel_id: u64, term: u64);
    fn start_following(&self, channel_id: u64, term: u64);
    fn stop_following(&self, channel_id: u64, term: u64);
}

/// A template-method trait descript the actions used in `RawNode::advance()`.
pub trait AdvanceTemplate: RoleObserver {
    /// Submit the read replies (request_id => index id) to FSM.
    ///
    /// When a entry has been applied, those request `index_id` is less than the
    /// log is finished.
    fn submit_read_replies(&mut self, finished_reads: &mut HashMap<u64, u64>);

    /// Submit apply task with chosen entries to FSM.
    ///
    /// When applying is done, user should invoke
    /// `RawNode::submit_apply_result()` to nofity sdcons update states.
    ///
    /// **WARNING**: please skip internal entry but record it's index id.  Call
    /// `is_internal_request()` to determine whether a request is proposed by
    /// internal implemention.
    fn submit_apply_task(
        &mut self,
        first_apply_index_id: u64,
        last_apply_index_id: u64,
        entries: &mut Vec<Entry>,
    );

    /// Submit a snapshot to FSM to apply.
    ///
    /// FSM should download this snapshot from remote firstly, then load it into
    /// FSM. After that, user should invokes
    /// `RawNode::finish_snapshot_loading()` to notify sdcons update states.
    fn apply_snapshot(&mut self, snapshot_desc: &SnapshotDesc);

    /// Notify FSM to create a new snapshot.
    ///
    /// When new snapshot is done, user should invokes
    /// `RawNode::finish_checkpoint()` to notify sdcons update states.
    fn checkpoint(&mut self);

    /// Submit BuildMessage task.
    ///
    /// When msg is built and replicated, user should invoke
    /// `RawNode::log_replicated()` to nofity sdcons update states.
    fn build_replicate_msgs(&mut self, builders: &mut Vec<MsgBuilder>);

    /// Dispatch messages.
    fn emit_msgs(&mut self, msgs: &mut Vec<Message>);
}

/// A trait is used by `RawNode` to load logs from persistent device.
pub trait LogReader {
    /// The index id of applied logs which has already persisted in FSM.  Sdcons
    /// uses this value to load and recovery logs into memory storage.  Only
    /// logs which index id is larged than `applied_id` will be transmit to
    /// FSM.
    fn applied_index_id(&self) -> u64;

    /// Return the maximum applied entry id of specified channel from the
    /// persistent device.
    fn channel_applied_entry_id(&self, channel_id: u64) -> u64;

    /// Read the term of specified log of target channel from the persistent
    /// device.
    fn read_meta(&self, channel_id: u64, id: u64) -> Result<u64, IoError>;

    /// Read the entries log between range [start, end) of specified channel
    /// from the persistent device.
    fn read_entries(&self, channel_id: u64, start: u64, end: u64) -> Result<Vec<Entry>, IoError>;

    /// Read the indexes log between range [start, end) from the persistent
    /// device.
    fn read_indexes(&self, start: u64, end: u64) -> Result<Vec<LogIndex>, IoError>;
}

/// A structure that users can use directly, and any sdcons-related operations
/// need to be forwarded through this structure.
#[derive(Debug)]
pub struct RawNode {
    id: u64,
    cons: Sdcons,
    hard_states: HashMap<u64, HardState>,
    member_states: HashMap<u64, MemberState>,
    roles: HashMap<u64, Role>,
}

impl RawNode {
    fn early_apply<T>(&mut self, ready: &mut Ready, template: &mut T)
    where
        T: AdvanceTemplate,
    {
        if !ready.finished_reads.is_empty() {
            template.submit_read_replies(&mut ready.finished_reads);
        }

        if !ready.post_apply && ready.chosen_entries.len() > 0 {
            template.submit_apply_task(
                ready.first_apply_index_id,
                ready.last_apply_index_id,
                &mut ready.chosen_entries,
            );
        }

        // bcast leader msgs before stabling entries.
        for (id, role) in &ready.roles {
            if *role != Role::Leader {
                continue;
            }
            match ready.msgs.remove(id) {
                None => {}
                Some(mut e) => template.emit_msgs(&mut e),
            }
        }
    }

    fn post_apply<T>(&mut self, ready: &mut PostReady, template: &mut T)
    where
        T: AdvanceTemplate,
    {
        if ready.chosen_entries.len() > 0 {
            template.submit_apply_task(
                ready.first_apply_index_id,
                ready.last_apply_index_id,
                &mut ready.chosen_entries,
            );
        }
    }

    fn is_hard_states_changed(&self, ready: &Ready) -> bool {
        if ready.hard_states.len() != self.hard_states.len() {
            false
        } else {
            self.hard_states
                .iter()
                .map(|(i, s)| ready.hard_states.get(i).map(|h| *h != *s))
                .filter(|p| p.unwrap_or(true))
                .count()
                > 0
        }
    }

    fn is_member_states_changed(&self, ready: &Ready) -> bool {
        if ready.member_states.len() != self.member_states.len() {
            false
        } else {
            self.member_states
                .iter()
                .map(|(i, s)| ready.member_states.get(i).map(|h| *h != *s))
                .filter(|p| p.unwrap_or(true))
                .count()
                > 0
        }
    }

    fn build_stable_task(&mut self, ready: &mut Ready) -> Option<WriteTask> {
        let should_update_hard_states = self.is_hard_states_changed(ready);
        let should_update_member_states = self.is_member_states_changed(ready);
        if !should_update_hard_states
            && !should_update_member_states
            && ready.unstable_entries.is_empty()
            && ready.unstable_indexes.is_empty()
        {
            return None;
        }

        let mut write_task = WriteTask {
            should_stable_metas: ready.should_stable_metas,
            committed_stats: HashMap::new(),
            hard_states: None,
            member_states: None,
            unstable_entries: Vec::new(),
            unstable_indexes: Vec::new(),
            post_ready: self.build_post_ready(ready),
        };
        if should_update_hard_states {
            self.hard_states = ready.hard_states.clone();
            write_task.hard_states = Some(std::mem::take(&mut ready.hard_states));
        }
        if should_update_member_states {
            self.member_states = ready.member_states.clone();
            write_task.member_states = Some(std::mem::take(&mut ready.member_states));
        }
        std::mem::swap(&mut write_task.committed_stats, &mut ready.committed_stats);
        std::mem::swap(&mut write_task.unstable_entries, &mut ready.unstable_entries);
        std::mem::swap(&mut write_task.unstable_indexes, &mut ready.unstable_indexes);
        Some(write_task)
    }

    fn build_post_ready(&mut self, ready: &mut Ready) -> Option<PostReady> {
        if ready.msgs.is_empty() && !ready.post_apply {
            None
        } else {
            let mut post_ready = PostReady {
                first_apply_index_id: ready.first_apply_index_id,
                last_apply_index_id: ready.last_apply_index_id,
                chosen_entries: Vec::new(),
                msgs: HashMap::new(),
            };
            if ready.post_apply {
                std::mem::swap(&mut post_ready.chosen_entries, &mut ready.chosen_entries);
            }
            std::mem::swap(&mut ready.msgs, &mut post_ready.msgs);
            Some(post_ready)
        }
    }

    fn construct_channel_metas<R: LogReader>(
        id: u64,
        membership: &HashMap<u64, MemberState>,
        reader: &R,
    ) -> Result<HashMap<u64, EntryMeta>, IoError> {
        let mut channel_metas = HashMap::new();
        for (channel_id, _) in membership {
            let channel_applied_entry_id = reader.channel_applied_entry_id(*channel_id);
            let entry_term = reader.read_meta(*channel_id, channel_applied_entry_id)?;
            debug!(
                "node {} channel {} already apply to {}, entry term {}",
                id, channel_id, channel_applied_entry_id, entry_term
            );
            channel_metas.insert(
                *channel_id,
                EntryMeta {
                    id: channel_applied_entry_id,
                    term: entry_term,
                },
            );
        }
        let index_applied_id = reader.applied_index_id();
        let index_term = reader.read_meta(INDEX_CHANNEL_ID, index_applied_id)?;
        debug!(
            "node {} channel {} already apply to {}, index term {}",
            id, INDEX_CHANNEL_ID, index_applied_id, index_term
        );
        channel_metas.insert(
            INDEX_CHANNEL_ID,
            EntryMeta {
                id: index_applied_id,
                term: index_term,
            },
        );
        Ok(channel_metas)
    }

    fn load_channel_entries<L: LogMetaView, R: LogReader>(
        id: u64,
        mem_store: &mut MemStorage,
        log_view: &L,
        reader: &R,
    ) -> Result<(), IoError> {
        for (channel_id, _desc) in &log_view.membership() {
            let applied_entry_id = reader.channel_applied_entry_id(*channel_id);
            let (start, end) = log_view.range_of(*channel_id);
            if end <= start {
                debug!("node {} channel {} entries is empty", id, *channel_id);
                continue;
            }
            if end <= applied_entry_id || applied_entry_id + 1u64 < start {
                // Out of range
                return Err(IoError::new(std::io::ErrorKind::UnexpectedEof, "out of range"));
            }
            // TODO(patrick) read entries by slice.
            let entries = reader.read_entries(*channel_id, applied_entry_id + 1u64, end)?;
            mem_store.append_entries(*channel_id, entries);
        }

        let applied_index_id = reader.applied_index_id();
        let (start, end) = log_view.range_of(INDEX_CHANNEL_ID);
        if end <= start {
            debug!("node {} channel {} indexes is empty", id, INDEX_CHANNEL_ID);
            return Ok(());
        }
        if end <= applied_index_id || applied_index_id + 1u64 < start {
            // Out of range
            return Err(IoError::new(std::io::ErrorKind::UnexpectedEof, "out of range"));
        }
        let indexes = reader.read_indexes(applied_index_id + 1u64, end)?;
        mem_store.extend_indexes(indexes);
        Ok(())
    }

    fn transform_roles<T>(&mut self, ready: &mut Ready, template: &mut T)
    where
        T: AdvanceTemplate,
    {
        // Only notify index channel or local channel.
        for (id, prev_role) in &self.roles {
            if *id != INDEX_CHANNEL_ID && *id != self.id {
                continue;
            }
            let cur_role = ready.roles.get(id).unwrap_or(&Role::Follower);
            if *cur_role == *prev_role {
                continue;
            }

            let term = ready
                .hard_states
                .get(id)
                .map(|s| s.current_term)
                .unwrap_or(INITIAL_TERM);
            match prev_role {
                Role::Follower => template.stop_following(*id, term),
                Role::Leader => template.stop_leading(*id, term),
                _ => {}
            }
            match cur_role {
                Role::Follower => template.start_following(*id, term),
                Role::Leader => template.stop_leading(*id, term),
                _ => {}
            }
        }
    }
}

impl RawNode {
    /// Advance the timer of the underlying consensus algorithm.
    ///
    /// **IMPORTANT*: the user needs to ensure that the clock step is stable.
    pub fn tick(&mut self) {
        self.cons.tick()
    }

    /// Call this function to step to new state when received new message.
    pub fn step(&mut self, msg: Message) {
        self.cons.step(msg)
    }

    /// Submit a leased read request, and get an request id.
    ///
    /// When this request is validated by sdcons,
    /// `AdvanceTemplate::submit_read_replies()` will be invoked to return the
    /// index id which this request should wait on.
    ///
    /// REQUIRES:
    /// - local leader
    /// - transfer leader
    pub fn leased_read(&mut self) -> Result<u64, Error> {
        self.cons.leased_read()
    }

    /// Submit tasks to replicate to other nodes.
    ///
    /// REQUIRES:
    /// - local leader
    /// - transfer leader
    pub fn submit_task(&mut self, request_id: u64, task: Task) -> Result<u64, Error> {
        self.cons.submit_task(request_id, task)
    }

    /// Submit a membership changing request.
    ///
    /// REQUIRED:
    /// - index leader
    /// - local leader
    /// - no leadership transfering
    /// - no pending configs
    pub fn change_config(&mut self, members: Vec<u64>) -> Result<u64, Error> {
        self.cons.change_config(members)
    }

    /// Submit a control command to sdcons.
    pub fn control(&mut self, c: Control) -> Result<(), Error> {
        self.cons.control(c)
    }

    /// See `AdvanceTemplate::submit_apply_task()` for details.
    pub fn submit_apply_result(&mut self, from: u64, to: u64) -> bool {
        self.cons.submit_apply_result(from, to)
    }

    /// See `AdvanceTemplate::submit_apply_task()` for details.
    pub fn submit_stable_result(&mut self, channel_id: u64, from: u64, to: u64) -> bool {
        self.cons.submit_stable_result(channel_id, from, to)
    }

    /// See `AdvanceTemplate::build_replicate_msgs()` for details.
    pub fn log_replicated(&mut self, node_id: u64, channel_id: u64, next_id: u64) {
        self.cons.log_replicated(node_id, channel_id, next_id)
    }

    /// See `AdvanceTemplate::apply_snapshot()` for details.
    pub fn finish_snapshot_loading(
        &mut self,
        received: bool,
        hard_states: &HashMap<u64, HardState>,
        desc: &SnapshotDesc,
    ) {
        self.cons
            .finish_snapshot_loading(received, hard_states, desc)
    }

    /// See `AdvanceTemplate::checkpoint()` for details.
    pub fn finish_checkpoint(&mut self) {
        self.cons.finish_checkpoint();
    }

    pub fn is_sending_snapshot(&self) -> bool {
        self.cons.is_sending_snapshot()
    }

    /// Return the leader id of the index channel.
    pub fn leader_id(&self) -> Option<u64> {
        self.cons.leader_id()
    }
}

impl RawNode {
    /// Build an `RawNode` from specified metas and logs.
    ///
    /// A valid id of node should in range (0, std::u64::MAX), see
    /// sdcons/constant.rs for details.
    pub fn build<L: LogMetaView, R: LogReader>(
        id: u64,
        opts: SdconsOption,
        log_view: &L,
        reader: &R,
    ) -> Result<Self, Error> {
        if id < 1 || id == std::u64::MAX {
            return Err(Error::InvalidNodeId(id));
        }

        // read entries from log storage and insert to memstorage.
        let membership = log_view.membership();
        debug!("node {} current membership {:?}", id, membership);
        let channel_metas = Self::construct_channel_metas(id, &membership, reader)?;
        let mut mem_store = MemStorage::recovery(id, &channel_metas);
        Self::load_channel_entries(id, &mut mem_store, log_view, reader)?;
        let hard_states = log_view.hard_states();
        let roles = hard_states
            .iter()
            .map(|(i, _)| (*i, Role::Follower))
            .collect::<HashMap<_, _>>();

        mem_store.mark_all_stabled();
        Ok(RawNode {
            id: id,
            cons: Sdcons::new(
                id,
                reader.applied_index_id(),
                opts,
                mem_store,
                &membership,
                &hard_states,
            ),
            hard_states,
            member_states: membership,
            roles,
        })
    }

    /// Advances the underlying consensus state machine and returns the write
    /// task. Caller should invokes `RawNode::post_advance` to submit the
    /// write task result if the write task returned isn't none.
    pub fn advance<L, T>(&mut self, log_meta_view: &L, template: &mut T) -> Option<WriteTask>
    where
        L: LogMetaView,
        T: AdvanceTemplate,
    {
        let mut ready = self.cons.advance(log_meta_view);

        self.transform_roles(&mut ready, template);
        self.early_apply(&mut ready, template);
        if ready.should_checkpoint {
            template.checkpoint();
        }
        if let Some(desc) = &ready.pending_snapshot {
            template.apply_snapshot(desc);
        }
        if let Some(builders) = &mut ready.cop_msgs {
            template.build_replicate_msgs(builders);
        }

        let write_task = self.build_stable_task(&mut ready);
        if write_task.is_none() {
            // Manually invoke post advance if not any stable tasks exists.
            match self.build_post_ready(&mut ready) {
                Some(post_ready) => self.post_advance(post_ready, template),
                None => {}
            }
        }
        write_task
    }

    /// Submit the write task execution result.
    pub fn post_advance<T>(&mut self, mut post_ready: PostReady, template: &mut T)
    where
        T: AdvanceTemplate,
    {
        self.post_apply(&mut post_ready, template);
        for (_remote_id, msgs) in &mut post_ready.msgs {
            if msgs.len() > 0 {
                template.emit_msgs(msgs);
            }
        }
    }
}
