use super::{Point, Vector};
use ttf_parser::OutlineBuilder;

pub(crate) struct OutlineScaler<'b, T: ?Sized> {
    inner: &'b mut T,
    scale: Vector<f32>,
}

impl<'b, T: ?Sized> OutlineScaler<'b, T> {
    pub(crate) fn new(inner: &'b mut T, scale: Vector<f32>) -> Self {
        Self { inner, scale }
    }
}

impl<T: OutlineBuilder + ?Sized> OutlineBuilder for OutlineScaler<'_, T> {
    fn move_to(&mut self, x: f32, y: f32) {
        self.inner.move_to(x * self.scale.x, y * self.scale.y)
    }

    fn line_to(&mut self, x1: f32, y1: f32) {
        self.inner.line_to(x1 * self.scale.x, y1 * self.scale.y)
    }

    fn quad_to(&mut self, x1: f32, y1: f32, x2: f32, y2: f32) {
        self.inner.quad_to(
            x1 * self.scale.x,
            y1 * self.scale.y,
            x2 * self.scale.x,
            y2 * self.scale.y,
        )
    }

    fn curve_to(&mut self, x1: f32, y1: f32, x2: f32, y2: f32, x3: f32, y3: f32) {
        self.inner.curve_to(
            x1 * self.scale.x,
            y1 * self.scale.y,
            x2 * self.scale.x,
            y2 * self.scale.y,
            x3 * self.scale.x,
            y3 * self.scale.y,
        )
    }

    fn close(&mut self) {
        self.inner.close()
    }
}

pub(crate) struct OutlineTranslator<'b, T: ?Sized> {
    inner: &'b mut T,
    translation: Point<f32>,
}

impl<'b, T: ?Sized> OutlineTranslator<'b, T> {
    pub(crate) fn new(inner: &'b mut T, translation: Point<f32>) -> Self {
        Self { inner, translation }
    }
}

impl<T: OutlineBuilder + ?Sized> OutlineBuilder for OutlineTranslator<'_, T> {
    fn move_to(&mut self, x: f32, y: f32) {
        self.inner
            .move_to(x + self.translation.x, y + self.translation.y)
    }

    fn line_to(&mut self, x1: f32, y1: f32) {
        self.inner
            .line_to(x1 + self.translation.x, y1 + self.translation.y)
    }

    fn quad_to(&mut self, x1: f32, y1: f32, x2: f32, y2: f32) {
        self.inner.quad_to(
            x1 + self.translation.x,
            y1 + self.translation.y,
            x2 + self.translation.x,
            y2 + self.translation.y,
        )
    }

    fn curve_to(&mut self, x1: f32, y1: f32, x2: f32, y2: f32, x3: f32, y3: f32) {
        self.inner.curve_to(
            x1 + self.translation.x,
            y1 + self.translation.y,
            x2 + self.translation.x,
            y2 + self.translation.y,
            x3 + self.translation.x,
            y3 + self.translation.y,
        )
    }

    fn close(&mut self) {
        self.inner.close()
    }
}
