extern crate core_graphics;

use super::ffi;
use core_graphics::display::*;
use std::mem;

#[derive(Debug)]
pub struct Display {
  id: u32,
}

impl Display {
  fn new(id: u32) -> Self {
    Display { id }
  }

  pub fn id(&self) -> u32 {
    self.id
  }

  pub fn x(&self) -> f64 {
    unsafe { CGDisplayBounds(self.id).origin.x }
  }

  pub fn y(&self) -> f64 {
    unsafe { CGDisplayBounds(self.id).origin.y }
  }

  pub fn width(&self) -> f64 {
    unsafe { CGDisplayBounds(self.id).size.width }
  }

  pub fn height(&self) -> f64 {
    unsafe { CGDisplayBounds(self.id).size.height }
  }

  pub fn scale(&self) -> f64 {
    unsafe {
      let display_mode = CGDisplayCopyDisplayMode(self.id);
      let pixel_width = CGDisplayModeGetPixelWidth(display_mode);

      CGDisplayModeRelease(display_mode);

      (pixel_width as f64) / self.width()
    }
  }

  pub fn rotation(&self) -> f64 {
    unsafe { CGDisplayRotation(self.id) }
  }

  pub fn displays<'s>() -> Result<Vec<Display>, &'s str> {
    let max_displays: u32 = 16;
    let mut active_displays = unsafe { mem::MaybeUninit::<[u32; 16]>::uninit().assume_init() };
    let mut display_count: u32 = 0;

    unsafe {
      // 不包含镜像显示器
      if CGGetActiveDisplayList(
        max_displays,
        active_displays.as_mut_ptr(),
        &mut display_count,
      ) != 0
      {
        return Err("CGGetActiveDisplayList ffi call Error");
      }
    };

    let mut displays = Vec::with_capacity(16);

    for i in 0..display_count as usize {
      let display_id = *active_displays.get(i).unwrap();
      displays.push(Display::new(display_id));
    }

    Ok(displays)
  }

  pub fn get_display_with_point<'s>(x: f64, y: f64) -> Result<Display, &'s str> {
    let point = CGPoint { x, y };
    let max_displays: u32 = 16;
    let mut displays = unsafe { mem::MaybeUninit::<[u32; 16]>::uninit().assume_init() };
    let mut display_count: u32 = 0;

    unsafe {
      if ffi::CGGetDisplaysWithPoint(
        point,
        max_displays,
        displays.as_mut_ptr(),
        &mut display_count,
      ) != 0
      {
        return Err("CGGetDisplaysWithPoint ffi call Error");
      }
    }

    let mut dsps = Vec::with_capacity(16);

    for i in 0..display_count as usize {
      let display_id = *displays.get(i).unwrap();
      dsps.push(Display::new(display_id));
    }

    if display_count == 0 {
      return Err("CGGetDisplaysWithPoint get 0 display");
    }

    let display_id = *displays.get(0).unwrap();
    Ok(Display::new(display_id))
  }
}

#[cfg(test)]
mod tests {
  use super::Display;
  #[test]
  fn displays() {
    let result = Display::displays().unwrap();
    assert_eq!(result.len(), 1);
  }

  #[test]
  fn get_display_with_point() {
    let result = Display::get_display_with_point(10.0, 10.0).unwrap();
    assert_eq!(result.width(), 1680.0);
  }
}
