// SPDX-License-Identifier: GPL-3.0-only

use core::fmt;

#[derive(Clone)]
#[repr(C)]
pub struct CppVector<T> {
    begin: *const T,
    end: *const T,
    capacity: *const T,
}

pub struct CppVectorIter<'a, T> {
    vec: &'a CppVector<T>,
    ptr: *const T,
}

impl<'a, T> Iterator for CppVectorIter<'a, T> {
    type Item = &'a T;

    fn next(&mut self) -> Option<Self::Item> {
        if self.ptr == self.vec.end {
            return None;
        }
        unsafe {
            let ret = Some(&*self.ptr);
            self.ptr = self.ptr.add(1);
            ret
        }
    }
}

impl<T: fmt::Debug> fmt::Debug for CppVector<T> {
    fn fmt(&self, fmt: &mut fmt::Formatter) -> fmt::Result {
        write!(fmt, "std::vector(")?;
        fmt.debug_list().entries(self.iter()).finish()?;
        write!(fmt, ")")
    }
}

impl<T> CppVector<T> {
    pub fn iter(&self) -> CppVectorIter<T> {
        CppVectorIter {
            vec: self,
            ptr: self.begin,
        }
    }
}
