/*-
* Scram-rs
* Copyright (C) 2021  Aleksandr Morozov
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/


use super::scram_error::{ScramResult, ScramRuntimeError, ScramErrorCode};
use super::{scram_error};
use super::scram_common::ScramCommon;

pub use super::scram_sync;
pub use super::scram_async;


/// A returned datatype from both Server/Client parser.  
/// If [ScramParse::Output] then a response was prepared.  
/// If [ScramParse::Completed] then no other action is required.
pub enum ScramParse
{
    /// Data is prepared to be sent to Server or Client
    Output(String),

    /// Final stage, no more parsing is required
    Completed,
}

impl ScramParse
{
    /// Is current instance is Output(...)
    pub 
    fn is_output(&self) -> bool
    {
        match *self
        {
            ScramParse::Output(_) => return true,
            ScramParse::Completed => return false
        }
    }

    /// Is current instance is Completed
    pub 
    fn is_final(&self) -> bool
    {
        match *self
        {
            ScramParse::Output(_) => return false,
            ScramParse::Completed => return true
        }
    }

    /// Extracts data from Output(...). Will throw an error if
    /// current state is Completed.
    pub 
    fn extract_output(self) -> ScramResult<String>
    {
        match self
        {
            ScramParse::Output(r) => 
                return Ok(r),
            ScramParse::Completed => 
                scram_error!(ScramErrorCode::InternalError, "completed, nothing to extract"),
        }
    }
}

/// A SCRAM nonce initialization and customization.
/// Use implemented functions, don't use enum fields directly.
pub enum ScramNonce<'sn>
{
    /// Nonce is not provided by user, autogenerate
    None,

    /// A nonce is encoded as plain text 
    Plain(&'sn [u8]),

    /// A nonce is encoded as base64
    Base64(&'sn str),
}

impl<'sn> ScramNonce<'sn>
{
    /// Initialize ScramNonce so the data will be autogenerated
    pub 
    fn none() -> Self
    {
        return Self::None;
    }

    /// Initialize ScramNonce with plain data.
    pub 
    fn plain(p: &'sn [u8]) -> ScramNonce<'sn>
    {
        return Self::Plain(p);
    }

    /// Initialize ScramNonce with base64 encoded nonce. 
    pub 
    fn base64(b: &'sn str) -> ScramNonce<'sn>
    {
        return Self::Base64(b);
    }

    /// Extract Nonce
    /// Will throw error if base64 will fail to encode the provided data to base64.
    /// Will throw error if argument length is 0.
    pub 
    fn get_nonce(self) -> ScramResult<String>
    {
        match self
        {
            ScramNonce::None => 
            {
                return Ok(
                    base64::encode(
                        ScramCommon::sc_random(ScramCommon::SCRAM_RAW_NONCE_LEN)?
                    )
                );
            },
            ScramNonce::Plain(p) => 
            {
                if p.len() > ScramCommon::SCRAM_RAW_NONCE_LEN
                {
                    scram_error!(
                        ScramErrorCode::InternalError,
                        "nonce length is > {}, actual: '{}'", 
                        ScramCommon::SCRAM_RAW_NONCE_LEN, p.len()
                    );
                }

                return Ok(base64::encode(p));
            },
            ScramNonce::Base64(b) => 
            {
                if b.len() == 0
                {
                    scram_error!(ScramErrorCode::InternalError, "base64 nonce length is 0");
                }
                
                return Ok(b.to_string())
            }
        };
    }
}


