/// Stack data structure
///
pub struct Stack<T> {
    data: Vec<T>,
}

impl<T> Stack<T> {
    /// Creates an empty stack
    /// # Examples
    /// ```
    /// use scorpius::data_structure::stack::Stack;
    /// let stack: Stack<i32> = Stack::new();
    /// ```
    pub fn new() -> Self {
        Stack {
            data: Vec::new(),
        }
    }

    /// Push an item onto the stack
    /// The function has a time complexity of O(1)
    ///
    /// # Arguments
    /// * `item` - The value to push
    /// # Example
    /// ```
    /// use scorpius::data_structure::stack::Stack;
    /// let mut stack = Stack::new();
    /// stack.push(1);
    /// assert_eq!(stack.peek(), Some(&1));
    /// ```
    pub fn push(&mut self, item: T) {
        self.data.push(item);
    }

    /// Pop an item off the stack
    /// The function has a time complexity of O(1)
    ///
    /// # Returns
    /// * `Option<T>` - The value popped off the stack
    /// # Example
    /// ```
    /// use scorpius::data_structure::stack::Stack;
    /// let mut stack = Stack::new();
    /// stack.push(1);
    /// assert_eq!(stack.pop(), Some(1));
    /// assert_eq!(stack.pop(), None);
    /// ```
    pub fn pop(&mut self) -> Option<T> {
        self.data.pop()
    }

    /// Peek at the top of the stack
    /// The function has a time complexity of O(1)
    ///
    /// # Returns
    /// * `Option<&T>` - The value at the top of the stack
    /// # Example
    /// ```
    /// use scorpius::data_structure::stack::Stack;
    /// let mut stack = Stack::new();
    /// stack.push(1);
    /// assert_eq!(stack.peek(), Some(&1));
    /// ```
    pub fn peek(&self) -> Option<&T> {
        self.data.last()
    }
}

/// Provide default implementation
/// # Examples
/// ```
/// use scorpius::data_structure::stack::Stack;
/// let stack: Stack<i32> = Stack::default();
/// assert_eq!(stack.peek(), None);
/// ```
impl<T> Default for Stack<T> {
    fn default() -> Self {
        Self::new()
    }
}

/// A stack that can be iterated over
/// Iteration is in LIFO order and has a time complexity of O(n)
///
/// # Example (iterating over a stack using for loop)
/// ```
/// use scorpius::data_structure::stack::Stack;
/// let mut stack = Stack::new();
/// stack.push(1);
/// stack.push(2);
/// stack.push(3);
/// let mut i = 3;
/// for item in stack {
///    println!("{}", item);
///    assert_eq!(item, i);
///    i -= 1;
/// }
/// ```
///
/// # Example (iterating over a stack using into_iter())
/// ```
/// use scorpius::data_structure::stack::Stack;
/// let mut stack = Stack::new();
/// stack.push(1);
/// stack.push(2);
/// stack.push(3);
/// let mut i = 3;
/// for item in stack.into_iter() {
///   println!("{}", item);
///   assert_eq!(item, i);
///   i -= 1;
/// }
/// ```
impl <T> Iterator for Stack<T> {
    type Item = T;

    fn next(&mut self) -> Option<Self::Item> {
        self.pop()
    }
}

// tests
#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn create_stack() {
        let stack: Stack<i32> = Stack::new();
        assert_eq!(stack.data.len(), 0);
    }

    #[test]
    fn push_to_stack() {
        let mut stack = Stack::new();
        stack.push(1);
        assert_eq!(stack.data.len(), 1);
    }

    #[test]
    fn pop_from_stack() {
        let mut stack = Stack::new();
        stack.push(1);
        stack.push(2);
        assert_eq!(stack.data.len(), 2);
        assert_eq!(stack.pop(), Some(2));
        assert_eq!(stack.pop(), Some(1));
        assert_eq!(stack.pop(), None);
        assert_eq!(stack.pop(), None);
    }

    #[test]
    fn peek_from_stack() {
        let mut stack = Stack::new();
        stack.push(1);
        stack.push(2);
        assert_eq!(stack.data.len(), 2);
        assert_eq!(stack.peek(), Some(&2));
    }

    #[test]
    fn iterate_stack() {
        let mut stack = Stack::new();
        stack.push(1);
        stack.push(2);
        stack.push(3);
        let mut iter = stack.into_iter();
        assert_eq!(iter.next(), Some(3));
        assert_eq!(iter.next(), Some(2));
        assert_eq!(iter.next(), Some(1));
        assert_eq!(iter.next(), None);
        assert_eq!(iter.next(), None);
    }
}