use {
    super::{Entry, OrderedMap},
    std::{
        collections::hash_map::DefaultHasher,
        hash::{Hash, Hasher},
    },
};

pub struct HashMap<K, V> {
    capacity: usize,
    data: Vec<OrderedMap<K, V>>,
}

impl<K: Clone + Hash + Ord, V: Clone> HashMap<K, V> {
    pub fn new() -> Self {
        Self {
            capacity: 17,
            data: (0..17).map(|_| OrderedMap::new()).collect(),
        }
    }

    pub fn get(&self, key: K) -> Option<V> {
        Some(self.data[self.hash(key.clone())].get(key)?)
    }

    pub fn insert(&mut self, key: K, value: V) -> Option<V> {
        let i = self.hash(key.clone());
        self.data[i].insert(key, value)
    }

    pub fn entries(&self) -> Vec<Entry<K, V>> {
        self.data.iter().map(|o| o.entries()).flatten().collect()
    }

    fn hash(&self, key: K) -> usize {
        let mut hasher = DefaultHasher::new();
        key.hash(&mut hasher);
        hasher.finish() as usize % self.capacity
    }
}

impl<K: Clone + Hash + Ord, V: Clone> From<Vec<(K, V)>> for HashMap<K, V> {
    fn from(kv: Vec<(K, V)>) -> Self {
        let mut map = Self::new();
        for (key, value) in kv.into_iter() {
            map.insert(key, value);
        }
        map
    }
}
