use crate::Directive;
use crate::Scfg;
use std::fmt;
use std::io;

#[derive(Debug)]
enum ErrorKind {
    UnexpectedClosingBrace,
    Io(io::Error),
    ShellWords(shell_words::ParseError),
}

#[derive(Debug)]
pub struct Error {
    kind: ErrorKind,
    lineno: usize,
}

impl fmt::Display for Error {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "parsing error at line {}: ", self.lineno)?;
        match &self.kind {
            ErrorKind::UnexpectedClosingBrace => write!(f, "unexpected '}}'"),
            ErrorKind::Io(err) => write!(f, "io: {}", err),
            ErrorKind::ShellWords(err) => write!(f, "{}", err),
        }
    }
}

impl std::error::Error for Error {
    fn source(&self) -> Option<&(dyn std::error::Error + 'static)> {
        match &self.kind {
            ErrorKind::Io(err) => Some(err),
            ErrorKind::ShellWords(err) => Some(err),
            _ => None,
        }
    }
}

pub fn document(mut r: impl io::BufRead) -> Result<Scfg, Error> {
    let mut lineno = 1;
    let (block, closing_brace) = read_block(&mut r, &mut lineno)?;
    if closing_brace {
        return Err(Error {
            kind: ErrorKind::UnexpectedClosingBrace,
            lineno,
        });
    }
    Ok(block)
}

/// Reads a block.
///
/// Returns `(block, closing_brace)` where `closing_brace` is true if parsing stopped on '}', and
/// false if parsing stopped on EOF.
///
/// `lineno` must be set the line number of the first line of the block, and is set to the line
/// number of the closing bracket or EOF.
fn read_block<R: io::BufRead>(r: &mut R, lineno: &mut usize) -> Result<(Scfg, bool), Error> {
    let mut block = Scfg::new();
    let mut line = String::new();

    loop {
        line.clear();
        let n = r.read_line(&mut line).map_err(|err| Error {
            kind: ErrorKind::Io(err),
            lineno: *lineno,
        })?;
        if n == 0 {
            // reached EOF.
            return Ok((block, false));
        }
        let line = line.trim();

        let mut words = shell_words::split(&line).map_err(|err| Error {
            kind: ErrorKind::ShellWords(err),
            lineno: *lineno,
        })?;
        if words.is_empty() {
            // line is either empty or a comment.
            continue;
        }

        let last_byte = *line.as_bytes().last().unwrap();
        if words.len() == 1 && last_byte == b'}' {
            // The line is a litteral '}' (end of block).
            return Ok((block, true));
        }

        let has_child = words.last().unwrap() == "{" && last_byte == b'{'; // avoid matching `"{"`
        let (name, directive) = if has_child {
            words.pop(); // remove brace
            let name = if words.is_empty() {
                String::new()
            } else {
                words.remove(0)
            };
            *lineno += 1;
            let (child, closing_brace) = read_block(r, lineno)?;
            if !closing_brace {
                return Err(Error {
                    kind: ErrorKind::Io(io::ErrorKind::UnexpectedEof.into()),
                    lineno: *lineno,
                });
            }
            (
                name,
                Directive {
                    params: words,
                    child: Some(child),
                },
            )
        } else {
            let name = words.remove(0);
            (
                name,
                Directive {
                    params: words,
                    child: None,
                },
            )
        };
        block.add_directive(name, directive);

        *lineno += 1;
    }
}
