/*[License]
    - Author: Molese <molese@engineer.com> 2022

    Licensed under either of

    * Apache License, Version 2.0 (LICENSE-APACHE or https://www.apache.org/licenses/LICENSE-2.0)
    * MIT license (LICENSE-MIT or https://opensource.org/licenses/MIT)

    at your option.
*/

#![allow(missing_debug_implementations)]
#![deny(
    missing_copy_implementations,
    trivial_casts,
    trivial_numeric_casts,
    unstable_features,
    unused_import_braces,
    unused_qualifications
)]
#![deny(clippy::all)]

//! Scatterbrained implementation of minigrep.

use std::{env, error::Error, fs::File, io::Read};

/// The necessary configurations for initializing scatterbrainedsearch.
///
pub struct Config {
    /// The file along with the path in which the pattern search will be conducted.
    pub file: String,
    /// The pattern to search for in the contents of the file.
    pub pattern: String,
}

impl Config {
    /// Initializes a new Config.
    ///
    /// Returns error if incorrect option is passed.
    pub fn new(mut args: env::Args) -> Result<Config, &'static str> {
        if args.len() > 3 {
            return Err("Expected 2 arguments");
        }

        args.next();

        let (file, pattern): (String, String) = (
            match args.next() {
                Some(arg) => arg,
                None => return Err("No file provided"),
            },
            match args.next() {
                Some(arg) => arg,
                None => return Err("No pattern provided"),
            },
        );

        Ok(Config { file, pattern })
    }
}

/// Returns the line(s) where the pattern is matched in the contents of the file.
pub fn search<'a>(pattern: &'a str, content: &'a str) -> Vec<String> {
    let mut result: Vec<String> = Vec::new();

    for (lineno, linecon) in content.lines().enumerate() {
        if linecon.contains(pattern) {
            result.push(format!("{}: {}", lineno + 1, linecon));
        }
    }

    result
}

/// Runs the library logic.
/// Expects a reference to [`Config`](struct@Config).
///
/// # Errors
///
/// Returns error on the following situations:
///
/// - `file` doesn't exist.
/// - `file` could not be read (unauthorized read permission).
pub fn run(config: Config) -> Result<(), Box<dyn Error>> {
    let (pattern, content): (&String, &String) = (&config.pattern, &config.file);
    let (mut file, mut data): (File, String) = (File::open(content)?, String::new());

    file.read_to_string(&mut data)?;

    let result: Vec<String> = search(pattern, &data);

    if result.is_empty() {
        return Err((format!("No match found for {} in {}", pattern, content)).into());
    }

    let (mut line_count, mut total_occurrence): (u128, u128) = (0u128, 0u128);

    let matched_result: String = result
        .iter()
        .map(|line: &String| {
            let occurrence: u128 = line.matches(pattern).count().try_into().unwrap();
            total_occurrence += occurrence;
            line_count += 1;
            format!("{} ~{}\n", line, occurrence)
        })
        .collect();

    let input_summary: String = format!("Lines matching {} in {}:", pattern, content);

    let output_summary: String = format!(
        "Matched: {} occurrence{} in {} line{}.",
        total_occurrence,
        if total_occurrence > 1 { "s" } else { "" },
        line_count,
        if line_count > 1 { "s" } else { "" }
    );

    println!(
        "{}\n\n{}\n{}",
        input_summary, matched_result, output_summary
    );

    Ok(())
}

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn case_sensitive_search() {
        let pattern: &str = "OMG";
        let content: &str = "\n\n\n\nOMG, Rust is so cool!";

        assert_eq!(vec!["5: OMG, Rust is so cool!"], search(pattern, content));
    }

    #[test]
    fn case_insensitive_search() {
        let pattern: &str = "test";
        let content: &str = "Hello, World!\nThis is a test string.\nThis is for testing purposes.";

        assert_eq!(
            vec![
                "2: This is a test string.",
                "3: This is for testing purposes."
            ],
            search(pattern, content)
        );
    }
}
