/*[License]
    - Author: Molese <molese@protonmail.com> 2022

    Licensed under either of

    * Apache License, Version 2.0 (LICENSE-APACHE or https://www.apache.org/licenses/LICENSE-2.0)
    * MIT license (LICENSE-MIT or https://opensource.org/licenses/MIT)

    at your option.
*/

#![allow(missing_debug_implementations)]
#![deny(
    missing_copy_implementations,
    trivial_casts,
    trivial_numeric_casts,
    unstable_features,
    unused_import_braces,
    unused_qualifications
)]
#![deny(clippy::all)]

//! Scatterbrained implementation of minigrep.

use std::{env, error::Error, fs::File, io::Read};

/// The necessary configurations for initializing scatterbrainedsearch.
///
pub struct Config {
    /// The file along with the path in which the pattern search will be conducted.
    pub file: String,
    /// The pattern to search for in the contents of the file.
    pub pattern: String,
}

impl Config {
    /// Initializes a new Config.
    ///
    /// Returns error if incorrect option is passed.
    pub fn new(mut args: env::Args) -> Result<Config, &'static str> {
        if args.len() > 3 {
            return Err("Expected 2 arguments");
        }

        /* Discard the first argument (args[0]), which is the filename along with path */
        args.next();

        /* arg0[1] & arg0[2] are file & pattern respectively. */
        let (file, pattern): (String, String) = (
            match args.next() {
                Some(arg) => arg,
                None => return Err("No file provided"),
            },
            match args.next() {
                Some(arg) => arg,
                None => return Err("No pattern provided"),
            },
        );

        /* var names are same as in the struct, so e don't need explicit key:val initialization style. */
        Ok(Config { file, pattern })
    }
}

/* The lifetime parameters says that the search results will be valid as long as the contents are valid.
 * Can last even after pattern goes out of scope. */
/// Returns the line(s) where the pattern is matched in the contents of the file.
pub fn search<'a>(pattern: &'a str, content: &'a str) -> Vec<String> {
    let mut result: Vec<String> = Vec::new();

    // TODO: Standardize lineno
    for (lineno, linecon) in content.lines().enumerate() {
        if linecon.contains(pattern) {
            result.push(format!("{}: {}", lineno + 1, linecon));
        }
    }

    result
}

/// Returns summary which includes total matched occurence(s) of the pattern & line respectively.
pub fn summarize(occurrence: u128, line_count: u128) -> String {
    format!(
        "Matched: {} occurrence{} in {} line{}.",
        occurrence,
        if occurrence > 1 { "s" } else { "" },
        line_count,
        if line_count > 1 { "s" } else { "" }
    )
}

/// Runs the library logic.
/// Expects a reference to [`Config`](struct@Config).
///
/// # Errors
///
/// Returns error on the following situations:
///
/// - `file` doesn't exist.
/// - `file` could not be read (unauthorized read permission).
pub fn run(config: Config) -> Result<(), Box<dyn Error>> {
    /* `?` will return the error(ending fn execution)
     * if it encounters an `Err` in the `Result` it follows. */
    let (mut file, mut data): (File, String) = (File::open(&config.file)?, String::new());

    file.read_to_string(&mut data)?;

    let result: Vec<String> = search(&config.pattern, &data);

    if result.is_empty() {
        return Err((format!("No match found for {} in {}", &config.pattern, &config.file)).into());
    } else {
        let (mut line_count, mut total_occurrence): (u128, u128) = (0u128, 0u128);

        let mut matched_result: String = String::new();

        for line in result {
            let occurrence: u128 = line.matches(&config.pattern).count().try_into().unwrap();
            total_occurrence += occurrence;

            let formatted_output: String = format!("{} ~{}\n", line, occurrence);

            matched_result.push_str(&*formatted_output);
            line_count += 1;
        }

        let input_summary: String =
            format!("Lines matching {} in {}:", &config.pattern, &config.file);

        println!(
            "{}\n\n{}\n{}",
            input_summary,
            matched_result,
            summarize(total_occurrence, line_count)
        );
    }

    /* `()` is a unit type.
     * We do, however, care about the errors that might occur, and that's why the result type
     * exists with a dynamic error return type. */
    Ok(())
}

#[cfg(test)]
mod test {
    /* use every fn from above */
    use super::*;

    #[test]
    fn case_sensitive() {
        let pattern: &str = "omg";
        let data: &str = "\n\n\n\nomg, rust is so cool!";

        assert_eq!(vec!["5: omg, rust is so cool!"], search(pattern, data));
    }

    #[test]
    fn case_sensitive_multiline() {
        let pattern: &str = "th";
        let data: &str = "Rust is the best.\nIt is the best.\nThe best!";

        assert_eq!(
            vec!["1: Rust is the best.", "2: It is the best."],
            search(pattern, data)
        );
    }

    #[test]
    fn case_insensitive() {
        let pattern: &str = "ok";
        let data: &str = "Ok, I'm going crazy!";

        assert_ne!(vec!["1: Ok, I'm going crazy!"], search(pattern, data));
    }
}
