// Copyright 2019-2021 Parity Technologies (UK) Ltd.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

use crate::prelude::{
    borrow::{
        Cow,
        ToOwned,
    },
    boxed::Box,
    collections::{
        BTreeMap,
        BTreeSet,
        VecDeque,
    },
    marker::PhantomData,
    string::String,
    vec::Vec,
};

use crate::{
    build::*,
    MetaType,
    Path,
    Type,
    TypeDefArray,
    TypeDefCompact,
    TypeDefPhantom,
    TypeDefPrimitive,
    TypeDefSequence,
    TypeDefTuple,
    TypeInfo,
};

macro_rules! impl_metadata_for_primitives {
    ( $( $t:ty => $ident_kind:expr, )* ) => { $(
        impl TypeInfo for $t {
            type Identity = Self;

            fn type_info() -> Type {
                $ident_kind.into()
            }
        }
    )* }
}

impl_metadata_for_primitives!(
    bool => TypeDefPrimitive::Bool,
    char => TypeDefPrimitive::Char,
    u8 => TypeDefPrimitive::U8,
    u16 => TypeDefPrimitive::U16,
    u32 => TypeDefPrimitive::U32,
    u64 => TypeDefPrimitive::U64,
    u128 => TypeDefPrimitive::U128,
    i8 => TypeDefPrimitive::I8,
    i16 => TypeDefPrimitive::I16,
    i32 => TypeDefPrimitive::I32,
    i64 => TypeDefPrimitive::I64,
    i128 => TypeDefPrimitive::I128,
);

impl<T: TypeInfo + 'static, const N: usize> TypeInfo for [T; N] {
    type Identity = Self;

    fn type_info() -> Type {
        TypeDefArray::new(N as u32, MetaType::new::<T>()).into()
    }
}

macro_rules! impl_metadata_for_tuple {
    ( $($ty:ident),* ) => {
        impl<$($ty),*> TypeInfo for ($($ty,)*)
        where
            $(
                $ty: TypeInfo+ 'static,
            )*
        {
            type Identity = Self;

            fn type_info() -> Type {
                TypeDefTuple::new(tuple_meta_type!($($ty),*)).into()
            }
        }
    }
}

impl_metadata_for_tuple!();
impl_metadata_for_tuple!(A);
impl_metadata_for_tuple!(A, B);
impl_metadata_for_tuple!(A, B, C);
impl_metadata_for_tuple!(A, B, C, D);
impl_metadata_for_tuple!(A, B, C, D, E);
impl_metadata_for_tuple!(A, B, C, D, E, F);
impl_metadata_for_tuple!(A, B, C, D, E, F, G);
impl_metadata_for_tuple!(A, B, C, D, E, F, G, H);
impl_metadata_for_tuple!(A, B, C, D, E, F, G, H, I);
impl_metadata_for_tuple!(A, B, C, D, E, F, G, H, I, J);
impl_metadata_for_tuple!(A, B, C, D, E, F, G, H, I, J, K);
impl_metadata_for_tuple!(A, B, C, D, E, F, G, H, I, J, K, L);
impl_metadata_for_tuple!(A, B, C, D, E, F, G, H, I, J, K, L, M);
impl_metadata_for_tuple!(A, B, C, D, E, F, G, H, I, J, K, L, M, N);
impl_metadata_for_tuple!(A, B, C, D, E, F, G, H, I, J, K, L, M, N, O);
impl_metadata_for_tuple!(A, B, C, D, E, F, G, H, I, J, K, L, M, N, O, P);
impl_metadata_for_tuple!(A, B, C, D, E, F, G, H, I, J, K, L, M, N, O, P, Q);
impl_metadata_for_tuple!(A, B, C, D, E, F, G, H, I, J, K, L, M, N, O, P, Q, R);
impl_metadata_for_tuple!(A, B, C, D, E, F, G, H, I, J, K, L, M, N, O, P, Q, R, S);
impl_metadata_for_tuple!(A, B, C, D, E, F, G, H, I, J, K, L, M, N, O, P, Q, R, S, T);

impl<T> TypeInfo for Vec<T>
where
    T: TypeInfo + 'static,
{
    type Identity = [T];

    fn type_info() -> Type {
        Self::Identity::type_info()
    }
}

impl<T> TypeInfo for VecDeque<T>
where
    T: TypeInfo + 'static,
{
    type Identity = [T];

    fn type_info() -> Type {
        Self::Identity::type_info()
    }
}

impl<T> TypeInfo for Option<T>
where
    T: TypeInfo + 'static,
{
    type Identity = Self;

    fn type_info() -> Type {
        Type::builder()
            .path(Path::prelude("Option"))
            .type_params(tuple_meta_type![T])
            .variant(Variants::new().variant("None", |v| v).variant("Some", |v| {
                v.fields(Fields::unnamed().field(|f| f.ty::<T>()))
            }))
    }
}

impl<T, E> TypeInfo for Result<T, E>
where
    T: TypeInfo + 'static,
    E: TypeInfo + 'static,
{
    type Identity = Self;

    fn type_info() -> Type {
        Type::builder()
            .path(Path::prelude("Result"))
            .type_params(tuple_meta_type!(T, E))
            .variant(
                Variants::new()
                    .variant("Ok", |v| v.fields(Fields::unnamed().field(|f| f.ty::<T>())))
                    .variant("Err", |v| {
                        v.fields(Fields::unnamed().field(|f| f.ty::<E>()))
                    }),
            )
    }
}

impl<T> TypeInfo for Cow<'static, T>
where
    T: ToOwned + TypeInfo + ?Sized + 'static,
{
    type Identity = Self;

    fn type_info() -> Type {
        Type::builder()
            .path(Path::prelude("Cow"))
            .type_params(tuple_meta_type!(T))
            .composite(Fields::unnamed().field(|f| f.ty::<T>()))
    }
}

impl<K, V> TypeInfo for BTreeMap<K, V>
where
    K: TypeInfo + 'static,
    V: TypeInfo + 'static,
{
    type Identity = Self;

    fn type_info() -> Type {
        Type::builder()
            .path(Path::prelude("BTreeMap"))
            .type_params(tuple_meta_type![(K, V)])
            .composite(Fields::unnamed().field(|f| f.ty::<[(K, V)]>()))
    }
}

impl<T> TypeInfo for BTreeSet<T>
where
    T: TypeInfo + 'static,
{
    type Identity = Self;

    fn type_info() -> Type {
        Type::builder()
            .path(Path::prelude("BTreeSet"))
            .type_params(tuple_meta_type![T])
            .composite(Fields::unnamed().field(|f| f.ty::<[T]>()))
    }
}

impl<T> TypeInfo for Box<T>
where
    T: TypeInfo + ?Sized + 'static,
{
    type Identity = T;

    fn type_info() -> Type {
        Self::Identity::type_info()
    }
}

impl<T> TypeInfo for &T
where
    T: TypeInfo + ?Sized + 'static,
{
    type Identity = T;

    fn type_info() -> Type {
        Self::Identity::type_info()
    }
}

impl<T> TypeInfo for &mut T
where
    T: TypeInfo + ?Sized + 'static,
{
    type Identity = T;

    fn type_info() -> Type {
        Self::Identity::type_info()
    }
}

impl<T> TypeInfo for [T]
where
    T: TypeInfo + 'static,
{
    type Identity = Self;

    fn type_info() -> Type {
        TypeDefSequence::of::<T>().into()
    }
}

impl TypeInfo for str {
    type Identity = Self;

    fn type_info() -> Type {
        TypeDefPrimitive::Str.into()
    }
}

impl TypeInfo for String {
    type Identity = str;

    fn type_info() -> Type {
        Self::Identity::type_info()
    }
}

impl<T> TypeInfo for PhantomData<T>
where
    T: TypeInfo + ?Sized + 'static,
{
    type Identity = Self;

    fn type_info() -> Type {
        TypeDefPhantom::new(MetaType::new::<T>()).into()
    }
}

impl<T> TypeInfo for scale::Compact<T>
where
    T: TypeInfo + 'static,
{
    type Identity = Self;
    fn type_info() -> Type {
        TypeDefCompact::new(MetaType::new::<T>()).into()
    }
}

#[cfg(feature = "bit-vec")]
mod bit_vec {
    use super::*;

    impl<O, T> TypeInfo for bitvec::vec::BitVec<O, T>
    where
        O: bitvec::order::BitOrder + TypeInfo + 'static,
        T: bitvec::store::BitStore + TypeInfo + 'static,
    {
        type Identity = Self;

        fn type_info() -> Type {
            crate::TypeDefBitSequence::new::<O, T>().into()
        }
    }

    impl TypeInfo for bitvec::order::Lsb0 {
        type Identity = Self;

        fn type_info() -> Type {
            Type::builder()
                .path(Path::new("Lsb0", "bitvec::order"))
                .composite(Fields::unit())
        }
    }

    impl TypeInfo for bitvec::order::Msb0 {
        type Identity = Self;

        fn type_info() -> Type {
            Type::builder()
                .path(Path::new("Msb0", "bitvec::order"))
                .composite(Fields::unit())
        }
    }
}
