#!/usr/bin/perl -w
#
# Load a VDatum grid file and output two files containing C-style arrays named
# GEOID:
#
# 1. geoid_model_1_degree.inc: Downsampled geoid data (1 x 1 degree gid)
# 2. geoid_model_15_minute.inc: Full resolution geoid data (0.25 x 0.25 degree grid)
#
# See https://vdatum.noaa.gov/docs/gtx_info.html#dev_gtx_binary for details
# of the VDatum file format.
#
# (C) 2020 Swift Navigation
#

use strict;

# file names
use constant INPUT_DATA    => "egm08_25.gtx";
use constant OUTPUT_LO_RES => "geoid_model_1_degree.inc";
use constant OUTPUT_HI_RES => "geoid_model_15_minute.inc";

# boundaries
use constant MIN_LON => 0;
use constant MAX_LON => 360;
use constant MIN_LAT => -90;
use constant MAX_LAT => 90;


# hash contains height values in metres keyed with "$lat,$lon"
# (so that we can perform random lookups)
my %heights;


# read data from '$infile', store data in %heights
sub read_file($) {
    my $infile = shift || die;

    open(FILE, "<$infile") || die "Cannot open $infile for reading";
    binmode FILE;

    sub read_uint32() {
        die if((read FILE, my $data, 4) != 4);
        return unpack("L>", $data);
    }

    sub read_double() {
        die if((read FILE, my $data, 8) != 8);
        return unpack("d>", $data);
    }

    sub read_float() {
        die if((read FILE, my $data, 4) != 4);
        return unpack("f>", $data);
    }

    # compare 2 floats to 7 decimal places
    sub float_cmp($$) {
        return sprintf("%.7f", $_[0]) eq sprintf("%.7f", $_[1]);
    }

    # parse header
    my $start_lat = read_double();
    my $start_lon = read_double();
    my $delta_lat = read_double();
    my $delta_lon = read_double();
    my $nrows = read_uint32();
    my $ncols = read_uint32();

    my $end_lat = $start_lat + ($nrows-1) * $delta_lat;
    my $end_lon = $start_lon + ($ncols-1) * $delta_lon;

    # sanity checks
    die unless float_cmp($start_lat, -90);
    die unless float_cmp($start_lon, -180);
    die unless float_cmp($end_lat, 90);
    die unless float_cmp($end_lon + $delta_lon, 180);

    # read height data
    for my $row (0..$nrows-1) {
        for my $col (0..$ncols-1) {
            my $lat = $start_lat + $row * $delta_lat;
            my $lon = $start_lon + $col * $delta_lon;
            $lon += 360 if($lon < 0);
            $heights{"$lat,$lon"} = read_float();
        }
    }

    close FILE;
}


# return height value for '$lat'/'$lon' (in metres)
sub get_height($$) {
    my($lat, $lon) = @_;

    my $height = $heights{"$lat,$lon"};
    die "$lat/$lon" unless defined($height);
    return $height;
}


# write geoid data with '$spacing' (in degrees) to '$outfile'
sub write_geoid_file($$) {
    my($outfile, $spacing) = @_;

    open(FILE, ">$outfile") || die "Cannot open $outfile for writing";

    my $rows = (MAX_LON - MIN_LON) / $spacing + 1;
    my $cols = (MAX_LAT - MIN_LAT) / $spacing + 1;

    # write header
    print FILE <<END;
/* File automatically generated by $0, do not edit */

static const float LAT_GRID_SPACING_DEG = $spacing;
static const float LON_GRID_SPACING_DEG = $spacing;


static const float GEOID[$rows][$cols] = {
END

    sub write_row($$) {
        my($lon, $spacing) = @_;

        print FILE "    {";
        for(my $lat = MIN_LAT; $lat <= MAX_LAT; $lat += $spacing) {
            printf FILE "%ff", get_height($lat, $lon);

            if($lat != 90) {
                print FILE ", ";
            }
        }
        print FILE "}";
    }

    # generate 2D array where each "row" defines the height values for a
    # single line of longitude
    for(my $lon = MIN_LON; $lon < MAX_LON; $lon += $spacing) {
        write_row($lon, $spacing);
        print FILE ",\n";
    }
    write_row(0, $spacing);

    # write footer
    print FILE <<END;

};
END
    close(FILE);
}


read_file(INPUT_DATA);
write_geoid_file(OUTPUT_LO_RES, 1);
write_geoid_file(OUTPUT_HI_RES, 0.25);
