#!/usr/bin/env python

# Copyright (C) 2021 Swift Navigation Inc.
# Contact: Swift Navigation <dev@swift-nav.com>
#
# This source is subject to the license found in the file 'LICENSE' which must
# be be distributed together with this source. All other rights reserved.
#
# THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
# EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
#

#
# This script uses a file generated by the profiling tool Heaptrack as input
# and writes the peak heap memory consumption into a log file.
#
# Heaptrack comes with a tool called heaptrack_print which is used to analyze
# the collected data and print the result.
#
# USAGE
#
#   python parse_heaptrack.py [OPTIONS]
#
# Run the script by supplying an input file path to a Heaptrack file and a file
# path to where the log file should be created.
#
# OPTIONS
# * -i, --input_file:  Sets the input file path.
# * -o, --output_file: Sets the output file path.
# * -m, --message:     Adds a message to the reported memory usage.
#
import argparse
import re
import subprocess
import sys

parser = argparse.ArgumentParser(description='Log peak heap memory consumption reported by Heaptrack.')
optional = parser._action_groups.pop()
required = parser.add_argument_group('required arguments')
required.add_argument('-i','--input_file',
                      help='File path where a Heaptrack file is located',
                      required=True)
required.add_argument('-o','--output_file',
                      help='File path where the log should be created',
                      required=True)
optional.add_argument('-m','--message',
                      help='Custom message that gets concatenated with the reported memory usage',
                      default='Heap memory usage:')
parser._action_groups.append(optional)
args = parser.parse_args()

try:
  foutput = open(args.output_file,"a")
except IOError:
  sys.exit()

p_heaptrack_print = subprocess.Popen(['which', 'heaptrack_print'], stdout=subprocess.PIPE)
program_path = p_heaptrack_print.stdout.read().rstrip()
p_heaptrack_print.terminate()

p = subprocess.Popen([program_path, args.input_file], stdout=subprocess.PIPE)
text = []
for line in p.stdout:
  text.append(line)
for string in reversed(text):
  if "peak heap memory consumption" in string:
    m = re.search('\d',string)
    result = string[m.start():].strip()
    message = "{} {}\n".format(args.message, result)
    foutput.write(message)
    break
foutput.close()
p.terminate()
