use std::{io, vec};

fn main() {
    let lines = read_all_lines();
    let max_len = lines.iter().map(|x| x.len()).max().unwrap() + 1;

    let line_vectors = lines
        .iter()
        .map(move |s| s.clone() + &str::repeat(" ", max_len - s.len()))
        .map(|s| s.chars().collect::<Vec<_>>())
        .collect::<Vec<_>>();

    debug_assert!(line_vectors.iter().all(|x| x.len() == max_len));

    for (index, line) in line_vectors.iter().enumerate() {
        let mut new_line = String::new();
        for (i, c) in line.iter().enumerate() {
            let new_val = if !c.is_whitespace() {
                let pt = Point { row: index, col: i };
                let loc = get_location(&line_vectors, pt);
                let jiggy = loc.get_jiggy();
                // println!("{:?}", loc);
                // println!("{:?} - jiggy", jiggy);
                if let Some(v) = jiggy {
                    v
                } else {
                    '🔥'
                }
            } else {
                ' '
            };

            new_line.push(new_val);
        }

        println!("│{}│", new_line);
    }

    // for line in z {
    //     println!("Original:")
    //     println!("|{:?}|", line);
    // }
}

fn get_location(vec: &Vec<Vec<char>>, anchor: Point) -> Location {
    // println!("Current point: {:?} - CanSample? top {} btm {}", anchor, can_sample_top, can_sample_bottom);
    // println!("Current character: {}", vec[anchor.row][anchor.col]);
    let family = kidnap(vec, &anchor);

    return Location {
        // position: Point { ..anchor },
        value: vec[anchor.row][anchor.col],
        family,
    };
}

fn kidnap(vec: &Vec<Vec<char>>, anchor: &Point) -> Vec<char> {
    let mut family = Vec::new();

    if anchor.row != 0 {
        family.push(vec[anchor.row - 1][anchor.col]); //top
    } else {
        family.push(' ')
    }

    family.push(vec[anchor.row][anchor.col - 1]); //left
    //family.push(vec[anchor.row][anchor.col]); //value --- Uncomment to go back to .1
    
    family.push(vec[anchor.row][anchor.col + 1]); //right

    if anchor.row != vec.len() - 1 {
        family.push(vec[anchor.row + 1][anchor.col]); //btm
    } else {
        family.push(' ')
    }

    return family;
}

fn read_all_lines() -> Vec<String> {
    let mut buf = String::new();
    buf.reserve(4096);
    let stdin = io::stdin();

    let mut vec = Vec::<String>::new();

    while let Ok(_x @ 1..) = stdin.read_line(&mut buf) {
        vec.push(format!(" {}", buf.trim_end()));
        buf.clear();
    }

    return vec;
}

#[derive(Debug)]
struct Location {
    // position: Point,
    value: char,
    family: Vec<char>,
}

#[derive(Debug)]
struct Point {
    row: usize,
    col: usize,
}

trait IntoJiggy {
    fn into_jiggy(&self) -> Option<char>;
}

impl Location {
    fn get_jiggy(&self) -> Option<char> {
        match self.value {
            // '-' => translate_dashy(self),
            //'-' => translate_dashy_bools(self),
            '-' => translate_dashy_dirns(self),
            // '|' => translate_pipe_bools(self),
            '|' => translate_pipey_dirns(self),
            '>' | '<' | '^' | 'v' => translate_pointy(self),
            //'"' => Some('ʺ'), //term output or w/e
            x => Some(x),
        }
    }
}

fn translate_dashy(loc: &Location) -> Option<char> {
    match loc.family[..] {
        [x, '-', '-', ' ', '|'] if !x.can_connect_from(Direction::Bottom)   => Some('┐'),
        [x, ' ', '-', '-', '|'] if !x.can_connect_from(Direction::Bottom)   => Some('┌'),
        ['|', '-', '-', ' ', x] if !x.can_connect_from(Direction::Top)      => Some('┘'),
        ['|', ' ', '-', '-', x] if !x.can_connect_from(Direction::Top)      => Some('└'),
        [z, x, '-', y, '|'] if x.can_connect_from(Direction::Right) && y.can_connect_from(Direction::Left) && !z.can_connect_from(Direction::Bottom) =>
        {
            Some('┬')
        }
        ['|', x, '-', y, z]
            if x.is_connectable() && y.is_connectable() && (z.is_whitespace() || z == '-') =>
        {
            Some('┴')
        }
        ['|', '-', '-', '-', '|'] => Some('┼'),
        _ => Some('─'),
    }
}

fn connecty_bits(loc: &Location) -> Vec<bool> {
    loc.family.iter().enumerate().map(|(i,v)| {
        let dirn = match i { //CN: Directions to center
            0 => Direction::Bottom,
            1 => Direction::Right,
            2 => Direction::Left,
            3 => Direction::Top,
            _ => unreachable!()
         };
         
        //  if loc.value == '|'{
        //     println!("Target: {} to the {:?} of {} -- Can connect from: {}, should bridge? {}", v, dirn.opposite(), loc.value,v.can_connect_from(dirn),v.should_bridge_from(dirn) || loc.value.should_bridge_from(dirn.opposite()));
        //  }
         
         v.can_connect_from(dirn) && (v.should_bridge_from(dirn) || loc.value.should_bridge_from(dirn.opposite()))
    }).collect()
}

fn connecty_bits_new(loc: &Location) -> Vec<Direction> {
    let mut vector = loc.family.iter().enumerate().map(|(i,v)| {
        let dirn = match i { //CN: Directions to center
            0 => Direction::Bottom,
            1 => Direction::Right,
            2 => Direction::Left,
            3 => Direction::Top,
            _ => unreachable!()
         };
         
        //  if loc.value == '|'{
        //     println!("Target: {} to the {:?} of {} -- Can connect from: {}, should bridge? {}", v, dirn.opposite(), loc.value,v.can_connect_from(dirn),v.should_bridge_from(dirn) || loc.value.should_bridge_from(dirn.opposite()));
        //  }
         
         if v.can_connect_from(dirn) && (v.should_bridge_from(dirn) || loc.value.should_bridge_from(dirn.opposite())) {
             Some(dirn.opposite())
         }
         else {
             None
         }
    }).filter_map(|x| {x}).collect::<Vec<Direction>>();
    vector.sort();
    // println!("{:?}", vector);
    return vector;
}

fn translate_dashy_bools(loc: &Location) -> Option<char> {
    match connecty_bits(loc).as_slice() {
        //Top left  right  btm
        [ false, true, false, true] => Some('┐'),
        [ false, false, true, true] => Some('┌'),
        [ true, true, false, false] => Some('┘'),
        [ true, false, true, false] => Some('└'),
        [ false, true, true, true] => Some('┬'),
        [ true, true, true, false] => Some('┴'),
        [ true, true, true, true] => Some('┼'),
        _ => Some('─'),
    }
}

fn translate_dashy_dirns(loc: &Location) -> Option<char> {
    match connecty_bits_new(loc).as_slice() {
        //Top left  right  btm
        [ Direction::Left, Direction::Bottom] => Some('┐'),
        [ Direction::Right, Direction::Bottom] => Some('┌'),
        [ Direction::Top, Direction::Left] => Some('┘'),
        [ Direction::Top, Direction::Right] => Some('└'),
        [ Direction::Left, Direction::Right, Direction::Bottom] => Some('┬'),
        [ Direction::Top, Direction::Left, Direction::Right] => Some('┴'),
        [ Direction::Top, Direction::Left, Direction::Right, Direction::Bottom] => Some('┼'),
        _ => Some('─'),
    }
}


fn translate_pipey_dirns(loc: &Location) -> Option<char> {
    match connecty_bits_new(loc).as_slice() {
        [ Direction::Top, Direction::Right, Direction::Bottom ] => Some('├'),
        [ Direction::Top, Direction::Left, Direction::Bottom ]  => Some('┤'),
        [ Direction::Top, Direction::Left] => Some('┘'),
        [ Direction::Top, Direction::Right] => Some('└'),
        [ Direction::Left, Direction::Bottom]  => Some('┐'),
        [ Direction::Right, Direction::Bottom] => Some('┌'),
        _ => Some('│'),
    }
}

fn translate_pipe_bools(loc: &Location) -> Option<char> {
    // println!("XLATE PIPES");
    match connecty_bits(loc).as_slice() {
        //Top left  right  btm
        [true, false, true, true ] => Some('├'),
        [true, true, false, true] => Some('┤'),
        [true, true, false, false] => Some('┘'),
        [true, false, true, false] => Some('└'),
        [false, true, false, true]  => Some('┐'),
        _ => Some('│'),
    }
}

fn translate_pointy(loc: &Location) -> Option<char> {
    match loc.family[..] {
        [_, '-', ..] => Some('►'),
        [.., '|'] => Some('▲'),
        [.., '-', _] => Some('►'),
        ['|',..] => Some('▼'), //Robert
        _ => None,
    }

    //Some('►')
}


fn translate_pipe(loc: &Location) -> Option<char> {
    match loc.family[..] {
        [_, ' ', '|', '-', y] if y.is_connectable() => Some('├'),
        [' ', ' ', '|', '-', y] if !y.is_connectable() => Some('┘'),
        [y, '-', '|', ' ', x] if x.is_connectable() && y.is_connectable() => Some('┤'),
        [y, '-', '|', ' ', ' '] if y.is_connectable() => Some('┘'),
        [' ', '-', '|', ' ', y] if y.is_connectable() => Some('┐'),
        _ => Some('│'),
    }
}

#[derive(PartialEq, Eq, Clone, Copy, Debug, PartialOrd, Ord)]
enum Direction {
    Top,
    Left,
    Right,
    Bottom,
}

impl Direction{
    fn opposite(&self) -> Direction {
        match self {
            Direction::Top => Self::Bottom,
            Direction::Left=> Self::Right,
            Direction::Bottom=> Self::Top, 
            Direction::Right => Self::Left
        }
    }
}

trait Connectable {
    fn is_connectable(&self) -> bool;
    fn can_connect_from(&self, dirn: Direction) -> bool;
    fn should_bridge_from(&self, dirn: Direction) -> bool;
}

//The concept of creating a branch is different than terminating a branch. 
//     |
//  ----Thing 
// makes sense that there would be a T-junc before Thing but
//   BBBBBBBBBBBBBBB    BBBBBBBBBBBBBBB
//   --------------- != ┼┼┼┼┼┼┼┼┼┼┼┼┼┼┼
//   AAAAAAAAAAAAAAA    AAAAAAAAAAAAAAA 
//
//
// should_bridge?
// letters -> Never
// pipes => top + bottom
// dashes => left and right
// pointies => never



impl Connectable for char {
    fn is_connectable(&self) -> bool {
        return !self.is_whitespace();
    }

    fn can_connect_from(&self, dirn: Direction) -> bool {
        match self {
            '-' => dirn == Direction::Left || dirn == Direction::Right || dirn == Direction::Top || dirn == Direction::Bottom,
            '|' => dirn == Direction::Left || dirn == Direction::Right || dirn == Direction::Top || dirn == Direction::Bottom,
            '>' => dirn == Direction::Left,
            '<' => dirn == Direction::Right,
            '^' => dirn == Direction::Bottom,
            'v' => dirn == Direction::Bottom,
            x if x.is_whitespace() => false,
            _ => true,
        }
    }

    fn should_bridge_from(&self, dirn: Direction) -> bool {
        match self {
            '-' => dirn == Direction::Left || dirn == Direction::Right,
            '|' => dirn == Direction::Top || dirn == Direction::Bottom,
            _ => false,
        }
    }
}
