#![deny(clippy::all, clippy::nursery)]
#![deny(nonstandard_style, rust_2018_idioms)]
// #![deny(missing_docs, missing_debug_implementations)]
#![forbid(unsafe_code)]

use anyhow::Result;
use derive_builder::Builder;
use serde::Deserialize;
use serde_repr::Deserialize_repr;
use std::env;

#[derive(Builder, Debug)]
#[builder(setter(into, strip_option))]
pub struct Client {
    #[builder(default = "String::from(\"https://api.savvycal.com\")")]
    base_url: String,
    #[builder(default)]
    api_token: Option<String>,
    #[builder(default)]
    http_client: reqwest::Client,
}

impl Default for Client {
    fn default() -> Self {
        ClientBuilder::default().build().unwrap()
    }
}

#[derive(Builder, Deserialize, PartialEq, Debug)]
#[builder(setter(into))]
pub struct User {
    pub id: String,
    pub email: String,
    pub display_name: String,
    pub last_name: String,
    pub avatar_url: String,
    pub time_zone: String,
    pub time_format: String,
    pub first_day_of_week: FirstDayOfWeek,
    pub plan: String,
}

#[derive(Deserialize_repr, PartialEq, Clone, Debug)]
#[repr(u8)]
pub enum FirstDayOfWeek {
    Sunday = 0,
    Monday,
}

impl Client {
    pub fn from_env() -> Result<Self> {
        let client = ClientBuilder::default()
            .api_token(env::var("SAVVYCAL_TOKEN")?)
            .build()?;

        Ok(client)
    }

    pub async fn get_current_user(&self) -> Result<User> {
        Ok(self.get("/v1/me").await?.json::<User>().await?)
    }

    async fn get(&self, path: &str) -> Result<reqwest::Response> {
        let mut builder = self.http_client.get(format!("{}{}", self.base_url, path));

        if let Some(token) = &self.api_token {
            builder = builder.bearer_auth(token)
        }

        let resp = builder.send().await?.error_for_status()?;

        Ok(resp)
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use pretty_assertions::assert_eq;
    use wiremock::matchers::{method, path};
    use wiremock::{Mock, MockServer, ResponseTemplate};

    #[tokio::test]
    async fn test_get_current_user() {
        let server = MockServer::start().await;
        let resp = ResponseTemplate::new(200).set_body_raw(include_str!("testdata/me.json"), "application/json");

        Mock::given(method("GET"))
            .and(path("/v1/me"))
            .respond_with(resp)
            .expect(1)
            .mount(&server)
            .await;

        let got = ClientBuilder::default()
            .base_url(&server.uri())
            .build()
            .unwrap()
            .get_current_user()
            .await
            .unwrap();

        let want = UserBuilder::default()
            .id("user_0123456789")
            .email("john.doe@example.com")
            .display_name("John Doe")
            .last_name("Doe")
            .avatar_url("https://secure.gravatar.com/avatar/0123456789")
            .time_zone("Europe/Berlin")
            .time_format("24hr")
            .first_day_of_week(FirstDayOfWeek::Monday)
            .plan("basic")
            .build()
            .unwrap();

        assert_eq!(got, want);
    }
}
