#![deny(clippy::all, clippy::nursery)]
#![deny(nonstandard_style, rust_2018_idioms)]
// #![deny(missing_docs, missing_debug_implementations)]
#![forbid(unsafe_code)]

use anyhow::Result;
use derive_builder::Builder;
use serde::Deserialize;
use std::env;

#[derive(Builder, Debug)]
pub struct Client {
    #[builder(default = "String::from(\"https://api.savvycal.com\")")]
    base_url: String,
    #[builder(default = "env::var(\"SAVVYCAL_TOKEN\").expect(\"SAVVYCAL_TOKEN not found\")")]
    api_token: String,
    #[builder(default)]
    http_client: reqwest::Client,
}

impl Default for Client {
    fn default() -> Self {
        ClientBuilder::default().build().unwrap()
    }
}

#[derive(Deserialize, Debug)]
pub struct User {
    pub id: String,
    pub email: String,
    pub display_name: String,
    pub last_name: String,
    pub avatar_url: String,
    pub time_zone: String,
    pub time_format: String,
    pub first_day_of_week: u8,
    pub plan: String,
}

impl Client {
    pub async fn get_current_user(&self) -> Result<User> {
        let user = self
            .http_client
            .get(format!("{}/v1/me", self.base_url))
            .bearer_auth(&self.api_token)
            .send()
            .await?
            .error_for_status()?
            .json::<User>()
            .await?;

        Ok(user)
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[tokio::test]
    #[ignore]
    async fn test_get_current_user() {
        let user = Client::default().get_current_user().await.unwrap();
        dbg!(user);
    }
}
