#![allow(unused_imports)]

use {
    git2::{RepositoryInitOptions, Time},
    std::time::{SystemTime, UNIX_EPOCH},
    ::{
        clap::Parser,
        eyre::{bail, ensure, eyre, Result as Fallible, WrapErr},
        git2::{Commit, ErrorCode as GitErrorCode, Repository, Signature},
        std::{
            env, fs,
            process::{exit, Command},
        },
        tracing::{debug, error, info, log, trace, warn},
    },
};

/// Would you like to SAVE the change?
///
/// Commits everything in the current Git repository, no questions asked.
#[derive(Parser, Debug, Clone)]
#[clap(version)]
struct Args {
    /// Use a manual commit message instead of the default generated message.
    #[clap(display_order = 1_1, long, short, env = "SAVE_MESSAGE")]
    message: Option<String>,

    /// The name to use for the commit. If not specified, this will fall
    /// back to name configured in Git. If not specified, this will fall
    /// back to the last commit's name. If not specified, this will fall
    /// back to a default placeholder value.
    #[clap(display_order = 1_2, long, env = "GIT_AUTHOR_NAME")]
    name: Option<String>,

    /// The email to use for the commit. If not specified, this will fall
    /// back to email configured in Git. If not specified, this will fall
    /// back to the last commit's email. If not specified, this will fall
    /// back to a default placeholder value.
    #[clap(display_order = 1_3, long, env = "GIT_AUTHOR_EMAIL")]
    email: Option<String>,

    /// Creates a new commit even if there are no changes to save.
    #[clap(display_order = 1_4, long, short = 'y', env = "SAVE_ALLOW_EMPTY")]
    allow_empty: bool,

    /// Use UTC, instead of the system time zone.
    #[clap(
        display_order = 1_5,
        long,
        parse(try_from_str),
        default_value = "true",
        env = "SAVE_USE_UTC"
    )]
    use_utc: bool,

    /// Timestamp will be a multiple of this number of seconds.
    #[clap(
        display_order = 2_1,
        long,
        default_value_t = 64,
        env = "SAVE_STEP_SECONDS"
    )]
    step_seconds: i64,

    /// Timestamp will snap to a multiple of this many seconds when catching up.
    #[clap(
        display_order = 2_2,
        long,
        default_value_t = 64 * 64 * 4,
        env = "SAVE_SNAP_SECONDS"
    )]
    snap_seconds: i64,

    /// Timestamp will lag up to this many seconds behind current time before
    /// catching up.
    #[clap(
        display_order = 2_3,
        long,
        default_value_t = 64 * 64 * 16,
        env = "SAVE_SLACK_SECONDS"
    )]
    slack_seconds: i64,

    /// Log level to use (trace, debug, info, warn, or error).
    #[clap(
        display_order = 3_1,
        long,
        default_value = "debug",
        env = "SAVE_LOG_LEVEL"
    )]
    log_level: String,

    /// If no Git repository is found, automatically initialize one in the
    /// current directory.
    #[clap(
        long,
        parse(try_from_str),
        default_value = "true",
        env = "SAVE_AUTO_INIT"
    )]
    auto_init: bool,

    /// The default branch name to use when automatically initializing a
    /// repository.
    #[clap(long, default_value = "trunk", env = "SAVE_AUTO_INIT_BRANCH")]
    auto_init_branch: String,

    /// Treat this as the current timestamp in seconds, ignoring the system
    /// clock.
    #[clap(long)]
    now: Option<i64>,
}

fn main() -> Fallible<()> {
    let args = init();

    trace!("{:#?}", args);

    ensure!(args.step_seconds >= 0, "step_seconds must be non-negative");
    ensure!(args.snap_seconds >= 0, "snap_seconds must be non-negative");
    ensure!(
        args.slack_seconds >= 0,
        "slack_seconds must be non-negative"
    );
    ensure!(
        args.step_seconds == 0 || args.snap_seconds % args.step_seconds == 0,
        "snap_seconds must be a multiple of step_seconds"
    );
    ensure!(
        args.step_seconds == 0 || args.slack_seconds % args.step_seconds == 0,
        "slack_seconds must be a multiple of step_seconds"
    );
    ensure!(
        args.slack_seconds >= args.step_seconds,
        "slack_seconds must be greater than snap_seconds"
    );

    let repo = match Repository::open_from_env() {
        Ok(repo) => {
            if repo.is_bare() {
                bail!(
                    "Found Git repository, but it was bare (no working directory): {:?}",
                    repo.path()
                );
            }

            debug!("Found Git repository: {:?}", repo.workdir().unwrap());
            repo
        },
        Err(_err) => {
            let path = std::env::current_dir()?;
            let empty = fs::read_dir(&path)?.next().is_none();

            if args.auto_init {
                if !empty || args.allow_empty {
                    info!(
                        "No Git repository found, attempting to initializing a new one in: {:?}",
                        path
                    );
                    Repository::init_opts(
                        path,
                        RepositoryInitOptions::new()
                            .initial_head(&args.auto_init_branch)
                            .no_reinit(true),
                    )?
                } else {
                    bail!(
                        "No Git repository found, and current directory has nothing to save: {:?}",
                        path
                    );
                }
            } else {
                bail!("No Git repository found.");
            }
        },
    };

    let head = match repo.head() {
        Ok(head) => Some(head.peel_to_commit().unwrap()),
        Err(err) if err.code() == GitErrorCode::UnbornBranch => None,
        Err(err) => {
            bail!("Unexpected error from Git: {:#?}", err);
        },
    };

    let config = repo.config()?;

    let user_name: String = {
        if let Some(args_name) = args.name {
            trace!(
                "Using author name from command line argument: {:?}",
                &args_name
            );
            args_name
        } else if let Ok(config_name) = config.get_string("user.name") {
            debug!(
                "Using author name from Git configuration: {:?}",
                &config_name
            );
            config_name
        } else if let Some(previous_name) = head
            .as_ref()
            .and_then(|x| x.author().name().map(|x| x.to_string()))
        {
            info!(
                "Using author name from previous commit: {:?}",
                &previous_name
            );
            previous_name
        } else {
            let placeholder_name = "save";
            warn!(
                "No author name found, falling back to placeholder: {:?}",
                &placeholder_name
            );
            placeholder_name.to_string()
        }
    };

    let user_email: String = if let Some(args_email) = args.email {
        trace!(
            "Using author email from command line argument: {:?}",
            &args_email
        );
        args_email
    } else if let Ok(config_email) = config.get_string("user.email") {
        debug!(
            "Using author email from Git configuration: {:?}",
            &config_email
        );
        config_email
    } else if let Some(previous_email) = head
        .as_ref()
        .and_then(|x| x.author().email().map(|x| x.to_string()))
    {
        info!(
            "Using author email from previous commit: {:?}",
            &previous_email
        );
        previous_email
    } else {
        let placeholder_email = "save";
        warn!(
            "No author email found, falling back to placeholder: {:?}",
            &placeholder_email
        );
        placeholder_email.to_string()
    };

    let generation_index = head
        .as_ref()
        .map(|commit| find_generation_index(commit) + 1)
        .unwrap_or(0);

    let mut index = repo.index()?;

    index.add_all(["*"], Default::default(), Default::default())?;
    let tree = index.write_tree()?;

    if let Some(ref head) = head {
        if tree == head.tree_id() {
            if args.allow_empty {
                info!("Committing with no changes.");
            } else {
                info!("No changes to commit (use -y/--allow-empty to commit anyway).");
                return Ok(());
            }
        }
    }

    index.write()?;

    let tree4 = &tree.to_string()[..4];
    let tree = repo.find_tree(tree)?;

    let revision_index = generation_index + 1;
    let message = args.message.unwrap_or_else(|| {
        let mut message = format!("r{}", revision_index);
        if let Some(ref head) = head {
            message += &format!("/{}/{}", tree4, &head.id().to_string()[..4]);
        } else if tree.iter().next().is_some() {
            format!("/{}", &tree4);
        }
        message
    });

    let previous_seconds = head.as_ref().map(|c| c.time().seconds()).unwrap_or(0);
    let time = Signature::now(&user_name, &user_email)?.when();
    let mut seconds = args.now.unwrap_or_else(|| time.seconds());
    let mut offset = time.offset_minutes();

    if args.use_utc {
        offset = 0;
    }

    let seconds_since_head = seconds - previous_seconds;

    if seconds_since_head < args.slack_seconds {
        seconds = previous_seconds + args.step_seconds;
    } else {
        seconds = seconds - seconds % args.snap_seconds
    }

    let time = Time::new(seconds, offset);

    let signature = Signature::new(&user_name, &user_email, &time)?;

    repo.commit(
        Some("HEAD"),
        &signature,
        &signature,
        &message,
        &tree,
        &head.iter().collect::<Vec<_>>(),
    )?;

    eprintln!();

    Command::new("git")
        .args(&[
            "--no-pager",
            "log",
            "--name-status",
            "--graph",
            "--decorate",
            "-n",
            "2",
        ])
        .status()?;

    eprintln!();

    Ok(())
}

/// The generation index is the number of edges of the longest path between the
/// given commit and an initial commit (one with no parents, which has an
/// implicit generation index of 0).
fn find_generation_index(commit: &git2::Commit) -> u32 {
    let mut generation_index = 0;
    let mut commits = vec![commit.clone()];

    // naive solution: walk the entire graph depth-first.
    // this could be pathological with a lot of branches.
    // we could use a smarter algorithm, and/or if we come
    // across a commit whose message matches the expected
    // format and tree hash, we trust that it's accurate.
    // however, that could be honestly mangled by a rebase
    // or something, so it might not do.
    loop {
        let mut next_generation_commits = vec![];
        for commit in commits.iter() {
            next_generation_commits.extend(commit.parents());
        }

        if next_generation_commits.is_empty() {
            break;
        } else {
            generation_index += 1;
            commits = next_generation_commits;
            continue;
        }
    }

    generation_index
}

fn init() -> Args {
    dotenv::dotenv().ok();

    let args = Args::parse();

    if !args.log_level.is_empty() {
        env::set_var("RUST_LOG", args.log_level.clone());
    }

    color_eyre::install().unwrap();

    tracing_subscriber::util::SubscriberInitExt::init(tracing_subscriber::Layer::with_subscriber(
        tracing_error::ErrorLayer::default(),
        tracing_subscriber::fmt()
            .with_env_filter(::tracing_subscriber::EnvFilter::from_default_env())
            .with_target(false)
            .with_span_events(
                tracing_subscriber::fmt::format::FmtSpan::NEW
                    | tracing_subscriber::fmt::format::FmtSpan::CLOSE,
            )
            .finish(),
    ));

    args
}
