use indexmap::IndexSet;
use itertools::Itertools;
use serde::{Serialize, Deserialize};

mod variable; pub use variable::CnfVariable;

mod clause;
pub use clause::CnfClause;

use crate::VarIdx;

#[derive(Debug, PartialEq, Eq, Clone, Serialize, Deserialize)]
pub struct CnfFormula {
    number_of_variables: usize,
    sampling_set: IndexSet<VarIdx>,
    clauses: IndexSet<CnfClause>,
}

impl CnfFormula {
    pub fn with_variables(number_of_variables: usize) -> Self {
        Self {
            number_of_variables,
            sampling_set: IndexSet::new(),
            clauses: IndexSet::new(),
        }
    }

    pub fn with_variables_and_capacity(number_of_variables: usize, capacity: usize) -> Self {
        Self {
            number_of_variables,
            sampling_set: IndexSet::new(),
            clauses: IndexSet::with_capacity(capacity),
        }
    }

    pub fn insert(mut self, clause: CnfClause) -> Self {
        if clause
            .max_index()
            .map(|index| index >= self.number_of_variables)
            .unwrap_or(false)
        {
            panic!("some variables are out of bound")
        }
        self.clauses.insert(clause);
        self
    }

    pub fn add_variables_to_sampling_set(
        mut self,
        variables: impl IntoIterator<Item = VarIdx>,
    ) -> Self {
        for variable in variables.into_iter() {
            if variable >= self.number_of_variables {
                panic!("variable is out of bound");
            }
            self.sampling_set.insert(variable);
        }
        self
    }

    pub fn contains(&self, clause: &CnfClause) -> bool {
        self.clauses.contains(clause)
    }

    pub fn number_of_variables(&self) -> usize {
        self.number_of_variables
    }

    pub fn number_of_clauses(&self) -> usize {
        self.clauses.len()
    }

    pub fn clauses(&self) -> impl Iterator<Item = &CnfClause> {
        self.clauses.iter()
    }

    pub fn clause(&self, idx: usize) -> Option<&CnfClause> {
        self.clauses.get_index(idx)
    }

    pub fn to_dimacs(&self) -> String {
        use std::iter::once;
        once(self.dimacs_header())
            .chain(once(self.dimacs_sampling_set()))
            .chain(self.clauses().map(|clause| clause.to_dimacs()))
            .join("\n")
    }

    pub fn dimacs_header(&self) -> String {
        format!(
            "p cnf {} {}",
            self.number_of_variables,
            self.number_of_clauses(),
        )
    }

    pub fn dimacs_sampling_set(&self) -> String {
        if !self.sampling_set.is_empty() {
            format!(
                "c ind {} 0",
                self.sampling_set
                    .iter()
                    .map(|var| (var + 1).to_string())
                    .join(" ")
            )
        } else {
            String::from("")
        }
    }
}

#[derive(Debug, Clone, Copy)]
pub struct ParsingError;

impl std::error::Error for ParsingError {}

impl std::fmt::Display for ParsingError {
    fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
        write!(f, "unable to parse")
    }
}
#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn build_simple_3sat() {
        use CnfVariable::{Neg, Pos};

        let formula = CnfFormula::with_variables_and_capacity(5, 3)
            .insert(CnfClause::new(vec![Pos(0), Neg(1), Pos(2)]))
            .insert(CnfClause::new(vec![Neg(2), Pos(3), Pos(4)]))
            .insert(CnfClause::new(vec![Pos(0), Pos(2), Pos(4)]));

        assert_eq!(formula.number_of_variables(), 5);
        assert_eq!(formula.number_of_clauses(), 3);
    }

    #[test]
    #[should_panic]
    fn panic_with_out_of_bound_variable() {
        CnfFormula::with_variables_and_capacity(5, 3)
            .insert(CnfClause::new(vec![CnfVariable::Pos(10)]));
    }

    #[test]
    fn dimacs() {
        use CnfVariable::{Neg, Pos};
        let dimacs = CnfFormula::with_variables_and_capacity(5, 3)
            .insert(CnfClause::new(vec![Pos(0), Neg(1), Pos(2)]))
            .insert(CnfClause::new(vec![Neg(2), Pos(3)]))
            .insert(CnfClause::new(vec![Pos(0), Neg(1), Pos(2), Neg(3), Pos(4)]))
            .to_dimacs();
        let expected = String::from("p cnf 5 3\n\n1 -2 3 0\n-3 4 0\n1 -2 3 -4 5 0");
        assert_eq!(dimacs, expected);
    }

    #[test]
    fn dimacs_with_sampling_set() {
        use CnfVariable::{Neg, Pos};
        let dimacs = CnfFormula::with_variables_and_capacity(5, 3)
            .insert(CnfClause::new(vec![Pos(0), Neg(1), Pos(2)]))
            .insert(CnfClause::new(vec![Neg(2), Pos(3)]))
            .insert(CnfClause::new(vec![Pos(0), Neg(1), Pos(2), Neg(3), Pos(4)]))
            .add_variables_to_sampling_set(0..=1)
            .to_dimacs();
        let expected = String::from("p cnf 5 3\nc ind 1 2 0\n1 -2 3 0\n-3 4 0\n1 -2 3 -4 5 0");
        assert_eq!(dimacs, expected);
    }
}
