use super::ParsingError;
use crate::VarIdx;
use CnfVariable::{Neg, Pos};
use serde::{Serialize, Deserialize};

use std::cmp::{Ord, Ordering, PartialOrd};

#[derive(Debug, PartialEq, Eq, Clone, Copy, Hash, Serialize, Deserialize)]
pub enum CnfVariable {
    Pos(VarIdx),
    Neg(VarIdx),
}

impl PartialOrd for CnfVariable {
    fn partial_cmp(&self, other: &Self) -> Option<Ordering> {
        self.index().partial_cmp(&other.index())
    }
}

impl Ord for CnfVariable {
    fn cmp(&self, other: &Self) -> Ordering {
        self.index().cmp(&other.index())
    }
}

impl CnfVariable {
    pub fn index(self) -> VarIdx {
        match self {
            Pos(index) | Neg(index) => index,
        }
    }

    pub fn is_pos(self) -> bool {
        match self {
            Pos(_) => true,
            _ => false,
        }
    }

    pub fn is_neg(self) -> bool {
        !self.is_pos()
    }

    pub fn to_dimacs(self) -> String {
        match self {
            Pos(index) => format!("{}", index + 1),
            Neg(index) => format!("-{}", index + 1),
        }
    }

    pub fn flip(self) -> Self {
        match self {
            Pos(idx) => Neg(idx),
            Neg(idx) => Pos(idx),
        }
    }

    pub fn from_str(var: &str) -> Result<Self, ParsingError> {
        if var.starts_with("-") {
            var.get(1..).ok_or(ParsingError).and_then(|inner_var| {
                inner_var
                    .parse()
                    .map(|v: usize| Neg(v - 1))
                    .map_err(|_| ParsingError)
            })
        } else {
            var.parse()
                .map(|v: usize| Pos(v - 1))
                .map_err(|_| ParsingError)
        }
    }
}

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn dimacs_positive() {
        let variable = Pos(5);
        assert_eq!(variable.to_dimacs(), String::from("6"));
    }

    #[test]
    fn dimacs_negative() {
        let variable = Neg(5);
        assert_eq!(variable.to_dimacs(), String::from("-6"));
    }
}
