// Miniscript
// Written in 2019 by
//     Sanket K <sanket1729>
//
// To the extent possible under law, the author(s) have dedicated all
// copyright and related and neighboring rights to this software to
// the public domain worldwide. This software is distributed without
// any warranty.
//
// You should have received a copy of the CC0 Public Domain Dedication
// along with this software.
// If not, see <http://creativecommons.org/publicdomain/zero/1.0/>.
//

//! # Miniscript Tests generated from dgpv's dot files
//!
//! Testcases from Miniscript dgpv's alloy spec.
//!
#[cfg(test)]
mod tests {
    use std::fmt;
    use {miniscript::types, Miniscript, Segwitv0};

    struct TestType(types::Type);

    impl fmt::Display for TestType {
        fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
            f.write_str(match self.0.corr.base {
                types::Base::B => "B",
                types::Base::K => "K",
                types::Base::V => "V",
                types::Base::W => "W",
            })?;
            f.write_str(match self.0.corr.input {
                types::Input::Zero => "z",
                types::Input::One => "o",
                types::Input::OneNonZero => "on",
                types::Input::Any => "",
                types::Input::AnyNonZero => "n",
            })?;
            if self.0.corr.dissatisfiable {
                fmt::Write::write_char(f, 'd')?;
            }
            if self.0.corr.unit {
                fmt::Write::write_char(f, 'u')?;
            }
            f.write_str(match self.0.mall.dissat {
                types::Dissat::None => "f",
                types::Dissat::Unique => "e",
                types::Dissat::Unknown => "",
            })?;
            if self.0.mall.safe {
                fmt::Write::write_char(f, 's')?;
            }
            if self.0.mall.non_malleable {
                fmt::Write::write_char(f, 'm')?;
            }
            Ok(())
        }
    }

    fn ms_test(ms: &str, exp_type_attrs: &str) {
        // Parse the miniscript
        let ms_node = Miniscript::<String, Segwitv0>::from_str_insane(ms).unwrap();
        let actual_types = format!("{}", TestType(ms_node.ty));
        let mut actual_types = actual_types.chars().collect::<Vec<_>>();
        actual_types.sort_by(|a, b| b.cmp(a));
        let mut expected_types = exp_type_attrs.chars().collect::<Vec<_>>();
        expected_types.sort_by(|a, b| b.cmp(a));

        assert_eq!(expected_types, actual_types);
    }

    // Check miniscripts that are incorrectly typed
    fn invalid_ms(ms: &str) {
        assert!(Miniscript::<String, Segwitv0>::from_str_insane(ms).is_err());
    }

    #[test]
    #[cfg_attr(feature="cargo-fmt", rustfmt_skip)]
    fn invalid_tests_from_alloy() {
        invalid_ms("or_b(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))");
        invalid_ms("or_b(s:pk_h(A),after(500000001))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(j:multi(2,A,B,C),pk_k(D))");
        invalid_ms("thresh(2,multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("dc:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_b(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_d(s:or_c(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("c:thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(s:or_d(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:or_c(after(500000001),s:after(500000001))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("or_b(j:multi(2,A,B,C),after(500000001))");
        invalid_ms("thresh(2,after(1),after(1),s:pk_k(A))");
        invalid_ms("or_d(pk_k(A),or_c(after(500000001),after(500000001)))");
        invalid_ms("ds:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("c:thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_c(multi(2,A,B,C),multi(2,D,E,F)),multi(2,G,I,J))");
        invalid_ms("j:or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(multi(2,A,B,C),s:multi(2,D,E,F))");
        invalid_ms("thresh(2,v:after(500000001),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(multi(2,A,B,C),after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),after(1))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_d(or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("csn:pk_k(A)");
        invalid_ms("or_d(or_d(after(500000001),multi(2,A,B,C)),j:multi(2,D,E,F))");
        invalid_ms("ns:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),or_c(after(500000001),multi(2,D,E,F)))");
        invalid_ms("or_b(pk_k(A),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("thresh(2,after(500000001),pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),after(500000001),or_i(or_b(c:after(500000001),pk_h(A)),0))");
        invalid_ms("or_d(or_c(multi(2,A,B,C),multi(2,D,E,F)),multi(2,G,I,J))");
        invalid_ms("or_d(or_c(after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:or_d(after(500000001),pk_h(A))");
        invalid_ms("or_b(s:or_d(after(500000001),after(500000001)),multi(2,A,B,C))");
        invalid_ms("or_b(and_v(after(500000001),1),pk_h(A))");
        invalid_ms("cda:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),svs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_d(after(1),pk_k(A))");
        invalid_ms("and_b(after(500000001),or_c(after(500000001),pk_h(A)))");
        invalid_ms("thresh(2,c:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(a:or_b(after(500000001),pk_k(A)),after(500000001))");
        invalid_ms("ds:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(1),multi(2,A,B,C))");
        invalid_ms("or_b(or_d(after(1),after(1)),multi(2,A,B,C))");
        invalid_ms("or_b(after(500000001),c:pk_k(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("d:and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("thresh(2,n:pk_k(A),after(500000001),pk_k(B))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:thresh(2,after(500000001),after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(pk_k(A),or_b(pk_k(B),after(500000001)))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),s:or_b(after(1),multi(2,B,C,D)))");
        invalid_ms("thresh(2,or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(vs:or_d(after(1),multi(2,A,B,C)),0))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("andor(pk_h(A),multi(2,B,C,D),after(500000001))");
        invalid_ms("or_d(s:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C))");
        invalid_ms("or_d(c:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(pk_k(A),multi(2,B,C,D))");
        invalid_ms("or_b(pk_k(A),s:or_b(multi(2,B,C,D),after(500000001)))");
        invalid_ms("and_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(after(1),after(1)))");
        invalid_ms("thresh(2,after(500000001),c:pk_k(A),pk_h(B))");
        invalid_ms("d:thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("cvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_h(A),vs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(B)))");
        invalid_ms("or_b(pk_k(A),thresh(2,after(500000001),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("jvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("and_b(s:or_c(after(500000001),after(500000001)),s:or_d(after(500000001),pk_k(A)))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),j:multi(2,G,I,J))");
        invalid_ms("jcs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(1))");
        invalid_ms("dvs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:thresh(2,multi(2,A,B,C),multi(2,D,E,F),after(1))");
        invalid_ms("andor(cs:after(500000001),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(pk_h(A),after(1))");
        invalid_ms("or_b(or_b(pk_h(A),after(500000001)),pk_k(B))");
        invalid_ms("or_d(after(500000001),and_v(multi(2,A,B,C),after(500000001)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),n:pk_h(A))");
        invalid_ms("c:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_d(after(500000001),pk_k(A)),pk_k(B))");
        invalid_ms("or_b(after(500000001),or_i(after(500000001),0))");
        invalid_ms("or_d(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),j:pk_h(D))");
        invalid_ms("n:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),multi(2,B,C,D))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),va:pk_k(A))");
        invalid_ms("or_b(s:thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_h(A))");
        invalid_ms("c:or_i(0,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_b(multi(2,A,B,C),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0))");
        invalid_ms("or_i(0,or_c(after(1),pk_h(A)))");
        invalid_ms("or_b(sn:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ca:after(1)");
        invalid_ms("and_b(pk_h(A),a:pk_k(B))");
        invalid_ms("or_b(after(500000001),c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("cs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("thresh(2,pk_h(A),pk_k(B),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),s:or_b(pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_i(0,a:or_c(n:or_i(pk_k(A),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("jc:or_b(multi(2,A,B,C),after(1))");
        invalid_ms("j:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ds:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(pk_k(A),multi(2,B,C,D))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:pk_k(A))");
        invalid_ms("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),pk_k(B))");
        invalid_ms("and_v(pk_k(A),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0))");
        invalid_ms("or_d(after(1),ns:after(500000001))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_b(pk_h(A),after(500000001)))");
        invalid_ms("or_d(after(1),or_c(after(1),after(1)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_b(pk_k(A),thresh(2,pk_k(B),after(500000001),after(500000001)))");
        invalid_ms("or_b(multi(2,A,B,C),v:after(500000001))");
        invalid_ms("j:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_b(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(pk_k(A),pk_k(B))");
        invalid_ms("thresh(2,pk_k(A),pk_h(B),j:multi(2,C,D,E))");
        invalid_ms("or_d(a:pk_k(A),a:pk_h(B))");
        invalid_ms("thresh(2,after(500000001),after(500000001),j:multi(2,A,B,C))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("thresh(2,s:after(1),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cvs:or_c(after(500000001),pk_k(A))");
        invalid_ms("or_d(or_c(after(1),pk_h(A)),after(1))");
        invalid_ms("or_b(multi(2,A,B,C),or_d(after(1),after(500000001)))");
        invalid_ms("or_b(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:after(500000001),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:andor(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("and_b(after(500000001),and_b(pk_k(A),after(500000001)))");
        invalid_ms("and_v(multi(2,A,B,C),after(500000001))");
        invalid_ms("cvs:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_d(after(500000001),after(500000001)),pk_h(A))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(pk_k(A),pk_k(B)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C))");
        invalid_ms("or_b(a:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),svs:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1)))");
        invalid_ms("or_d(or_d(after(1),after(1)),multi(2,A,B,C))");
        invalid_ms("or_d(thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),pk_k(B))");
        invalid_ms("or_d(a:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))");
        invalid_ms("or_d(multi(2,A,B,C),vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:or_c(multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_b(or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(B))");
        invalid_ms("or_b(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_k(A),pk_h(B),after(1))");
        invalid_ms("jd:thresh(2,after(1),pk_h(A),pk_h(B))");
        invalid_ms("or_i(0,or_c(after(1),j:multi(2,A,B,C)))");
        invalid_ms("or_d(s:thresh(2,after(500000001),after(500000001),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_b(or_i(0,multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_k(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(a:after(1),and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(after(1),after(1),after(1))))");
        invalid_ms("n:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),ca:pk_h(A))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D)))");
        invalid_ms("na:after(500000001)");
        invalid_ms("j:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),after(500000001))");
        invalid_ms("or_b(s:or_b(after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(c:or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("thresh(2,or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1),pk_k(A))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(pk_k(A),thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),after(1))");
        invalid_ms("c:or_d(c:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_k(D),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_d(or_c(or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:andor(after(1),after(1),0)),after(1))");
        invalid_ms("cvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_h(A))");
        invalid_ms("thresh(2,after(1),vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("and_b(after(500000001),and_v(after(500000001),after(500000001)))");
        invalid_ms("thresh(2,pk_h(A),pk_k(B),multi(2,C,D,E))");
        invalid_ms("andor(c:after(1),multi(2,A,B,C),pk_h(D))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D))");
        invalid_ms("c:or_d(or_d(after(500000001),after(500000001)),pk_k(A))");
        invalid_ms("or_d(svsvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),multi(2,A,B,C))");
        invalid_ms("thresh(2,pk_h(A),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),1)");
        invalid_ms("or_d(s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_h(A),multi(2,B,C,D),after(1))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),va:pk_h(A))");
        invalid_ms("or_b(after(500000001),j:multi(2,A,B,C))");
        invalid_ms("or_d(or_c(after(1),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,s:or_d(after(500000001),after(500000001)))");
        invalid_ms("or_d(pk_h(A),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)))");
        invalid_ms("or_b(sc:thresh(2,after(1),pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(B))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("ds:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:multi(2,A,B,C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(after(1),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("js:after(1)");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_h(A))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),multi(2,B,C,D))");
        invalid_ms("or_d(vsc:or_d(after(1),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ns:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("ns:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_d(s:or_c(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(a:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,c:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_d(vsvs:or_d(after(1),after(1)),pk_k(A))");
        invalid_ms("js:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_b(j:multi(2,A,B,C),j:multi(2,D,E,F))");
        invalid_ms("or_d(c:pk_h(A),c:after(500000001))");
        invalid_ms("or_d(nv:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(1))");
        invalid_ms("or_d(or_c(after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_b(after(500000001),multi(2,A,B,C)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),after(1))");
        invalid_ms("or_b(pk_h(A),or_c(after(1),after(1)))");
        invalid_ms("cs:or_b(multi(2,A,B,C),c:after(500000001))");
        invalid_ms("c:or_d(or_c(or_c(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),after(500000001)),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("csc:or_d(a:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("dc:pk_h(A)");
        invalid_ms("and_b(s:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ns:or_c(after(1),after(1))");
        invalid_ms("or_d(after(1),a:pk_h(A))");
        invalid_ms("c:or_c(after(1),pk_h(A))");
        invalid_ms("thresh(2,after(1),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),or_b(multi(2,B,C,D),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(pk_k(A),or_d(after(500000001),pk_k(B)))");
        invalid_ms("or_b(pk_k(A),sj:multi(2,B,C,D))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(multi(2,D,E,F),after(1)))");
        invalid_ms("thresh(2,v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(cs:or_c(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),s:or_b(after(500000001),multi(2,B,C,D)))");
        invalid_ms("or_d(after(1),and_v(after(1),1))");
        invalid_ms("or_b(after(500000001),n:pk_h(A))");
        invalid_ms("or_d(s:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_b(pk_k(A),s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(B)))");
        invalid_ms("thresh(2,after(500000001),pk_h(A),s:after(500000001))");
        invalid_ms("or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),or_c(after(1),multi(2,G,I,J)))");
        invalid_ms("ja:thresh(2,after(500000001),pk_h(A),after(500000001))");
        invalid_ms("or_d(pk_h(A),multi(2,B,C,D))");
        invalid_ms("thresh(2,c:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))");
        invalid_ms("cvs:or_d(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_d(j:pk_h(A),pk_k(B))");
        invalid_ms("or_d(or_c(after(1),after(1)),pk_k(A))");
        invalid_ms("cdvs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("or_b(va:multi(2,A,B,C),pk_k(D))");
        invalid_ms("and_b(a:pk_k(A),v:pk_k(B))");
        invalid_ms("or_b(multi(2,A,B,C),thresh(2,pk_h(D),multi(2,E,F,G),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),pk_k(A))");
        invalid_ms("thresh(2,after(500000001),d:after(500000001),after(500000001))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("c:or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_k(A))");
        invalid_ms("c:or_c(after(1),pk_k(A))");
        invalid_ms("thresh(2,s:multi(2,A,B,C),after(500000001),pk_h(D))");
        invalid_ms("c:thresh(2,pk_h(A),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_c(after(1),after(1)),pk_k(A))");
        invalid_ms("or_b(pk_h(A),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))");
        invalid_ms("or_d(sj:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(1),s:pk_k(A),after(1))");
        invalid_ms("da:pk_k(A)");
        invalid_ms("c:or_d(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:or_d(after(1),after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),pk_k(A))");
        invalid_ms("or_b(or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),multi(2,A,B,C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(svs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_b(multi(2,A,B,C),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_d(c:pk_k(A),after(500000001))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),after(500000001))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(cs:after(1),multi(2,A,B,C))");
        invalid_ms("thresh(2,c:thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_i(0,s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_i(0,s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_b(pk_k(A),s:thresh(2,after(500000001),after(500000001),multi(2,B,C,D)))");
        invalid_ms("c:or_b(a:multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(pk_h(A),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(B)))");
        invalid_ms("or_b(pk_h(A),and_v(multi(2,B,C,D),pk_k(E)))");
        invalid_ms("or_d(s:or_d(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("and_b(multi(2,A,B,C),pk_k(D))");
        invalid_ms("thresh(2,s:after(1),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("thresh(2,after(1),or_d(after(1),after(1)),multi(2,A,B,C))");
        invalid_ms("c:or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(500000001),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(c:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("or_d(multi(2,A,B,C),v:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D)))");
        invalid_ms("nsc:pk_h(A)");
        invalid_ms("thresh(2,s:pk_k(A),multi(2,B,C,D),after(500000001))");
        invalid_ms("cs:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),ca:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_c(or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(c:or_d(after(500000001),after(500000001)),pk_k(A))");
        invalid_ms("or_b(thresh(2,after(1),multi(2,A,B,C),after(1)),multi(2,D,E,F))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),pk_k(A))");
        invalid_ms("d:or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,B,C,D))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),s:multi(2,A,B,C))");
        invalid_ms("or_b(pk_k(A),s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)))");
        invalid_ms("ds:or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_d(or_c(or_c(after(1),pk_h(A)),pk_k(B)),after(1))");
        invalid_ms("thresh(2,pk_k(A),pk_k(B),c:after(500000001))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(500000001),multi(2,D,E,F))");
        invalid_ms("or_i(0,s:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,pk_h(A),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),pk_k(B))");
        invalid_ms("or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_b(multi(2,A,B,C),a:pk_k(D))");
        invalid_ms("and_v(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),1)");
        invalid_ms("jd:after(500000001)");
        invalid_ms("or_b(pk_h(A),after(500000001))");
        invalid_ms("thresh(2,pk_k(A),s:after(500000001),multi(2,B,C,D))");
        invalid_ms("or_d(pk_k(A),pk_k(B))");
        invalid_ms("or_d(or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(after(500000001),pk_k(A))");
        invalid_ms("and_v(c:after(1),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(after(1),j:after(1)))");
        invalid_ms("or_b(or_c(multi(2,A,B,C),or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("csc:after(1)");
        invalid_ms("d:thresh(2,multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,or_b(after(1),pk_k(A)),pk_k(B),after(1))");
        invalid_ms("or_d(after(1),cs:pk_k(A))");
        invalid_ms("cvs:thresh(2,after(1),pk_k(A),after(1))");
        invalid_ms("or_d(after(500000001),v:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(pk_k(A),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_b(jc:pk_h(A),multi(2,B,C,D))");
        invalid_ms("or_i(0,or_c(after(1),pk_k(A)))");
        invalid_ms("or_b(or_b(after(1),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("ns:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(multi(2,A,B,C),multi(2,D,E,F)))");
        invalid_ms("or_b(a:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("dvsvsvs:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(after(1),pk_k(A)))");
        invalid_ms("ds:or_b(after(500000001),multi(2,A,B,C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_b(vs:or_c(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(after(500000001),j:multi(2,D,E,F)))");
        invalid_ms("cs:thresh(2,after(1),after(1),pk_h(A))");
        invalid_ms("d:or_c(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("dac:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("thresh(2,after(1),pk_k(A),pk_k(B))");
        invalid_ms("or_d(c:or_d(after(1),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:or_c(after(1),multi(2,A,B,C))");
        invalid_ms("or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("and_b(vs:or_d(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("c:thresh(2,pk_h(A),after(1),vs:or_d(after(1),after(1)))");
        invalid_ms("js:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("c:or_d(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)))");
        invalid_ms("or_b(a:pk_h(A),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(and_b(pk_k(A),after(1)),pk_h(B))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:pk_h(D))");
        invalid_ms("thresh(2,after(1),after(1),c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("c:or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(500000001))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),s:or_c(thresh(2,after(1),pk_k(B),after(1)),after(1)))");
        invalid_ms("thresh(2,pk_h(A),after(1),pk_h(B))");
        invalid_ms("or_d(c:or_c(after(500000001),pk_k(A)),multi(2,B,C,D))");
        invalid_ms("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),cjaj:after(500000001)),pk_k(A))");
        invalid_ms("or_d(or_c(after(500000001),after(500000001)),a:after(500000001))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,v:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:after(1))");
        invalid_ms("or_b(thresh(2,multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(c:or_d(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D),pk_h(E))");
        invalid_ms("thresh(2,a:after(500000001),pk_k(A),after(500000001))");
        invalid_ms("or_d(pk_h(A),thresh(2,after(500000001),multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_c(after(1),after(1)),after(500000001))");
        invalid_ms("js:or_c(after(1),pk_h(A))");
        invalid_ms("c:thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_d(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)))");
        invalid_ms("or_d(vcda:and_v(after(500000001),1),pk_h(A))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:pk_h(A))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_b(pk_k(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),svs:or_d(after(500000001),after(500000001)),multi(2,A,B,C))");
        invalid_ms("d:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(pk_k(A),thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("cvs:or_b(after(500000001),pk_h(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(500000001),pk_h(D))");
        invalid_ms("or_b(vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:thresh(2,after(500000001),pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("and_b(pk_k(A),c:and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("js:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cvsca:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_b(or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:pk_h(A))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("thresh(2,after(1),or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_d(pk_k(A),or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(j:multi(2,A,B,C),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),c:after(1))");
        invalid_ms("thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),cs:after(500000001))");
        invalid_ms("c:and_b(or_c(after(1),after(1)),after(1))");
        invalid_ms("or_d(after(500000001),c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:or_b(after(500000001),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(multi(2,A,B,C),pk_k(D)))");
        invalid_ms("csc:or_c(after(1),pk_k(A))");
        invalid_ms("or_d(pk_k(A),or_b(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:or_c(after(1),multi(2,A,B,C))");
        invalid_ms("or_i(0,s:or_d(after(500000001),multi(2,A,B,C)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vsvs:thresh(2,after(500000001),multi(2,A,B,C),after(500000001)))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(after(1),pk_k(A)))");
        invalid_ms("dj:multi(2,A,B,C)");
        invalid_ms("or_d(pk_h(A),or_b(pk_h(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,a:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("thresh(2,v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_b(pk_k(A),s:or_b(pk_h(B),after(500000001)))");
        invalid_ms("and_v(multi(2,A,B,C),1)");
        invalid_ms("js:or_c(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(s:or_b(s:or_c(after(500000001),multi(2,A,B,C)),pk_k(D)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("cv:after(500000001)");
        invalid_ms("or_d(pk_h(A),a:pk_k(B))");
        invalid_ms("or_d(s:or_c(after(500000001),pk_k(A)),pk_k(B))");
        invalid_ms("thresh(2,or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ds:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:after(500000001))");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("csvs:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_b(svs:thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(B))");
        invalid_ms("or_b(sn:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),c:pk_h(A))");
        invalid_ms("c:or_b(or_c(after(1),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_b(after(1),a:multi(2,A,B,C))");
        invalid_ms("or_b(or_d(or_d(after(1),after(1)),after(1)),multi(2,A,B,C))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(svs:or_c(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("d:or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:pk_h(A)),sj:pk_k(B))");
        invalid_ms("or_b(multi(2,A,B,C),after(1))");
        invalid_ms("or_d(pk_k(A),thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("cs:pk_h(A)");
        invalid_ms("c:or_d(after(500000001),pk_h(A))");
        invalid_ms("ds:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),s:or_b(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("cs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("dj:pk_h(A)");
        invalid_ms("or_b(or_c(after(1),after(1)),pk_h(A))");
        invalid_ms("or_d(s:or_b(after(500000001),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(dc:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),after(500000001))");
        invalid_ms("n:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(or_b(multi(2,A,B,C),after(500000001)),a:after(500000001))");
        invalid_ms("thresh(2,v:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jn:pk_k(A)");
        invalid_ms("thresh(2,pk_h(A),or_c(after(500000001),multi(2,B,C,D)),after(500000001))");
        invalid_ms("and_b(after(1),after(1))");
        invalid_ms("thresh(2,after(500000001),pk_k(A),or_b(after(500000001),pk_k(B)))");
        invalid_ms("thresh(2,pk_k(A),pk_h(B),vs:or_b(pk_k(C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jv:thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("dn:after(500000001)");
        invalid_ms("or_b(a:after(500000001),pk_h(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vd:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(D))");
        invalid_ms("and_v(pk_h(A),after(500000001))");
        invalid_ms("or_d(pk_k(A),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("n:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(500000001))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sj:pk_k(A))");
        invalid_ms("thresh(2,d:multi(2,A,B,C),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_c(after(1),multi(2,D,E,F)))");
        invalid_ms("or_b(s:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cnvs:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(500000001),a:after(1))");
        invalid_ms("thresh(2,after(500000001),ds:after(500000001),after(500000001))");
        invalid_ms("n:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:after(500000001))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:pk_k(D))");
        invalid_ms("and_b(after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("dvs:multi(2,A,B,C)");
        invalid_ms("nvs:or_d(after(500000001),multi(2,A,B,C))");
        invalid_ms("j:or_c(after(1),after(1))");
        invalid_ms("or_b(n:pk_h(A),after(1))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("thresh(2,after(1),pk_h(A),c:pk_h(B))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_k(A),j:multi(2,B,C,D),pk_h(E))");
        invalid_ms("or_d(pk_k(A),or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_d(after(500000001),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_d(av:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_d(after(1),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),av:multi(2,A,B,C))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(B))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("cs:thresh(2,pk_k(A),pk_k(B),after(500000001))");
        invalid_ms("or_b(pk_k(A),a:after(500000001))");
        invalid_ms("thresh(2,pk_k(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B))");
        invalid_ms("or_i(0,or_c(c:after(500000001),multi(2,A,B,C)))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_h(A),thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_b(multi(2,A,B,C),or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,j:after(500000001),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jcjnv:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ns:pk_k(A)");
        invalid_ms("thresh(2,s:pk_k(A),multi(2,B,C,D),s:multi(2,E,F,G))");
        invalid_ms("cs:or_c(after(1),multi(2,A,B,C))");
        invalid_ms("thresh(2,pk_h(A),pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F),multi(2,G,I,J))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:pk_k(D))");
        invalid_ms("or_d(after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("n:or_c(after(500000001),pk_k(A))");
        invalid_ms("n:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("d:pk_h(A)");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("n:or_c(after(1),after(500000001))");
        invalid_ms("or_b(pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("jn:and_v(andor(after(500000001),after(500000001),pk_k(A)),pk_k(B))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),thresh(2,multi(2,A,B,C),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(s:multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("cs:pk_k(A)");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),pk_h(D))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),thresh(2,after(500000001),pk_k(A),after(500000001)),after(500000001))");
        invalid_ms("or_b(svs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("cvacsc:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C))");
        invalid_ms("or_b(after(1),j:pk_h(A))");
        invalid_ms("or_d(after(500000001),andor(after(500000001),after(500000001),after(500000001)))");
        invalid_ms("or_b(vs:or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),after(500000001)),pk_h(A))");
        invalid_ms("or_i(0,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_d(v:pk_h(A),cvnsv:multi(2,B,C,D))");
        invalid_ms("c:thresh(2,after(500000001),pk_k(A),after(500000001))");
        invalid_ms("or_b(after(500000001),pk_h(A))");
        invalid_ms("or_d(s:after(500000001),multi(2,A,B,C))");
        invalid_ms("ds:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_k(A),pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ca:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:multi(2,A,B,C))");
        invalid_ms("or_d(pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("djc:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("c:or_c(pk_k(A),or_d(after(500000001),multi(2,B,C,D)))");
        invalid_ms("or_d(or_c(after(1),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(pk_k(A),pk_h(B))");
        invalid_ms("j:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(multi(2,A,B,C),s:thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(s:or_c(after(1),a:after(1)),a:after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),c:pk_k(A))");
        invalid_ms("or_b(multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("or_d(jd:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(or_c(or_c(after(1),after(1)),after(1)),after(1))");
        invalid_ms("ca:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("thresh(2,after(500000001),c:after(500000001),multi(2,A,B,C))");
        invalid_ms("or_b(s:or_d(after(1),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),v:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(B)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(pk_k(A),or_c(or_c(after(500000001),multi(2,B,C,D)),multi(2,E,F,G)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0))");
        invalid_ms("or_b(vs:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D))");
        invalid_ms("thresh(2,after(500000001),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001)),pk_k(A))");
        invalid_ms("or_d(a:or_c(after(500000001),after(500000001)),a:after(500000001))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_h(A))");
        invalid_ms("andor(multi(2,A,B,C),after(1),pk_k(D))");
        invalid_ms("or_d(s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_i(or_c(c:after(500000001),or_b(after(500000001),after(500000001))),0),after(500000001))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:multi(2,A,B,C))");
        invalid_ms("or_b(pk_k(A),pk_h(B))");
        invalid_ms("or_i(0,s:or_c(after(500000001),after(500000001)))");
        invalid_ms("djns:multi(2,A,B,C)");
        invalid_ms("j:or_c(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:pk_h(A))");
        invalid_ms("cs:or_d(after(1),after(500000001))");
        invalid_ms("andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_k(A),multi(2,B,C,D),svs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,E,F,G)))");
        invalid_ms("or_d(c:after(1),a:pk_k(A))");
        invalid_ms("or_d(and_v(after(1),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_d(c:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(B)))");
        invalid_ms("thresh(2,pk_k(A),j:multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(after(1),pk_k(A)),pk_h(B))");
        invalid_ms("thresh(2,c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),thresh(2,pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(C))");
        invalid_ms("or_d(or_c(after(500000001),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_i(0,s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("thresh(2,pk_h(A),after(500000001),d:multi(2,B,C,D))");
        invalid_ms("j:thresh(2,after(1),pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_h(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B))");
        invalid_ms("n:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:pk_h(A))");
        invalid_ms("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),a:pk_h(A))");
        invalid_ms("cs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(c:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B))");
        invalid_ms("csvs:or_b(pk_h(A),after(500000001))");
        invalid_ms("jc:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),or_b(after(500000001),pk_k(B)))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(or_c(pk_h(A),vs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))),multi(2,E,F,G))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_i(0,s:or_c(after(1),after(1)))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(or_b(pk_k(A),after(1)),pk_k(B))");
        invalid_ms("c:or_c(after(1),pk_k(A))");
        invalid_ms("thresh(2,pk_h(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("or_d(after(1),s:or_d(after(1),multi(2,A,B,C)))");
        invalid_ms("csvs:or_d(after(500000001),pk_k(A))");
        invalid_ms("d:pk_k(A)");
        invalid_ms("or_d(or_c(after(1),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_i(0,or_c(or_c(after(500000001),after(500000001)),after(500000001)))");
        invalid_ms("cvs:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_b(s:thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)),multi(2,E,F,G))");
        invalid_ms("or_b(s:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("or_b(j:pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:after(1)))");
        invalid_ms("or_d(pk_k(A),s:or_d(after(1),after(1)))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(a:or_b(pk_h(A),after(500000001)),after(500000001))");
        invalid_ms("j:thresh(2,pk_k(A),multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("c:or_c(vs:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("andor(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("jvs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))))");
        invalid_ms("ds:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(pk_k(A),and_b(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(after(500000001),and_v(after(500000001),multi(2,A,B,C)))");
        invalid_ms("and_v(after(500000001),multi(2,A,B,C))");
        invalid_ms("thresh(2,s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),sv:or_i(or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),0))");
        invalid_ms("ca:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("d:thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:after(500000001))");
        invalid_ms("c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("ns:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),svsvs:or_d(after(500000001),multi(2,A,B,C)))");
        invalid_ms("or_b(multi(2,A,B,C),a:after(1))");
        invalid_ms("or_d(or_c(or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1)),after(500000001))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:or_c(after(500000001),after(500000001)))");
        invalid_ms("or_d(s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))");
        invalid_ms("or_i(0,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("cs:or_b(pk_h(A),after(500000001))");
        invalid_ms("c:after(500000001)");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_b(or_c(after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),pk_k(D),after(500000001))");
        invalid_ms("thresh(2,pk_k(A),pk_k(B),j:multi(2,C,D,E))");
        invalid_ms("thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B))");
        invalid_ms("or_d(pk_h(A),or_b(multi(2,B,C,D),after(500000001)))");
        invalid_ms("thresh(2,after(1),pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("da:multi(2,A,B,C)");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(1),after(1)))");
        invalid_ms("or_b(vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_h(A),or_c(after(500000001),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(pk_k(A),s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:or_c(c:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(after(500000001),j:after(500000001))");
        invalid_ms("or_b(s:or_d(after(1),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("nav:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_d(or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),multi(2,A,B,C))");
        invalid_ms("or_i(0,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("j:or_c(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_d(after(1),or_b(after(1),after(1)))");
        invalid_ms("nsvs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("cs:thresh(2,pk_h(A),multi(2,B,C,D),after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),pk_k(B))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,D,E,F))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:or_d(multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("and_b(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("cs:thresh(2,after(500000001),pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),or_i(multi(2,B,C,D),0))");
        invalid_ms("dsvs:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("js:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("cvj:after(500000001)");
        invalid_ms("or_d(after(500000001),n:or_d(after(500000001),after(500000001)))");
        invalid_ms("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),pk_k(A))");
        invalid_ms("or_d(d:andor(after(1),after(1),after(1)),vs:after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),j:multi(2,B,C,D))");
        invalid_ms("and_v(after(500000001),after(500000001))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,c:after(1),multi(2,A,B,C),after(1))");
        invalid_ms("d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jnj:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_h(A),pk_k(B))");
        invalid_ms("or_b(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(c:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,after(500000001),pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_c(c:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(vs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(cs:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vsc:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001)))");
        invalid_ms("or_d(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(1))");
        invalid_ms("thresh(2,a:after(1),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(500000001))");
        invalid_ms("thresh(2,s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D))");
        invalid_ms("d:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),pk_k(D))");
        invalid_ms("dj:multi(2,A,B,C)");
        invalid_ms("or_d(s:or_c(or_c(or_c(after(1),after(1)),after(1)),after(1)),after(1))");
        invalid_ms("or_b(s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("d:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),s:after(500000001))");
        invalid_ms("or_b(multi(2,A,B,C),or_c(after(1),multi(2,D,E,F)))");
        invalid_ms("j:or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(pk_k(A),pk_h(B))),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:or_d(pk_k(A),nv:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_c(after(500000001),after(500000001)),pk_h(A))");
        invalid_ms("c:thresh(2,or_d(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("c:or_c(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(a:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(1))");
        invalid_ms("or_d(s:or_b(multi(2,A,B,C),after(1)),multi(2,D,E,F))");
        invalid_ms("thresh(2,after(1),multi(2,A,B,C),c:pk_k(D))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001))");
        invalid_ms("da:or_c(avcv:pk_k(A),after(1))");
        invalid_ms("thresh(2,after(1),pk_h(A),s:multi(2,B,C,D))");
        invalid_ms("ca:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(1),pk_k(A),pk_h(B))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sv:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("dns:multi(2,A,B,C)");
        invalid_ms("csj:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(v:multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_b(after(1),cs:pk_h(A))");
        invalid_ms("cd:after(1)");
        invalid_ms("d:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("or_b(pk_k(A),s:or_c(after(500000001),pk_k(B)))");
        invalid_ms("d:andor(after(1),c:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),n:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(multi(2,A,B,C),vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("n:or_c(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_b(or_d(after(1),after(1)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(d:after(500000001),or_c(after(500000001),pk_k(A)))");
        invalid_ms("or_d(multi(2,A,B,C),vsvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("j:or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0)");
        invalid_ms("or_d(pk_k(A),d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)))");
        invalid_ms("j:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jd:pk_h(A)");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sc:or_c(after(500000001),after(500000001)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(after(500000001),pk_k(A)))");
        invalid_ms("j:or_c(after(1),or_c(after(1),s:multi(2,A,B,C)))");
        invalid_ms("js:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(after(500000001),after(500000001)))");
        invalid_ms("or_d(and_b(pk_k(A),a:pk_k(B)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:after(500000001))");
        invalid_ms("or_i(0,s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),pk_h(D))");
        invalid_ms("and_b(pk_k(A),a:pk_h(B))");
        invalid_ms("csc:or_c(after(500000001),after(500000001))");
        invalid_ms("or_i(0,and_v(after(1),or_c(after(1),after(1))))");
        invalid_ms("njv:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_d(a:after(1),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("ca:multi(2,A,B,C)");
        invalid_ms("or_d(s:or_c(after(500000001),after(500000001)),multi(2,A,B,C))");
        invalid_ms("or_b(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jn:after(1)");
        invalid_ms("thresh(2,j:multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_b(thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jvs:or_b(multi(2,A,B,C),after(500000001))");
        invalid_ms("or_b(multi(2,A,B,C),s:or_c(after(1),after(1)))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)))");
        invalid_ms("thresh(2,thresh(2,v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1)),c:after(1),after(1))");
        invalid_ms("and_v(s:thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),after(500000001))");
        invalid_ms("or_d(pk_k(A),pk_h(B))");
        invalid_ms("or_d(s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_d(after(500000001),after(500000001)),pk_k(A))");
        invalid_ms("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),pk_h(B))");
        invalid_ms("or_b(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),svscs:or_d(after(1),after(1)))");
        invalid_ms("and_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(after(500000001),multi(2,A,B,C))");
        invalid_ms("cs:or_d(after(1),multi(2,A,B,C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),pk_k(B))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_h(A),s:or_d(after(500000001),multi(2,B,C,D)))");
        invalid_ms("or_b(dcs:thresh(2,after(500000001),after(500000001),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("n:or_c(after(1),pk_h(A))");
        invalid_ms("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("or_d(vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ds:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(multi(2,A,B,C),vsvs:or_b(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),pk_k(A))");
        invalid_ms("d:or_i(thresh(2,multi(2,A,B,C),pk_k(D),dn:pk_h(E)),0)");
        invalid_ms("or_d(s:or_b(multi(2,A,B,C),after(500000001)),pk_h(D))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ns:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_h(A),multi(2,B,C,D),j:multi(2,E,F,G))");
        invalid_ms("thresh(2,dc:after(1),after(1),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),d:pk_h(B))");
        invalid_ms("and_v(and_b(multi(2,A,B,C),after(1)),after(1))");
        invalid_ms("or_d(after(500000001),or_d(after(1),and_v(after(1),and_b(after(1),after(1)))))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("andor(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_i(0,or_c(after(500000001),multi(2,A,B,C)))");
        invalid_ms("thresh(2,pk_k(A),v:after(500000001),pk_k(B))");
        invalid_ms("or_b(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),after(500000001))");
        invalid_ms("d:or_c(or_c(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(B)))");
        invalid_ms("and_v(after(1),1)");
        invalid_ms("or_d(pk_h(A),thresh(2,multi(2,B,C,D),multi(2,E,F,G),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(s:or_b(pk_h(A),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B))),pk_k(C))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:pk_h(A))");
        invalid_ms("or_d(and_b(pk_k(A),and_b(after(500000001),after(500000001))),pk_k(B))");
        invalid_ms("or_b(s:or_d(after(1),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("thresh(2,s:multi(2,A,B,C),after(500000001),pk_h(D))");
        invalid_ms("or_b(j:multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("jvs:or_d(after(1),multi(2,A,B,C))");
        invalid_ms("c:or_i(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),0)");
        invalid_ms("c:or_d(c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(pk_k(A),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cvsca:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("c:or_d(after(1),j:multi(2,A,B,C))");
        invalid_ms("or_b(a:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("n:or_c(s:after(1),after(1))");
        invalid_ms("or_d(c:or_c(after(1),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("cs:or_c(after(500000001),pk_k(A))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_b(and_b(pk_k(A),after(1)),after(1))");
        invalid_ms("or_d(or_c(after(1),pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("na:and_b(after(500000001),and_v(or_b(after(500000001),after(500000001)),after(500000001)))");
        invalid_ms("js:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:thresh(2,after(500000001),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:after(1))");
        invalid_ms("cvs:or_d(after(1),multi(2,A,B,C))");
        invalid_ms("c:or_c(after(1),multi(2,A,B,C))");
        invalid_ms("d:or_b(s:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(after(1),c:pk_k(A))");
        invalid_ms("or_b(after(500000001),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("nv:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_b(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_b(thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),pk_k(B)),multi(2,C,D,E))");
        invalid_ms("js:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(a:after(1),thresh(2,pk_h(A),after(1),ca:after(1)))");
        invalid_ms("or_d(c:or_c(after(500000001),multi(2,A,B,C)),after(500000001))");
        invalid_ms("d:multi(2,A,B,C)");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_k(D),j:pk_k(E))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001))");
        invalid_ms("or_d(pk_k(A),s:or_b(multi(2,B,C,D),after(500000001)))");
        invalid_ms("ns:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))");
        invalid_ms("cs:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,v:pk_h(A),multi(2,B,C,D),multi(2,E,F,G))");
        invalid_ms("ns:or_b(multi(2,A,B,C),after(500000001))");
        invalid_ms("or_d(s:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D))");
        invalid_ms("j:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_b(or_b(after(1),pk_k(A)),pk_k(B))");
        invalid_ms("c:or_c(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_b(thresh(2,after(500000001),pk_h(A),after(500000001)),pk_k(B))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_d(after(1),pk_h(A))");
        invalid_ms("thresh(2,v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),pk_k(D))");
        invalid_ms("c:or_d(after(1),pk_h(A))");
        invalid_ms("cs:or_c(after(1),pk_h(A))");
        invalid_ms("andor(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(or_i(0,or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_h(D))),j:multi(2,E,F,G))");
        invalid_ms("jd:pk_k(A)");
        invalid_ms("or_b(cd:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(aj:pk_k(A),multi(2,B,C,D))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(500000001),multi(2,A,B,C))");
        invalid_ms("or_i(0,s:or_d(after(500000001),pk_k(A)))");
        invalid_ms("thresh(2,pk_k(A),multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),pk_h(D))");
        invalid_ms("or_d(or_b(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))),pk_k(G))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(after(1),0))");
        invalid_ms("da:pk_k(A)");
        invalid_ms("or_b(multi(2,A,B,C),and_v(after(1),after(1)))");
        invalid_ms("thresh(2,s:after(500000001),multi(2,A,B,C),pk_h(D))");
        invalid_ms("thresh(2,after(500000001),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_d(vs:or_d(after(1),after(1)),after(1))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(c:or_b(v:or_d(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001)),pk_h(B))");
        invalid_ms("and_b(or_c(after(500000001),after(500000001)),pk_k(A))");
        invalid_ms("and_b(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),ad:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_d(pk_k(A),s:or_c(after(500000001),pk_k(B)))");
        invalid_ms("or_b(s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_h(A),pk_h(B))");
        invalid_ms("or_b(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:multi(2,A,B,C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_h(A),s:or_b(multi(2,B,C,D),after(500000001)))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),after(500000001))");
        invalid_ms("ds:pk_h(A)");
        invalid_ms("dns:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(pk_h(A),or_d(after(500000001),multi(2,B,C,D))))");
        invalid_ms("or_d(pk_h(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:after(1),multi(2,A,B,C))");
        invalid_ms("js:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:pk_h(A))");
        invalid_ms("or_d(c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),after(500000001))");
        invalid_ms("or_b(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)))");
        invalid_ms("or_b(pk_k(A),multi(2,B,C,D))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),a:after(1))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("or_d(v:multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("thresh(2,pk_h(A),j:multi(2,B,C,D),pk_k(E))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_b(pk_h(A),multi(2,B,C,D))");
        invalid_ms("thresh(2,pk_h(A),vs:thresh(2,after(1),after(1),multi(2,B,C,D)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(and_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_h(A),pk_h(B))");
        invalid_ms("or_d(after(500000001),or_b(and_v(cv:after(500000001),after(500000001)),after(500000001)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_d(or_b(after(1),pk_h(A)),pk_k(B))");
        invalid_ms("or_d(a:after(500000001),after(1))");
        invalid_ms("or_d(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(B)))");
        invalid_ms("jd:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(s:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("ds:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(or_c(c:after(500000001),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),ac:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(thresh(2,multi(2,A,B,C),after(1),after(1)),multi(2,D,E,F))");
        invalid_ms("or_b(s:thresh(2,pk_k(A),after(1),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(a:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),multi(2,B,C,D))");
        invalid_ms("or_d(pk_h(A),s:or_b(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:or_c(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_d(after(1),after(1)),pk_h(A))");
        invalid_ms("ds:or_c(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(after(500000001),caja:and_v(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("ds:thresh(2,pk_k(A),pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(pk_h(A),after(500000001))");
        invalid_ms("j:thresh(2,after(500000001),multi(2,A,B,C),after(500000001))");
        invalid_ms("or_d(vs:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_b(av:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,or_c(after(1),after(1)),after(1),multi(2,A,B,C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("andor(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("and_v(and_b(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("thresh(2,pk_k(A),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)),multi(2,E,F,G))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(500000001),pk_h(A))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("and_v(or_d(after(1),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("d:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jv:after(500000001)");
        invalid_ms("or_i(0,pk_h(A))");
        invalid_ms("or_b(multi(2,A,B,C),vs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)))");
        invalid_ms("ns:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),pk_h(B))");
        invalid_ms("c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(pk_h(A),pk_k(B))");
        invalid_ms("or_d(s:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("js:thresh(2,after(1),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),or_c(after(500000001),pk_k(B)))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(1),c:after(1))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))");
        invalid_ms("d:or_d(after(1),multi(2,A,B,C))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(1),multi(2,D,E,F))");
        invalid_ms("or_d(or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("and_b(or_c(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("nv:after(1)");
        invalid_ms("or_d(after(1),n:after(1))");
        invalid_ms("c:or_b(or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(pk_k(A),c:after(500000001)),pk_k(B))");
        invalid_ms("or_b(cn:or_i(0,after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(pk_k(A),or_c(after(1),multi(2,B,C,D)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(1),after(1)))");
        invalid_ms("or_b(after(500000001),a:pk_k(A))");
        invalid_ms("or_d(s:or_c(after(1),pk_h(A)),multi(2,B,C,D))");
        invalid_ms("c:thresh(2,multi(2,A,B,C),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_d(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ja:after(500000001)");
        invalid_ms("thresh(2,multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(svs:thresh(2,after(500000001),multi(2,A,B,C),pk_h(D)),after(1))");
        invalid_ms("thresh(2,s:pk_k(A),after(1),after(1))");
        invalid_ms("jd:or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(1),c:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(pk_k(A),or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("djda:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(s:or_c(after(1),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("or_d(j:multi(2,A,B,C),or_c(vs:multi(2,D,E,F),d:multi(2,G,I,J)))");
        invalid_ms("d:thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_b(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jds:multi(2,A,B,C)");
        invalid_ms("or_d(or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:or_b(multi(2,A,B,C),after(500000001))");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(pk_h(A),s:or_c(after(500000001),after(500000001)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),multi(2,A,B,C))");
        invalid_ms("andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D))");
        invalid_ms("or_d(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vc:after(500000001))");
        invalid_ms("or_d(s:or_d(after(500000001),v:after(500000001)),pk_k(A))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),multi(2,D,E,F),pk_k(G))");
        invalid_ms("thresh(2,or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1),multi(2,A,B,C))");
        invalid_ms("js:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_b(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("thresh(2,multi(2,A,B,C),s:after(500000001),multi(2,D,E,F))");
        invalid_ms("cvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(a:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(500000001))");
        invalid_ms("cs:or_b(after(500000001),multi(2,A,B,C))");
        invalid_ms("c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cvs:or_c(after(1),after(1))");
        invalid_ms("cvs:or_b(after(1),multi(2,A,B,C))");
        invalid_ms("ds:or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),after(500000001),multi(2,D,E,F))");
        invalid_ms("or_d(after(500000001),c:or_b(multi(2,A,B,C),after(500000001)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_b(s:or_c(s:or_d(after(500000001),after(500000001)),after(500000001)),pk_k(A))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),multi(2,D,E,F),pk_h(G))");
        invalid_ms("j:or_d(or_c(after(1),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_b(j:or_b(pk_k(A),after(500000001)),pk_h(B))");
        invalid_ms("or_d(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),v:pk_h(D))");
        invalid_ms("nvs:multi(2,A,B,C)");
        invalid_ms("cvac:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_d(vs:or_d(after(1),after(1)),after(500000001))");
        invalid_ms("or_d(cs:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("d:thresh(2,multi(2,A,B,C),multi(2,D,E,F),multi(2,G,I,J))");
        invalid_ms("or_d(s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("jsvs:or_c(after(500000001),after(500000001))");
        invalid_ms("or_b(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(500000001),d:multi(2,A,B,C))");
        invalid_ms("or_d(ca:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_h(D),c:after(1))");
        invalid_ms("and_v(pk_h(A),csc:after(500000001))");
        invalid_ms("or_b(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(v:pk_h(A),multi(2,B,C,D))");
        invalid_ms("or_d(after(1),s:or_c(after(1),after(1)))");
        invalid_ms("andor(pk_k(A),multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(s:thresh(2,after(500000001),after(500000001),pk_k(A)),pk_h(B)),multi(2,C,D,E))");
        invalid_ms("thresh(2,c:after(500000001),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),a:multi(2,D,E,F))");
        invalid_ms("thresh(2,after(1),multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(1),after(1)))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_h(D),a:after(500000001))");
        invalid_ms("or_d(or_b(pk_h(A),after(1)),pk_h(B))");
        invalid_ms("or_d(multi(2,A,B,C),a:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),pk_h(D))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jsj:or_c(after(1),multi(2,A,B,C))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(a:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(j:pk_h(A),multi(2,B,C,D))");
        invalid_ms("or_b(after(1),after(500000001))");
        invalid_ms("or_i(0,s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_d(after(500000001),n:pk_k(A))");
        invalid_ms("andor(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_b(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("or_d(da:pk_h(A),after(500000001))");
        invalid_ms("jvs:or_b(pk_h(A),after(500000001))");
        invalid_ms("or_d(or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("cj:after(500000001)");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_b(pk_k(A),j:pk_h(B))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),pk_k(D))");
        invalid_ms("or_d(after(500000001),pk_h(A))");
        invalid_ms("j:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_i(0,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,thresh(2,pk_h(A),multi(2,B,C,D),v:or_d(multi(2,E,F,G),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(I))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:multi(2,A,B,C))");
        invalid_ms("c:thresh(2,after(500000001),after(500000001),pk_k(A))");
        invalid_ms("or_b(after(500000001),c:pk_h(A))");
        invalid_ms("js:multi(2,A,B,C)");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(500000001),v:multi(2,A,B,C)))");
        invalid_ms("or_d(and_b(pk_k(A),multi(2,B,C,D)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(pk_k(A),after(1))");
        invalid_ms("or_b(or_c(after(1),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("andor(after(500000001),after(500000001),d:after(500000001))");
        invalid_ms("and_v(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_h(A),after(500000001),v:pk_k(B))");
        invalid_ms("or_d(s:or_c(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:thresh(2,multi(2,A,B,C),pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:multi(2,A,B,C))");
        invalid_ms("j:or_d(s:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),j:multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),j:multi(2,B,C,D))");
        invalid_ms("or_b(after(1),c:pk_h(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),ac:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jj:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:or_b(after(500000001),multi(2,A,B,C)))");
        invalid_ms("cs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("ns:thresh(2,pk_k(A),pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_i(0,or_c(after(1),multi(2,A,B,C)))");
        invalid_ms("or_d(or_b(after(1),pk_k(A)),pk_k(B))");
        invalid_ms("and_b(s:or_b(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("and_v(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(s:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,v:multi(2,A,B,C),pk_h(D),pk_k(E))");
        invalid_ms("andor(or_d(after(1),after(1)),after(1),j:after(1))");
        invalid_ms("or_d(and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(500000001),csvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("cs:or_b(multi(2,A,B,C),after(500000001))");
        invalid_ms("thresh(2,after(1),c:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(after(1),j:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:after(1)))");
        invalid_ms("or_d(vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("cs:thresh(2,after(500000001),pk_k(A),after(500000001))");
        invalid_ms("or_b(pk_k(A),or_d(after(1),multi(2,B,C,D)))");
        invalid_ms("or_d(or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),pk_k(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),after(500000001))");
        invalid_ms("thresh(2,after(1),multi(2,A,B,C),n:pk_h(D))");
        invalid_ms("cs:thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("ca:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:or_b(pk_h(A),after(500000001))");
        invalid_ms("or_d(s:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_i(0,vs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("thresh(2,after(500000001),pk_h(A),after(500000001))");
        invalid_ms("thresh(2,pk_k(A),pk_h(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(after(500000001),after(1))");
        invalid_ms("cs:thresh(2,pk_k(A),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(multi(2,A,B,C),pk_h(D))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:andor(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("jc:thresh(2,after(1),after(1),multi(2,A,B,C))");
        invalid_ms("c:thresh(2,multi(2,A,B,C),after(1),after(1))");
        invalid_ms("thresh(2,c:pk_k(A),pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:and_b(a:after(1),multi(2,A,B,C))");
        invalid_ms("dsj:pk_h(A)");
        invalid_ms("thresh(2,after(500000001),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),dv:after(500000001))");
        invalid_ms("dvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))),after(1))");
        invalid_ms("ds:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_d(after(500000001),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("jsvs:or_d(after(500000001),after(500000001))");
        invalid_ms("cs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:or_d(v:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(after(500000001),s:after(500000001))");
        invalid_ms("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,or_c(vs:or_c(after(500000001),after(500000001)),after(500000001)))");
        invalid_ms("or_b(pk_k(A),s:or_d(after(500000001),pk_k(B)))");
        invalid_ms("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("cs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_i(0,s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_b(s:after(500000001),pk_h(A))");
        invalid_ms("ns:or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,v:pk_k(A),multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(v:or_d(s:thresh(2,pk_h(A),after(500000001),pk_k(B)),pk_k(C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("andor(a:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cvs:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("and_v(n:pk_k(A),n:pk_k(B))");
        invalid_ms("or_d(or_c(or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(after(500000001),after(500000001)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,after(1),pk_k(A),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:thresh(2,after(1),after(1),pk_h(A))");
        invalid_ms("or_i(0,or_c(after(1),andor(after(1),after(1),c:after(1))))");
        invalid_ms("dcaj:pk_k(A)");
        invalid_ms("thresh(2,multi(2,A,B,C),after(500000001),c:after(500000001))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),after(500000001),after(500000001))");
        invalid_ms("or_d(or_c(after(500000001),after(500000001)),pk_k(A))");
        invalid_ms("jvs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_b(pk_h(A),s:after(1))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),pk_h(A))");
        invalid_ms("or_b(multi(2,A,B,C),thresh(2,multi(2,D,E,F),after(500000001),after(500000001)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),pk_k(B))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(500000001),after(500000001))))");
        invalid_ms("or_d(s:or_b(multi(2,A,B,C),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(after(1),multi(2,D,E,F)))");
        invalid_ms("or_b(after(500000001),thresh(2,multi(2,A,B,C),after(500000001),c:or_c(after(500000001),after(500000001))))");
        invalid_ms("or_d(after(1),or_b(after(500000001),after(500000001)))");
        invalid_ms("thresh(2,a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(or_i(0,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),n:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("or_d(s:or_b(after(1),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("js:thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("and_v(after(1),or_d(after(1),multi(2,A,B,C)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),aj:multi(2,A,B,C))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:multi(2,A,B,C))");
        invalid_ms("or_d(or_b(d:multi(2,A,B,C),after(500000001)),pk_k(D))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("d:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_b(cv:after(500000001),pk_k(A))");
        invalid_ms("thresh(2,pk_h(A),v:pk_h(B),multi(2,C,D,E))");
        invalid_ms("or_d(after(500000001),v:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("or_d(or_i(0,multi(2,A,B,C)),pk_k(D))");
        invalid_ms("c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("cvs:or_c(after(500000001),after(500000001))");
        invalid_ms("and_v(pk_h(A),multi(2,B,C,D))");
        invalid_ms("c:or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_i(0,s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))))");
        invalid_ms("andor(after(500000001),and_v(and_v(after(500000001),ca:after(500000001)),after(500000001)),and_v(after(500000001),a:after(500000001)))");
        invalid_ms("or_d(and_v(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jnv:pk_k(A)");
        invalid_ms("or_d(sj:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_h(A),v:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(after(500000001),multi(2,A,B,C)),after(1))");
        invalid_ms("or_d(pk_k(A),thresh(2,after(1),pk_k(B),after(1)))");
        invalid_ms("or_b(sc:or_b(multi(2,A,B,C),after(500000001)),after(500000001))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(500000001))");
        invalid_ms("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(after(500000001),and_v(after(500000001),after(500000001)))");
        invalid_ms("or_b(s:multi(2,A,B,C),or_b(after(1),pk_k(D)))");
        invalid_ms("thresh(2,after(1),s:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(dv:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),multi(2,A,B,C))");
        invalid_ms("or_d(pk_h(A),s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("or_d(pk_k(A),or_d(after(1),after(1)))");
        invalid_ms("or_d(after(500000001),after(1))");
        invalid_ms("or_b(multi(2,A,B,C),pk_k(D))");
        invalid_ms("or_d(pk_k(A),or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,pk_k(A),pk_k(B),v:multi(2,C,D,E))");
        invalid_ms("or_d(svs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),n:pk_k(D))");
        invalid_ms("or_d(after(1),c:or_c(or_i(0,d:after(1)),after(1)))");
        invalid_ms("or_i(0,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("or_i(0,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:multi(2,A,B,C)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sv:pk_k(A))");
        invalid_ms("thresh(2,after(1),pk_h(A),multi(2,B,C,D))");
        invalid_ms("d:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1))");
        invalid_ms("thresh(2,after(1),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:pk_h(A),after(500000001),after(500000001))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),thresh(2,after(500000001),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(a:pk_h(A),j:multi(2,B,C,D))");
        invalid_ms("thresh(2,v:after(500000001),multi(2,A,B,C),pk_k(D))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(1))");
        invalid_ms("jvs:or_d(after(1),after(1))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),pk_h(D))");
        invalid_ms("and_b(after(500000001),and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_d(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("thresh(2,pk_h(A),pk_h(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("csc:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("and_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("d:or_c(after(500000001),after(500000001))");
        invalid_ms("or_b(s:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_h(A),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(1))");
        invalid_ms("and_b(and_b(after(500000001),and_v(a:after(500000001),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),pk_k(A))");
        invalid_ms("n:or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_b(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),c:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("dv:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))");
        invalid_ms("or_b(multi(2,A,B,C),after(1))");
        invalid_ms("or_d(thresh(2,after(500000001),pk_k(A),after(500000001)),pk_k(B))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("ns:or_d(after(1),after(1))");
        invalid_ms("njj:after(1)");
        invalid_ms("n:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(after(500000001),a:pk_h(A))");
        invalid_ms("cs:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))");
        invalid_ms("or_d(after(500000001),n:after(1))");
        invalid_ms("or_d(s:or_c(after(1),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(after(1),after(1))");
        invalid_ms("or_d(or_c(or_c(s:or_d(after(500000001),after(500000001)),after(500000001)),after(500000001)),after(1))");
        invalid_ms("d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("ns:pk_h(A)");
        invalid_ms("and_v(s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:or_c(after(1),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,G,I,J)))");
        invalid_ms("or_b(or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,or_c(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001),v:multi(2,D,E,F))");
        invalid_ms("j:or_b(multi(2,A,B,C),after(1))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:or_d(after(500000001),pk_k(A)))");
        invalid_ms("n:and_v(pk_k(A),pk_h(B))");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),after(500000001)))");
        invalid_ms("or_d(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),pk_k(A))");
        invalid_ms("or_d(c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(cd:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,or_c(after(1),s:after(1)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sv:after(500000001))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("or_d(and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_b(or_b(after(1),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("n:or_d(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)))");
        invalid_ms("or_d(and_b(pk_k(A),pk_k(B)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:andor(after(500000001),after(500000001),pk_h(A))");
        invalid_ms("c:or_b(sc:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(500000001),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),pk_k(D))");
        invalid_ms("or_b(s:multi(2,A,B,C),v:after(500000001))");
        invalid_ms("or_b(v:after(500000001),multi(2,A,B,C))");
        invalid_ms("and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C))");
        invalid_ms("or_b(or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(after(500000001),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(500000001),pk_k(D))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,pk_h(A),c:after(500000001),multi(2,B,C,D))");
        invalid_ms("or_d(or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(pk_h(A),s:or_c(after(500000001),d:multi(2,B,C,D)))");
        invalid_ms("dj:and_v(pk_h(A),pk_k(B))");
        invalid_ms("or_b(or_c(after(500000001),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_b(after(1),pk_h(A))");
        invalid_ms("cs:thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(or_c(pk_k(A),or_d(after(500000001),after(500000001))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),j:or_c(after(500000001),multi(2,B,C,D)))");
        invalid_ms("n:pk_h(A)");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:pk_k(A),after(500000001))");
        invalid_ms("or_d(after(500000001),a:pk_k(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),v:multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,or_c(after(1),pk_k(A)),after(1),pk_k(B))");
        invalid_ms("cs:or_b(after(500000001),pk_h(A))");
        invalid_ms("cac:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_i(cv:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0)");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),v:pk_h(D))");
        invalid_ms("or_b(thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)),multi(2,E,F,G))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),svs:or_d(after(500000001),multi(2,A,B,C)))");
        invalid_ms("thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("and_b(v:after(1),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_c(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:pk_h(A))");
        invalid_ms("thresh(2,c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),after(1))");
        invalid_ms("or_d(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(d:or_c(after(1),after(1)),s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),multi(2,B,C,D))");
        invalid_ms("or_d(pk_k(A),v:or_d(after(1),after(1)))");
        invalid_ms("or_i(0,or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("ja:pk_k(A)");
        invalid_ms("or_b(pk_h(A),and_v(multi(2,B,C,D),multi(2,E,F,G)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),after(1))");
        invalid_ms("or_b(multi(2,A,B,C),or_d(after(1),after(1)))");
        invalid_ms("dc:pk_h(A)");
        invalid_ms("ds:or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),v:or_d(after(500000001),after(500000001)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C))");
        invalid_ms("csd:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),v:after(500000001))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:pk_h(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jns:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(vs:or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cn:after(500000001)");
        invalid_ms("cs:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_h(A),multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(j:multi(2,A,B,C),after(500000001)),multi(2,D,E,F))");
        invalid_ms("thresh(2,svs:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A),after(500000001))");
        invalid_ms("or_d(multi(2,A,B,C),svs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)))");
        invalid_ms("or_d(or_c(after(500000001),after(1)),after(500000001))");
        invalid_ms("c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(s:or_d(after(500000001),after(500000001)),multi(2,A,B,C))");
        invalid_ms("or_d(or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_d(ds:multi(2,A,B,C),pk_h(D))");
        invalid_ms("or_d(pk_k(A),or_d(multi(2,B,C,D),multi(2,E,F,G)))");
        invalid_ms("and_b(a:after(500000001),a:after(500000001))");
        invalid_ms("thresh(2,after(500000001),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(after(500000001),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001),pk_h(A))");
        invalid_ms("or_d(pk_h(A),after(1))");
        invalid_ms("or_b(j:multi(2,A,B,C),after(500000001))");
        invalid_ms("jv:pk_h(A)");
        invalid_ms("or_b(or_c(pk_k(A),or_b(multi(2,B,C,D),after(500000001))),multi(2,E,F,G))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),d:multi(2,D,E,F))");
        invalid_ms("or_b(or_b(pk_h(A),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(c:after(500000001),c:after(500000001))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(1)),pk_k(A))");
        invalid_ms("thresh(2,pk_h(A),pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("js:or_d(after(1),multi(2,A,B,C))");
        invalid_ms("or_i(0,s:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_h(A),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(after(1),c:pk_h(A))");
        invalid_ms("or_b(sj:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:multi(2,A,B,C),pk_k(D))");
        invalid_ms("thresh(2,pk_k(A),svs:or_d(after(500000001),multi(2,B,C,D)),after(500000001))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_i(pk_k(A),0),multi(2,B,C,D))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1))");
        invalid_ms("thresh(2,c:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(s:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))");
        invalid_ms("or_d(s:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("n:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_b(or_b(j:multi(2,A,B,C),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_c(after(500000001),after(500000001)),pk_h(A))");
        invalid_ms("or_b(multi(2,A,B,C),or_d(after(500000001),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D))");
        invalid_ms("c:or_d(after(1),pk_k(A))");
        invalid_ms("j:or_b(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_b(pk_h(A),s:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("andor(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(vd:after(500000001),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(c:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),pk_k(B))");
        invalid_ms("or_d(pk_h(A),after(1))");
        invalid_ms("or_d(c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))");
        invalid_ms("ns:multi(2,A,B,C)");
        invalid_ms("andor(pk_k(A),multi(2,B,C,D),after(500000001))");
        invalid_ms("c:or_i(or_c(d:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),0)");
        invalid_ms("or_b(s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))");
        invalid_ms("or_d(s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("and_b(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:after(500000001))");
        invalid_ms("nvs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:thresh(2,pk_h(A),multi(2,B,C,D),after(500000001))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_k(D),j:multi(2,E,F,G))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_b(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(pk_h(A),a:after(500000001))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),s:pk_k(D))");
        invalid_ms("or_b(j:multi(2,A,B,C),pk_h(D))");
        invalid_ms("c:thresh(2,j:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("or_d(ad:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),after(1))");
        invalid_ms("d:thresh(2,after(1),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(1))");
        invalid_ms("or_d(js:multi(2,A,B,C),pk_h(D))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_d(after(500000001),after(500000001)))");
        invalid_ms("and_v(c:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(pk_h(A),pk_k(B)))");
        invalid_ms("thresh(2,or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))");
        invalid_ms("or_d(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(D))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:thresh(2,after(1),multi(2,A,B,C),multi(2,D,E,F)))");
        invalid_ms("thresh(2,s:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("n:or_c(after(500000001),after(1))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("c:or_d(pk_k(A),va:or_b(after(500000001),pk_k(B)))");
        invalid_ms("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("c:or_d(or_d(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:or_d(after(500000001),after(500000001)),vs:or_d(after(500000001),after(500000001)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(500000001),after(1)))");
        invalid_ms("or_d(s:or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(500000001))");
        invalid_ms("thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:pk_k(B))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))");
        invalid_ms("or_d(s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("n:or_c(multi(2,A,B,C),or_b(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(s:or_b(s:thresh(2,after(1),pk_k(A),after(1)),pk_k(B)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,s:thresh(2,after(500000001),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("ns:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(multi(2,A,B,C),vsc:or_d(after(1),multi(2,D,E,F)))");
        invalid_ms("or_b(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),multi(2,A,B,C))");
        invalid_ms("d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(vs:thresh(2,after(1),multi(2,A,B,C),after(1)),multi(2,D,E,F))");
        invalid_ms("or_d(multi(2,A,B,C),a:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_b(s:or_c(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("or_d(pk_k(A),v:or_d(after(1),pk_k(B)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:or_c(after(500000001),after(500000001)))");
        invalid_ms("or_b(after(1),a:pk_h(A))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(pk_k(A),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("n:or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),after(1))");
        invalid_ms("and_b(and_b(after(1),after(1)),pk_k(A))");
        invalid_ms("thresh(2,after(500000001),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(or_d(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),pk_k(B))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:or_b(pk_h(A),after(1))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(pk_k(A),1)");
        invalid_ms("or_i(0,s:or_c(after(500000001),pk_h(A)))");
        invalid_ms("or_b(pk_h(A),sj:multi(2,B,C,D))");
        invalid_ms("or_d(pk_k(A),or_c(after(1),pk_k(B)))");
        invalid_ms("or_d(pk_k(A),or_b(multi(2,B,C,D),after(500000001)))");
        invalid_ms("or_d(pk_k(A),s:or_b(multi(2,B,C,D),after(1)))");
        invalid_ms("or_d(pk_k(A),svsvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("d:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("n:or_d(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(500000001))");
        invalid_ms("or_d(or_d(multi(2,A,B,C),or_c(after(1),multi(2,D,E,F))),after(500000001))");
        invalid_ms("or_d(a:after(500000001),anana:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("csc:or_d(after(1),after(1))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),j:multi(2,G,I,J))");
        invalid_ms("or_i(0,s:or_b(after(500000001),multi(2,A,B,C)))");
        invalid_ms("nsvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("c:thresh(2,s:or_c(after(500000001),multi(2,A,B,C)),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(B))");
        invalid_ms("j:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),n:or_d(after(1),after(1)))");
        invalid_ms("or_d(pk_k(A),dc:andor(after(500000001),after(500000001),c:pk_k(B)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),c:after(1))");
        invalid_ms("and_v(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),1)");
        invalid_ms("or_d(v:pk_h(A),s:pk_h(B))");
        invalid_ms("dj:and_v(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(after(1),c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(pk_h(A),or_c(after(1),c:pk_h(B)))");
        invalid_ms("or_b(multi(2,A,B,C),or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("andor(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("ns:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(1),d:after(1),pk_k(A))");
        invalid_ms("c:or_b(pk_k(A),after(1))");
        invalid_ms("or_d(or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:and_v(c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),j:multi(2,G,I,J))");
        invalid_ms("or_b(multi(2,A,B,C),or_c(after(500000001),multi(2,D,E,F)))");
        invalid_ms("or_b(multi(2,A,B,C),s:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D),pk_k(E)))");
        invalid_ms("or_b(s:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(1),d:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_b(pk_k(A),v:pk_k(B))");
        invalid_ms("or_d(or_c(after(500000001),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:after(500000001))");
        invalid_ms("dcvj:after(500000001)");
        invalid_ms("or_d(pk_k(A),or_b(n:after(500000001),pk_k(B)))");
        invalid_ms("or_d(after(1),c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("andor(pk_h(A),c:after(500000001),after(500000001))");
        invalid_ms("or_d(s:or_b(multi(2,A,B,C),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("cvsvs:or_c(after(1),multi(2,A,B,C))");
        invalid_ms("and_b(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D))");
        invalid_ms("or_b(pk_h(A),c:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,thresh(2,after(500000001),after(500000001),after(500000001)),after(500000001),after(500000001))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_b(pk_h(D),after(500000001)))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,after(500000001),s:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(vs:pk_k(A),multi(2,B,C,D))");
        invalid_ms("or_d(s:or_d(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("or_b(pk_k(A),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(a:after(1),pk_k(A))");
        invalid_ms("d:or_d(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(s:or_b(pk_h(A),after(1)),pk_k(B))");
        invalid_ms("and_v(a:after(1),after(1))");
        invalid_ms("or_b(pk_k(A),s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(1))");
        invalid_ms("or_d(after(500000001),a:pk_k(A))");
        invalid_ms("or_b(pk_h(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)))");
        invalid_ms("or_d(pk_k(A),pk_k(B))");
        invalid_ms("j:or_c(after(1),multi(2,A,B,C))");
        invalid_ms("or_b(pk_k(A),js:pk_h(B))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))");
        invalid_ms("and_b(va:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),after(500000001),pk_h(A))");
        invalid_ms("thresh(2,or_c(after(500000001),after(500000001)),pk_h(A),after(500000001))");
        invalid_ms("or_d(or_c(after(500000001),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(500000001),a:or_c(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_d(or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),pk_k(A))");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("thresh(2,d:after(500000001),after(500000001),after(500000001))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(multi(2,D,E,F),after(500000001)))");
        invalid_ms("or_b(c:after(1),pk_h(A))");
        invalid_ms("thresh(2,after(1),pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_i(0,s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))),after(1))");
        invalid_ms("or_d(sv:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(1),pk_h(A),pk_k(B))");
        invalid_ms("c:and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sc:or_c(after(500000001),after(500000001)),d:after(500000001))");
        invalid_ms("da:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("d:or_c(after(500000001),pk_k(A))");
        invalid_ms("dn:pk_h(A)");
        invalid_ms("js:or_d(after(500000001),after(500000001))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("cs:thresh(2,pk_k(A),after(500000001),after(500000001))");
        invalid_ms("thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(B)))");
        invalid_ms("or_d(c:after(1),pk_k(A))");
        invalid_ms("or_d(vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("thresh(2,c:after(1),pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)),after(500000001))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:or_d(after(500000001),multi(2,A,B,C)))");
        invalid_ms("cvs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("or_d(s:or_c(after(1),pk_k(A)),multi(2,B,C,D))");
        invalid_ms("or_d(svs:thresh(2,after(500000001),multi(2,A,B,C),multi(2,D,E,F)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D))");
        invalid_ms("or_d(pk_h(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(j:multi(2,A,B,C),after(500000001)))");
        invalid_ms("js:or_d(after(1),after(1))");
        invalid_ms("thresh(2,pk_k(A),or_b(multi(2,B,C,D),after(1)),after(1))");
        invalid_ms("thresh(2,after(1),after(1),pk_k(A))");
        invalid_ms("cscs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("n:or_d(multi(2,A,B,C),v:pk_h(D))");
        invalid_ms("or_d(s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),aj:multi(2,A,B,C)),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)))");
        invalid_ms("or_d(j:multi(2,A,B,C),pk_h(D))");
        invalid_ms("or_d(vsvsvs:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))");
        invalid_ms("j:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_d(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(pk_h(A),after(1)),pk_k(B))");
        invalid_ms("or_b(thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),after(500000001))");
        invalid_ms("cs:thresh(2,multi(2,A,B,C),pk_h(D),after(500000001))");
        invalid_ms("thresh(2,multi(2,A,B,C),vs:after(1),sns:multi(2,D,E,F))");
        invalid_ms("or_d(pk_h(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_b(multi(2,A,B,C),s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(pk_h(A),s:or_b(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("andor(after(1),after(1),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(multi(2,A,B,C),multi(2,D,E,F)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(after(1),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("and_b(and_b(multi(2,A,B,C),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(and_v(multi(2,A,B,C),multi(2,D,E,F)),pk_k(G))");
        invalid_ms("js:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,pk_k(A),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))),pk_k(A))");
        invalid_ms("or_b(or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,and_v(a:or_i(after(500000001),0),after(500000001)),pk_h(A),multi(2,B,C,D))");
        invalid_ms("or_d(thresh(2,after(500000001),after(500000001),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("cs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(vs:or_d(after(1),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_c(after(1),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("js:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_b(multi(2,A,B,C),s:or_c(after(500000001),after(500000001)))");
        invalid_ms("and_v(pk_h(A),1)");
        invalid_ms("thresh(2,after(500000001),or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D))");
        invalid_ms("or_b(ca:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(after(500000001),a:multi(2,A,B,C))");
        invalid_ms("or_b(or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_b(pk_h(A),after(500000001))");
        invalid_ms("or_b(pk_k(A),s:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(pk_k(A),s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),c:after(500000001))");
        invalid_ms("or_b(multi(2,A,B,C),aj:pk_k(D))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),pk_h(A))");
        invalid_ms("or_d(sv:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),multi(2,B,C,D))");
        invalid_ms("or_d(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(or_b(pk_h(A),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),v:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jcs:or_b(after(1),pk_h(A))");
        invalid_ms("cs:or_i(0,s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(1),pk_h(A)))");
        invalid_ms("or_b(s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),multi(2,B,C,D))");
        invalid_ms("or_b(s:or_c(after(1),pk_k(A)),multi(2,B,C,D))");
        invalid_ms("or_d(or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(after(500000001),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))),s:or_d(after(500000001),after(500000001)))");
        invalid_ms("j:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(a:or_c(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("or_d(a:after(1),a:after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))");
        invalid_ms("or_d(multi(2,A,B,C),and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("j:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),pk_k(B))");
        invalid_ms("or_d(or_i(multi(2,A,B,C),0),pk_h(D))");
        invalid_ms("or_d(after(1),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,c:after(500000001),pk_k(A),after(500000001))");
        invalid_ms("or_d(or_b(vs:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("c:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(after(500000001),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("nac:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_i(0,s:or_b(multi(2,A,B,C),after(500000001)))");
        invalid_ms("and_b(pk_h(A),and_b(after(1),pk_k(B)))");
        invalid_ms("or_d(j:multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("c:or_d(after(1),multi(2,A,B,C))");
        invalid_ms("dj:pk_k(A)");
        invalid_ms("or_i(0,vs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,multi(2,A,B,C),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D))");
        invalid_ms("c:or_d(after(500000001),pk_k(A))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_b(pk_h(A),after(500000001)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("andor(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),pk_k(D))");
        invalid_ms("jaj:multi(2,A,B,C)");
        invalid_ms("thresh(2,s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),after(500000001),after(500000001))");
        invalid_ms("or_d(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_b(or_b(c:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(vs:or_c(or_c(after(1),after(1)),after(1)),multi(2,A,B,C))");
        invalid_ms("or_i(0,or_c(after(1),after(1)))");
        invalid_ms("andor(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("j:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("d:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,or_c(after(500000001),v:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_h(A),or_b(pk_k(B),or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("c:thresh(2,pk_k(A),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),thresh(2,pk_k(B),after(1),after(1)))");
        invalid_ms("d:after(500000001)");
        invalid_ms("c:thresh(2,after(500000001),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_d(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("nv:or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(a:after(1),after(500000001))");
        invalid_ms("j:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("j:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:pk_k(A)");
        invalid_ms("and_b(a:pk_h(A),c:after(500000001))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(s:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),s:pk_k(D),pk_k(E))");
        invalid_ms("c:or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("or_d(d:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,s:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(multi(2,A,B,C),s:or_c(after(500000001),pk_h(D)))");
        invalid_ms("or_d(thresh(2,after(1),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(multi(2,A,B,C),vs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),d:pk_h(A))");
        invalid_ms("thresh(2,pk_h(A),after(1),pk_h(B))");
        invalid_ms("or_d(pk_k(A),thresh(2,pk_k(B),after(500000001),after(500000001)))");
        invalid_ms("or_d(s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(500000001))");
        invalid_ms("cd:after(500000001)");
        invalid_ms("c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("thresh(2,after(500000001),pk_k(A),s:after(500000001))");
        invalid_ms("c:thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),multi(2,B,C,D))");
        invalid_ms("or_b(multi(2,A,B,C),v:multi(2,D,E,F))");
        invalid_ms("or_i(0,svs:or_d(after(500000001),multi(2,A,B,C)))");
        invalid_ms("andor(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_b(pk_h(D),after(1)))");
        invalid_ms("or_b(pk_h(A),s:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("andor(c:after(500000001),after(500000001),multi(2,A,B,C))");
        invalid_ms("c:or_c(thresh(2,multi(2,A,B,C),after(1),after(1)),after(1))");
        invalid_ms("thresh(2,d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),multi(2,A,B,C))");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("ds:or_b(multi(2,A,B,C),after(1))");
        invalid_ms("or_d(or_c(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),or_c(pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(after(500000001),c:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("da:or_b(a:multi(2,A,B,C),a:multi(2,D,E,F))");
        invalid_ms("jvs:or_d(after(500000001),s:pk_k(A))");
        invalid_ms("nj:or_d(after(500000001),after(500000001))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),multi(2,A,B,C))");
        invalid_ms("c:or_b(pk_h(A),or_d(after(500000001),after(500000001)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("thresh(2,after(500000001),or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_d(svs:thresh(2,after(500000001),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),pk_h(D),pk_k(E))");
        invalid_ms("or_d(vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(B))");
        invalid_ms("cvs:or_d(after(1),pk_h(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))");
        invalid_ms("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("thresh(2,s:or_b(multi(2,A,B,C),after(500000001)),multi(2,D,E,F),pk_h(G))");
        invalid_ms("and_b(d:multi(2,A,B,C),pk_k(D))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_b(pk_k(A),pk_h(B))");
        invalid_ms("andor(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(multi(2,A,B,C),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_d(c:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(v:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_d(s:or_c(after(1),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("thresh(2,s:pk_k(A),after(500000001),pk_h(B))");
        invalid_ms("or_d(vc:pk_k(A),after(500000001))");
        invalid_ms("andor(pk_h(A),after(500000001),a:after(500000001))");
        invalid_ms("j:thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_k(D),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("n:pk_k(A)");
        invalid_ms("or_b(or_d(after(500000001),pk_h(A)),pk_k(B))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),pk_k(D))");
        invalid_ms("c:andor(after(500000001),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))))");
        invalid_ms("or_d(or_c(a:after(1),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("thresh(2,c:or_b(after(500000001),pk_h(A)),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),n:andor(pk_k(B),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:or_c(or_c(after(1),after(1)),pk_h(A))");
        invalid_ms("thresh(2,after(1),after(1),n:pk_k(A))");
        invalid_ms("cs:or_d(after(500000001),after(500000001))");
        invalid_ms("thresh(2,pk_h(A),after(500000001),s:after(500000001))");
        invalid_ms("or_d(s:after(500000001),v:pk_k(A))");
        invalid_ms("or_d(v:after(1),j:multi(2,A,B,C))");
        invalid_ms("or_b(multi(2,A,B,C),j:pk_h(D))");
        invalid_ms("ds:or_b(multi(2,A,B,C),after(500000001))");
        invalid_ms("or_d(or_c(after(1),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),multi(2,A,B,C))");
        invalid_ms("or_b(multi(2,A,B,C),or_c(multi(2,D,E,F),multi(2,G,I,J)))");
        invalid_ms("or_b(after(1),a:pk_k(A))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:multi(2,A,B,C))");
        invalid_ms("or_b(sc:pk_k(A),after(1))");
        invalid_ms("or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_d(after(1),pk_k(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_h(D),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("n:or_d(after(1),after(1))");
        invalid_ms("and_b(pk_h(A),after(1))");
        invalid_ms("or_d(vs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),multi(2,B,C,D))");
        invalid_ms("na:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:after(500000001))");
        invalid_ms("or_i(0,vs:or_c(after(1),multi(2,A,B,C)))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_i(0,or_c(after(500000001),multi(2,A,B,C)))");
        invalid_ms("cvs:or_b(pk_h(A),after(500000001))");
        invalid_ms("thresh(2,s:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))");
        invalid_ms("or_d(ac:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(D))");
        invalid_ms("or_b(or_b(multi(2,A,B,C),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("csvs:or_c(after(500000001),pk_h(A))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C))");
        invalid_ms("or_b(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001)))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(500000001),and_v(pk_k(A),after(500000001)))");
        invalid_ms("or_d(or_d(after(500000001),after(500000001)),multi(2,A,B,C))");
        invalid_ms("c:or_d(after(500000001),pk_h(A))");
        invalid_ms("or_d(after(1),or_d(c:thresh(2,j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(500000001))");
        invalid_ms("j:or_d(multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_d(pk_h(A),or_b(pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("js:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(pk_k(A),or_c(after(500000001),multi(2,B,C,D))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(svs:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A)))");
        invalid_ms("thresh(2,multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)),after(500000001))");
        invalid_ms("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),pk_k(B))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(500000001))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(or_c(after(1),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_d(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cva:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(after(500000001),pk_k(A))");
        invalid_ms("or_d(or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_h(A),multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cv:after(1)");
        invalid_ms("d:or_b(after(500000001),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(c:or_i(pk_h(A),0),after(500000001))");
        invalid_ms("or_d(pk_k(A),thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B),pk_k(C)))");
        invalid_ms("thresh(2,pk_h(A),pk_k(B),c:after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vn:pk_k(A))");
        invalid_ms("nvc:pk_h(A)");
        invalid_ms("or_d(j:multi(2,A,B,C),pk_k(D))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),multi(2,B,C,D))");
        invalid_ms("or_b(multi(2,A,B,C),j:pk_k(D))");
        invalid_ms("thresh(2,after(1),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_h(A),pk_h(B))");
        invalid_ms("or_i(0,pk_k(A))");
        invalid_ms("or_d(after(500000001),ns:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:thresh(2,after(500000001),after(500000001),after(500000001))");
        invalid_ms("or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(after(1),or_b(after(1),pk_h(A)))");
        invalid_ms("or_b(ac:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(500000001),after(500000001)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(after(1),multi(2,A,B,C)))");
        invalid_ms("or_b(j:multi(2,A,B,C),or_b(pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("cav:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_d(s:or_d(after(1),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:and_v(multi(2,A,B,C),1)");
        invalid_ms("thresh(2,multi(2,A,B,C),vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)),pk_h(G))");
        invalid_ms("j:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(500000001),and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_i(or_i(after(1),0),0)");
        invalid_ms("c:or_d(and_v(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:pk_k(A),v:multi(2,B,C,D))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("n:or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),after(1))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(multi(2,A,B,C),pk_h(D))");
        invalid_ms("or_b(pk_h(A),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),c:after(1))");
        invalid_ms("c:or_c(pk_h(A),or_c(after(500000001),after(500000001)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(or_d(pk_k(A),or_i(multi(2,B,C,D),0)),after(500000001)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(s:pk_h(A),svs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("andor(pk_h(A),multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),or_b(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),after(1))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(pk_h(D),vs:or_b(multi(2,E,F,G),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C)))");
        invalid_ms("thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jda:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_b(or_b(multi(2,A,B,C),or_c(after(1),multi(2,D,E,F))),multi(2,G,I,J))");
        invalid_ms("or_d(v:pk_h(A),d:multi(2,B,C,D))");
        invalid_ms("or_d(pk_h(A),c:or_c(after(500000001),pk_k(B)))");
        invalid_ms("j:after(1)");
        invalid_ms("thresh(2,v:pk_h(A),s:multi(2,B,C,D),after(500000001))");
        invalid_ms("or_d(pk_k(A),s:or_c(after(1),pk_h(B)))");
        invalid_ms("thresh(2,multi(2,A,B,C),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("n:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vc:pk_h(A))");
        invalid_ms("or_b(after(500000001),s:multi(2,A,B,C))");
        invalid_ms("thresh(2,s:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("thresh(2,s:or_c(after(500000001),pk_h(A)),after(500000001),multi(2,B,C,D))");
        invalid_ms("thresh(2,after(1),multi(2,A,B,C),d:pk_h(D))");
        invalid_ms("or_d(pk_h(A),thresh(2,multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,E,F,G)))");
        invalid_ms("thresh(2,after(1),or_c(after(1),after(1)),pk_h(A))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_b(pk_k(A),s:or_d(after(500000001),multi(2,B,C,D)))");
        invalid_ms("cvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(s:or_c(s:or_d(after(1),multi(2,A,B,C)),multi(2,D,E,F)),pk_k(G))");
        invalid_ms("c:or_i(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),0)");
        invalid_ms("or_b(s:or_d(after(1),after(1)),multi(2,A,B,C))");
        invalid_ms("j:or_d(after(500000001),a:multi(2,A,B,C))");
        invalid_ms("c:or_b(after(500000001),pk_k(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D))");
        invalid_ms("d:and_v(multi(2,A,B,C),after(1))");
        invalid_ms("or_d(pk_h(A),or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("n:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_i(pk_h(A),0),after(1))");
        invalid_ms("ja:after(1)");
        invalid_ms("d:or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("n:or_d(after(500000001),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("nvsvs:thresh(2,after(1),after(1),pk_k(A))");
        invalid_ms("or_d(and_v(after(500000001),1),after(500000001))");
        invalid_ms("thresh(2,c:after(500000001),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(G)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),avs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("andor(after(500000001),c:after(500000001),a:multi(2,A,B,C))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("and_b(or_c(a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),a:pk_h(B)),after(500000001))");
        invalid_ms("ca:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_b(after(500000001),multi(2,D,E,F)))");
        invalid_ms("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:and_b(multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("thresh(2,v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_h(A))");
        invalid_ms("or_b(jc:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),s:or_c(after(500000001),after(500000001)))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))");
        invalid_ms("thresh(2,after(500000001),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(pk_k(A),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("or_b(or_b(after(500000001),pk_k(A)),pk_k(B))");
        invalid_ms("or_d(s:or_c(after(1),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(after(1),s:pk_k(A))");
        invalid_ms("or_d(sc:or_b(multi(2,A,B,C),after(500000001)),after(500000001))");
        invalid_ms("and_v(after(500000001),ca:pk_k(A))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),a:after(500000001))");
        invalid_ms("or_b(pk_k(A),cv:after(500000001))");
        invalid_ms("or_d(pk_h(A),s:or_c(s:or_d(after(500000001),pk_k(B)),pk_k(C)))");
        invalid_ms("or_b(or_c(after(500000001),pk_k(A)),multi(2,B,C,D))");
        invalid_ms("or_i(0,or_c(after(500000001),pk_k(A)))");
        invalid_ms("or_d(vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("n:or_d(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("cs:or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cacs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(s:or_c(after(1),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(thresh(2,after(1),multi(2,A,B,C),after(1)),pk_k(D))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(pk_k(A),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0))");
        invalid_ms("c:or_b(after(1),multi(2,A,B,C))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(1),pk_k(A)))");
        invalid_ms("thresh(2,j:thresh(2,pk_k(A),after(500000001),s:after(500000001)),after(500000001),pk_k(B))");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A),multi(2,B,C,D))");
        invalid_ms("or_d(and_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(D))");
        invalid_ms("thresh(2,after(1),c:pk_h(A),pk_h(B))");
        invalid_ms("or_d(or_c(after(500000001),after(500000001)),multi(2,A,B,C))");
        invalid_ms("or_b(thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),after(500000001))");
        invalid_ms("and_b(after(500000001),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),j:multi(2,A,B,C))");
        invalid_ms("cs:thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(B))");
        invalid_ms("or_b(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D)))");
        invalid_ms("or_d(pk_h(A),or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_v(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("cvs:or_d(after(500000001),pk_k(A))");
        invalid_ms("c:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001)))");
        invalid_ms("or_d(or_c(after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_b(thresh(2,after(500000001),multi(2,A,B,C),after(500000001)),multi(2,D,E,F))");
        invalid_ms("or_d(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("or_d(after(500000001),ca:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_k(A),after(1),multi(2,B,C,D))");
        invalid_ms("j:or_d(after(500000001),pk_h(A))");
        invalid_ms("or_d(or_d(or_c(after(1),pk_k(A)),after(1)),pk_k(B))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(or_c(after(500000001),after(500000001)),after(500000001)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(500000001),after(500000001)))");
        invalid_ms("or_b(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(D))");
        invalid_ms("thresh(2,after(500000001),j:multi(2,A,B,C),pk_k(D))");
        invalid_ms("or_d(or_c(after(500000001),after(500000001)),c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_d(and_b(multi(2,A,B,C),pk_k(D)),after(1))");
        invalid_ms("c:or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(1))");
        invalid_ms("thresh(2,multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),multi(2,D,E,F))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),pk_k(A))");
        invalid_ms("or_d(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("da:after(500000001)");
        invalid_ms("or_d(multi(2,A,B,C),pk_k(D))");
        invalid_ms("nja:after(1)");
        invalid_ms("or_b(pk_h(A),s:or_b(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_b(after(500000001),or_b(multi(2,A,B,C),after(500000001)))");
        invalid_ms("n:or_c(multi(2,A,B,C),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),1)");
        invalid_ms("d:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("js:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,v:multi(2,A,B,C),pk_k(D),pk_k(E))");
        invalid_ms("cv:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("andor(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:after(1)");
        invalid_ms("and_v(after(500000001),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("j:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("and_v(after(1),multi(2,A,B,C))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),c:pk_k(A))");
        invalid_ms("or_b(n:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),after(500000001))");
        invalid_ms("n:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("jdj:after(1)");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),pk_h(D))");
        invalid_ms("or_d(s:or_d(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_k(A),after(500000001),c:pk_k(B))");
        invalid_ms("or_d(or_c(or_c(after(500000001),after(500000001)),after(500000001)),after(500000001))");
        invalid_ms("or_d(or_d(after(1),after(1)),after(1))");
        invalid_ms("or_d(pk_k(A),or_b(pk_k(B),after(1)))");
        invalid_ms("thresh(2,after(1),multi(2,A,B,C),after(1))");
        invalid_ms("or_d(after(500000001),a:after(500000001))");
        invalid_ms("or_b(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("ca:or_b(a:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))");
        invalid_ms("or_d(thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),pk_k(A))");
        invalid_ms("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),pk_k(B))");
        invalid_ms("or_d(s:or_c(after(1),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),ds:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("n:or_c(after(500000001),pk_h(A))");
        invalid_ms("c:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),multi(2,B,C,D))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:multi(2,A,B,C))");
        invalid_ms("js:or_d(after(1),multi(2,A,B,C))");
        invalid_ms("dsd:multi(2,A,B,C)");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_i(0,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))),pk_h(A))");
        invalid_ms("or_b(d:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jd:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cn:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("andor(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("cacs:pk_k(A)");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(after(500000001),n:after(500000001))");
        invalid_ms("or_d(or_c(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(500000001))");
        invalid_ms("thresh(2,v:pk_k(A),multi(2,B,C,D),pk_k(E))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),pk_h(A))");
        invalid_ms("or_b(vs:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),or_c(after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),j:after(500000001))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_d(after(500000001),after(500000001)))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(s:multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(d:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(s:or_c(after(1),after(1)),after(500000001))");
        invalid_ms("or_b(pk_h(A),v:pk_h(B))");
        invalid_ms("or_b(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,after(500000001),after(500000001),multi(2,A,B,C))");
        invalid_ms("or_d(or_d(after(500000001),pk_k(A)),pk_h(B))");
        invalid_ms("or_b(multi(2,A,B,C),thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),multi(2,A,B,C))");
        invalid_ms("or_d(pk_h(A),or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,multi(2,A,B,C),c:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cad:multi(2,A,B,C)");
        invalid_ms("cn:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("ns:after(1)");
        invalid_ms("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),pk_h(B))");
        invalid_ms("or_b(multi(2,A,B,C),or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),s:pk_h(B))");
        invalid_ms("and_b(and_v(after(1),after(1)),after(1))");
        invalid_ms("c:thresh(2,after(500000001),after(500000001),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D))");
        invalid_ms("or_d(a:after(500000001),a:pk_k(A))");
        invalid_ms("or_b(pk_k(A),s:or_b(multi(2,B,C,D),after(500000001)))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_h(D),c:after(500000001))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),n:or_d(after(500000001),after(500000001)))");
        invalid_ms("j:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),pk_h(B))");
        invalid_ms("thresh(2,pk_k(A),v:after(1),multi(2,B,C,D))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:pk_k(A),after(1))");
        invalid_ms("or_d(pk_k(A),thresh(2,after(500000001),after(500000001),pk_k(B)))");
        invalid_ms("thresh(2,pk_k(A),pk_h(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_h(A),pk_k(B))");
        invalid_ms("or_b(after(500000001),pk_h(A))");
        invalid_ms("jvs:multi(2,A,B,C)");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_d(after(500000001),c:after(500000001)),after(500000001))");
        invalid_ms("d:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),pk_k(D))");
        invalid_ms("csvs:or_c(after(500000001),pk_k(A))");
        invalid_ms("or_d(after(1),after(500000001))");
        invalid_ms("or_b(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))");
        invalid_ms("or_d(ac:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:pk_k(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),or_b(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,G,I,J))");
        invalid_ms("or_d(after(500000001),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_v(or_d(after(500000001),after(500000001)),1)");
        invalid_ms("or_b(pk_k(A),or_b(multi(2,B,C,D),after(1)))");
        invalid_ms("dv:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("c:or_c(or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(after(500000001),multi(2,D,E,F)))");
        invalid_ms("or_d(and_b(after(1),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(or_d(or_c(after(1),after(1)),after(1)),pk_h(A)))");
        invalid_ms("thresh(2,d:after(1),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("na:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("cvs:thresh(2,multi(2,A,B,C),after(500000001),after(500000001))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),a:multi(2,A,B,C))");
        invalid_ms("thresh(2,d:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("caja:or_c(after(500000001),a:pk_k(A))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(1),pk_h(A)))");
        invalid_ms("nvs:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),multi(2,D,E,F),multi(2,G,I,J))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),snv:or_d(multi(2,A,B,C),multi(2,D,E,F)))");
        invalid_ms("cvs:or_d(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_d(pk_k(A),v:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_c(after(1),after(1)),after(1))");
        invalid_ms("thresh(2,s:multi(2,A,B,C),after(500000001),multi(2,D,E,F))");
        invalid_ms("or_b(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_d(after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("thresh(2,pk_k(A),c:after(500000001),multi(2,B,C,D))");
        invalid_ms("or_i(0,s:or_d(after(500000001),multi(2,A,B,C)))");
        invalid_ms("or_d(or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_b(multi(2,A,B,C),after(500000001))");
        invalid_ms("or_b(pk_k(A),s:thresh(2,after(500000001),after(500000001),pk_h(B)))");
        invalid_ms("or_d(a:pk_k(A),or_d(after(1),after(1)))");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(1),after(1))");
        invalid_ms("and_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:pk_k(A))");
        invalid_ms("j:or_b(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("or_b(j:pk_h(A),after(1))");
        invalid_ms("na:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("cd:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_b(multi(2,A,B,C),and_v(pk_h(D),after(500000001)))");
        invalid_ms("cs:or_c(c:and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),pk_k(B))");
        invalid_ms("c:or_d(or_c(s:after(1),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_h(A),svs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)))");
        invalid_ms("thresh(2,after(500000001),c:pk_k(A),multi(2,B,C,D))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("djs:pk_h(A)");
        invalid_ms("csvsva:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),pk_h(B))");
        invalid_ms("or_b(or_d(after(500000001),after(500000001)),pk_h(A))");
        invalid_ms("cvc:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("and_v(c:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(after(500000001),pk_h(A))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),d:after(500000001))");
        invalid_ms("n:or_c(d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("thresh(2,c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(c:pk_k(A),c:after(1))");
        invalid_ms("thresh(2,after(1),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),j:multi(2,D,E,F))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),svs:or_b(pk_h(A),after(1)))");
        invalid_ms("thresh(2,c:after(500000001),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_d(pk_h(A),d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(cs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),after(500000001))");
        invalid_ms("or_b(s:or_b(multi(2,A,B,C),after(500000001)),pk_k(D))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(1),nv:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(ac:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(1),c:after(1))");
        invalid_ms("c:or_b(pk_k(A),after(500000001))");
        invalid_ms("and_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("thresh(2,n:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(vs:or_d(after(500000001),pk_h(A)),pk_k(B))");
        invalid_ms("and_b(and_v(after(500000001),after(500000001)),multi(2,A,B,C))");
        invalid_ms("dn:or_d(after(500000001),after(500000001))");
        invalid_ms("thresh(2,a:after(500000001),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:pk_h(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("or_d(pk_k(A),s:or_c(after(1),after(1)))");
        invalid_ms("or_d(sj:after(500000001),after(500000001))");
        invalid_ms("or_i(0,s:or_c(after(500000001),multi(2,A,B,C)))");
        invalid_ms("dj:after(1)");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:and_b(c:pk_k(A),multi(2,B,C,D))");
        invalid_ms("thresh(2,or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F),pk_k(G))");
        invalid_ms("or_d(c:after(1),pk_h(A))");
        invalid_ms("nv:pk_h(A)");
        invalid_ms("j:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),cs:or_d(ac:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_d(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(500000001))");
        invalid_ms("or_d(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("cvs:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(vsvs:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,v:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,or_c(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))))");
        invalid_ms("or_b(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)))");
        invalid_ms("thresh(2,after(500000001),pk_k(A),after(500000001))");
        invalid_ms("d:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_c(after(1),multi(2,A,B,C)),s:multi(2,D,E,F))");
        invalid_ms("or_i(0,s:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,G,I,J)))");
        invalid_ms("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),pk_h(A))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),j:after(500000001))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("csc:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_b(or_b(pk_h(A),after(1)),pk_h(B))");
        invalid_ms("or_b(s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_h(A),a:pk_k(B),after(500000001))");
        invalid_ms("or_d(or_c(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))),v:after(500000001))");
        invalid_ms("c:or_c(after(500000001),after(1))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("j:or_c(multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("cs:or_c(after(500000001),multi(2,A,B,C))");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("andor(after(1),after(1),after(1))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),pk_h(A))");
        invalid_ms("or_b(d:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:or_d(after(1),pk_k(A))");
        invalid_ms("or_d(pk_h(A),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),multi(2,A,B,C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("thresh(2,s:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jsvs:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1))");
        invalid_ms("ns:or_c(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_k(A))");
        invalid_ms("or_b(multi(2,A,B,C),thresh(2,multi(2,D,E,F),multi(2,G,I,J),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:or_d(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_d(pk_h(A),or_d(after(500000001),multi(2,B,C,D)))");
        invalid_ms("csvs:or_c(or_c(after(1),pk_k(A)),pk_k(B))");
        invalid_ms("or_b(pk_h(A),j:multi(2,B,C,D))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),or_d(after(500000001),after(500000001)))");
        invalid_ms("thresh(2,pk_k(A),pk_k(B),or_c(after(1),after(1)))");
        invalid_ms("jc:after(500000001)");
        invalid_ms("thresh(2,c:after(500000001),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(pk_h(A),and_v(or_b(after(1),after(1)),after(1))),after(1))");
        invalid_ms("thresh(2,after(1),c:and_v(after(1),or_b(after(1),after(1))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_c(after(500000001),multi(2,A,B,C)),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:pk_h(A))");
        invalid_ms("cs:or_d(after(500000001),multi(2,A,B,C))");
        invalid_ms("thresh(2,v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),pk_h(D))");
        invalid_ms("or_d(or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(vs:or_d(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("or_b(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(D))");
        invalid_ms("or_b(multi(2,A,B,C),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,scsc:or_c(after(500000001),pk_h(A)))");
        invalid_ms("cvs:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),pk_h(B))");
        invalid_ms("c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_d(s:or_b(multi(2,A,B,C),after(1)),pk_k(D))");
        invalid_ms("d:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(D))");
        invalid_ms("cvs:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(s:after(1),pk_k(A))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(1),multi(2,A,B,C))");
        invalid_ms("ca:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(j:multi(2,A,B,C),or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_d(or_c(after(1),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("or_b(multi(2,A,B,C),or_c(after(500000001),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_k(A))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),multi(2,A,B,C))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),svs:or_b(multi(2,A,B,C),after(500000001)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("cvc:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("and_b(s:after(500000001),a:and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_b(multi(2,A,B,C),or_d(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:or_b(after(1),pk_k(A))");
        invalid_ms("thresh(2,pk_h(A),pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(after(1),after(1))");
        invalid_ms("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("jd:thresh(2,after(500000001),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("thresh(2,a:after(500000001),multi(2,A,B,C),pk_h(D))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),multi(2,B,C,D))");
        invalid_ms("and_b(a:pk_k(A),multi(2,B,C,D))");
        invalid_ms("thresh(2,svsvs:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),c:after(500000001))");
        invalid_ms("or_d(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("andor(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),v:multi(2,D,E,F),multi(2,G,I,J))");
        invalid_ms("thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_i(0,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),js:pk_h(A))");
        invalid_ms("or_b(cav:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,or_c(after(1),after(1)),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_d(s:or_d(after(500000001),after(500000001)),multi(2,A,B,C))");
        invalid_ms("or_b(thresh(2,after(500000001),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(D))");
        invalid_ms("or_d(s:or_c(after(1),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("nac:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sds:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(c:after(500000001),pk_h(A)),pk_k(B))");
        invalid_ms("or_d(s:or_c(after(1),multi(2,A,B,C)),after(1))");
        invalid_ms("thresh(2,s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A),after(500000001))");
        invalid_ms("d:or_b(multi(2,A,B,C),after(500000001))");
        invalid_ms("and_b(pk_k(A),after(500000001))");
        invalid_ms("or_d(vs:or_c(after(500000001),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_d(s:or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("dca:multi(2,A,B,C)");
        invalid_ms("ds:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("nsvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_b(or_b(after(1),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("c:thresh(2,pk_k(A),after(1),after(1))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(multi(2,A,B,C),pk_k(D))");
        invalid_ms("thresh(2,pk_k(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_d(vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:pk_h(A))");
        invalid_ms("c:or_i(or_c(after(1),after(1)),0)");
        invalid_ms("or_d(thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_d(or_d(after(500000001),j:after(500000001)),pk_k(A))");
        invalid_ms("or_d(or_c(after(500000001),and_b(after(500000001),after(500000001))),and_v(after(500000001),or_c(after(500000001),after(500000001))))");
        invalid_ms("ds:pk_k(A)");
        invalid_ms("c:thresh(2,after(500000001),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),multi(2,B,C,D))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jvc:or_b(pk_h(A),after(1))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(or_c(after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sc:or_b(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(500000001),or_d(after(500000001),multi(2,D,E,F)))");
        invalid_ms("thresh(2,after(1),n:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(cac:thresh(2,pk_h(A),after(1),after(1)),pk_h(B))");
        invalid_ms("thresh(2,pk_h(A),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:or_d(after(1),pk_h(A)))");
        invalid_ms("thresh(2,multi(2,A,B,C),j:or_d(thresh(2,v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D),multi(2,E,F,G)),pk_h(I)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jd:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("thresh(2,pk_k(A),or_b(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,E,F,G))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vd:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),j:multi(2,D,E,F))");
        invalid_ms("d:or_c(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,n:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("thresh(2,or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),after(500000001),pk_k(B))");
        invalid_ms("nvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,c:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("andor(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),s:multi(2,A,B,C))");
        invalid_ms("or_b(s:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_b(after(1),pk_h(A))");
        invalid_ms("or_i(0,and_v(after(500000001),pk_h(A)))");
        invalid_ms("nsv:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(thresh(2,after(1),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(multi(2,A,B,C),after(500000001)))");
        invalid_ms("or_d(multi(2,A,B,C),sc:after(500000001))");
        invalid_ms("or_b(after(1),c:or_d(after(1),after(1)))");
        invalid_ms("or_b(s:or_b(multi(2,A,B,C),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("da:pk_h(A)");
        invalid_ms("thresh(2,after(1),pk_k(A),multi(2,B,C,D))");
        invalid_ms("or_d(svs:or_d(after(1),after(1)),after(1))");
        invalid_ms("thresh(2,pk_k(A),pk_k(B),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_b(multi(2,D,E,F),after(500000001)))");
        invalid_ms("jv:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:thresh(2,after(500000001),multi(2,A,B,C),after(500000001))");
        invalid_ms("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),pk_h(B))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("or_d(pk_k(A),or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("ca:or_c(after(1),multi(2,A,B,C))");
        invalid_ms("or_b(pk_h(A),or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(after(500000001),multi(2,D,E,F)))");
        invalid_ms("or_d(pk_h(A),c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:pk_h(A),pk_k(B))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("js:pk_k(A)");
        invalid_ms("cs:thresh(2,multi(2,A,B,C),after(500000001),multi(2,D,E,F))");
        invalid_ms("or_d(after(500000001),cv:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("or_b(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("andor(pk_h(A),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,s:or_b(after(500000001),pk_h(A)))");
        invalid_ms("or_b(s:or_b(multi(2,A,B,C),after(1)),multi(2,D,E,F))");
        invalid_ms("thresh(2,pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(B)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(pk_h(A),or_b(pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("dj:pk_k(A)");
        invalid_ms("thresh(2,pk_h(A),after(1),after(1))");
        invalid_ms("thresh(2,c:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("j:thresh(2,after(500000001),pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),after(500000001))");
        invalid_ms("or_d(thresh(2,multi(2,A,B,C),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(D))");
        invalid_ms("or_b(after(1),d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("and_v(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),1)");
        invalid_ms("thresh(2,pk_h(A),j:multi(2,B,C,D),multi(2,E,F,G))");
        invalid_ms("cs:thresh(2,after(1),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A),multi(2,B,C,D))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(1)))");
        invalid_ms("or_d(after(1),j:after(1))");
        invalid_ms("or_d(vsvs:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:thresh(2,after(500000001),pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(after(1),multi(2,D,E,F)))");
        invalid_ms("or_d(c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),multi(2,A,B,C))");
        invalid_ms("or_d(thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_k(A))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001))");
        invalid_ms("cn:after(1)");
        invalid_ms("or_b(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("cvavs:or_c(after(500000001),pk_k(A))");
        invalid_ms("or_d(v:multi(2,A,B,C),or_i(pk_h(D),0))");
        invalid_ms("or_d(after(1),a:after(500000001))");
        invalid_ms("and_b(j:multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("thresh(2,after(500000001),pk_h(A),pk_k(B))");
        invalid_ms("or_d(a:after(1),after(1))");
        invalid_ms("c:or_b(multi(2,A,B,C),after(500000001))");
        invalid_ms("thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C))");
        invalid_ms("or_d(or_c(vs:or_d(after(1),pk_k(A)),pk_k(B)),pk_k(C))");
        invalid_ms("thresh(2,after(500000001),cs:or_b(pk_h(A),after(500000001)),or_i(after(500000001),0))");
        invalid_ms("or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(v:pk_k(A),pk_k(B))");
        invalid_ms("ns:thresh(2,multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(va:or_c(or_d(after(1),after(1)),after(1)),after(1))");
        invalid_ms("or_d(or_c(after(500000001),pk_k(A)),pk_h(B))");
        invalid_ms("or_d(vs:or_d(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,v:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_d(or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(B))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),pk_k(B))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:pk_h(A),after(500000001))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),s:or_b(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_v(pk_k(A),or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_h(A))");
        invalid_ms("or_b(multi(2,A,B,C),or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001),pk_k(A))");
        invalid_ms("or_d(pk_k(A),multi(2,B,C,D))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("and_v(n:after(1),1)");
        invalid_ms("thresh(2,multi(2,A,B,C),s:after(500000001),multi(2,D,E,F))");
        invalid_ms("thresh(2,thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),or_d(after(1),multi(2,B,C,D)))");
        invalid_ms("or_d(s:or_b(pk_h(A),after(500000001)),multi(2,B,C,D))");
        invalid_ms("c:andor(after(1),multi(2,A,B,C),after(1))");
        invalid_ms("or_b(pk_k(A),or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_b(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,j:after(500000001),after(500000001),after(500000001))");
        invalid_ms("thresh(2,after(500000001),s:pk_h(A),after(500000001))");
        invalid_ms("ca:after(500000001)");
        invalid_ms("j:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_b(or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_d(or_c(or_d(after(1),after(1)),after(1)),after(1))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(pk_k(D),after(1)))");
        invalid_ms("or_d(or_d(after(1),after(1)),pk_k(A))");
        invalid_ms("or_d(pk_k(A),cvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("thresh(2,multi(2,A,B,C),j:multi(2,D,E,F),multi(2,G,I,J))");
        invalid_ms("thresh(2,after(500000001),pk_h(A),d:multi(2,B,C,D))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(multi(2,A,B,C),thresh(2,after(500000001),multi(2,D,E,F),after(500000001)))");
        invalid_ms("thresh(2,after(1),multi(2,A,B,C),pk_k(D))");
        invalid_ms("j:or_d(after(500000001),a:after(500000001))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),va:after(1))");
        invalid_ms("thresh(2,v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(1),d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ns:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),s:after(500000001))");
        invalid_ms("or_d(after(1),n:or_c(after(1),after(1)))");
        invalid_ms("or_b(multi(2,A,B,C),ca:after(1))");
        invalid_ms("cvs:or_b(after(500000001),multi(2,A,B,C))");
        invalid_ms("c:and_b(and_v(after(1),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(1))");
        invalid_ms("c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,c:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(after(1),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),j:pk_k(A),after(500000001))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:pk_k(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_b(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(c:after(1),or_d(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))))");
        invalid_ms("and_b(after(500000001),thresh(2,c:and_b(after(500000001),after(500000001)),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(after(1),pk_k(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(1))");
        invalid_ms("thresh(2,v:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("jsvs:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:andor(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_d(thresh(2,pk_h(A),pk_h(B),after(500000001)),pk_k(C))");
        invalid_ms("and_b(jc:thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("ja:multi(2,A,B,C)");
        invalid_ms("ca:pk_k(A)");
        invalid_ms("or_d(or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))");
        invalid_ms("c:or_b(pk_h(A),after(1))");
        invalid_ms("j:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_k(D),j:multi(2,E,F,G))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D))");
        invalid_ms("or_d(pk_h(A),thresh(2,multi(2,B,C,D),multi(2,E,F,G),after(1)))");
        invalid_ms("andor(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(1))");
        invalid_ms("or_d(multi(2,A,B,C),vs:or_b(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(pk_k(A),or_c(after(1),multi(2,B,C,D)))");
        invalid_ms("jns:multi(2,A,B,C)");
        invalid_ms("or_d(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("c:thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_d(vs:or_c(after(1),after(1)),after(500000001))");
        invalid_ms("or_d(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))");
        invalid_ms("or_i(0,c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C))");
        invalid_ms("cs:thresh(2,multi(2,A,B,C),after(500000001),after(500000001))");
        invalid_ms("or_d(pk_k(A),thresh(2,multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,after(500000001),pk_k(A),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)))");
        invalid_ms("or_d(ca:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(1),multi(2,A,B,C))");
        invalid_ms("c:or_c(after(1),after(1))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(after(500000001),after(500000001)))");
        invalid_ms("jc:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("thresh(2,c:after(1),or_c(pk_k(A),or_i(after(1),0)),after(1))");
        invalid_ms("or_b(pk_k(A),thresh(2,after(500000001),after(500000001),pk_k(B)))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("csvs:or_b(multi(2,A,B,C),after(500000001))");
        invalid_ms("or_b(j:or_b(multi(2,A,B,C),after(1)),multi(2,D,E,F))");
        invalid_ms("thresh(2,or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(500000001),c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_d(j:pk_k(A),svs:or_d(after(1),after(1)))");
        invalid_ms("thresh(2,multi(2,A,B,C),n:pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:or_c(after(1),multi(2,A,B,C))");
        invalid_ms("thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),multi(2,B,C,D))");
        invalid_ms("or_d(and_v(pk_h(A),pk_k(B)),after(500000001))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),d:after(500000001))");
        invalid_ms("ns:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:thresh(2,after(1),multi(2,A,B,C),pk_h(D))");
        invalid_ms("and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("nvs:or_d(after(500000001),pk_k(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(pk_h(A),s:or_c(after(1),pk_k(B)))");
        invalid_ms("thresh(2,pk_k(A),multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(1),pk_k(A)))");
        invalid_ms("jcs:after(500000001)");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_i(0,s:or_d(after(1),after(1)))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_b(multi(2,A,B,C),after(500000001)),pk_k(D))");
        invalid_ms("csvs:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),pk_k(D))");
        invalid_ms("thresh(2,multi(2,A,B,C),v:pk_h(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(or_d(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),1)");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),dv:after(1))");
        invalid_ms("d:and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("thresh(2,after(500000001),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),multi(2,A,B,C))");
        invalid_ms("or_d(c:pk_k(A),c:after(500000001))");
        invalid_ms("thresh(2,after(1),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("cs:thresh(2,after(500000001),multi(2,A,B,C),pk_h(D))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(svs:thresh(2,after(500000001),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vc:after(500000001))");
        invalid_ms("or_d(s:or_b(multi(2,A,B,C),after(500000001)),multi(2,D,E,F))");
        invalid_ms("csc:or_b(multi(2,A,B,C),after(500000001))");
        invalid_ms("or_b(or_b(multi(2,A,B,C),after(500000001)),multi(2,D,E,F))");
        invalid_ms("or_d(pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("thresh(2,s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),vs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)),pk_k(G))");
        invalid_ms("and_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),pk_k(A))");
        invalid_ms("d:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(500000001))");
        invalid_ms("thresh(2,pk_h(A),pk_h(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),svs:thresh(2,after(500000001),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,multi(2,A,B,C),or_c(after(1),multi(2,D,E,F)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:or_c(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),pk_k(B)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:thresh(2,multi(2,A,B,C),after(500000001),after(500000001))");
        invalid_ms("thresh(2,d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),pk_h(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),av:and_v(after(500000001),or_b(after(500000001),after(500000001))))");
        invalid_ms("and_b(or_d(after(1),a:or_c(after(1),a:after(1))),after(1))");
        invalid_ms("andor(after(1),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("thresh(2,after(1),n:pk_h(A),multi(2,B,C,D))");
        invalid_ms("cvs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("and_v(pk_k(A),multi(2,B,C,D))");
        invalid_ms("thresh(2,pk_k(A),after(500000001),multi(2,B,C,D))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:after(500000001))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(vs:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A)))");
        invalid_ms("dv:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("and_v(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(pk_k(A),after(500000001)),j:multi(2,B,C,D))");
        invalid_ms("andor(c:after(1),after(1),pk_h(A))");
        invalid_ms("thresh(2,after(500000001),after(500000001),s:pk_h(A))");
        invalid_ms("cv:thresh(2,after(1),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),a:multi(2,A,B,C),after(500000001))");
        invalid_ms("or_b(c:after(500000001),pk_k(A))");
        invalid_ms("thresh(2,pk_h(A),after(500000001),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_c(after(1),pk_h(A)),multi(2,B,C,D))");
        invalid_ms("or_d(and_b(pk_k(A),pk_k(B)),multi(2,C,D,E))");
        invalid_ms("c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),pk_h(A))");
        invalid_ms("or_d(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("or_d(or_c(pk_h(A),or_c(after(500000001),after(500000001))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_b(multi(2,D,E,F),after(500000001)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(1),after(1)))");
        invalid_ms("or_b(or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("thresh(2,a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(500000001))");
        invalid_ms("d:thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D))");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("d:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,c:after(1),pk_h(A),multi(2,B,C,D))");
        invalid_ms("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),pk_k(B))");
        invalid_ms("or_d(or_c(after(500000001),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("or_d(s:or_d(after(500000001),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("thresh(2,after(1),n:or_i(or_d(pk_h(A),after(1)),0),after(1))");
        invalid_ms("or_b(pk_k(A),s:or_b(after(500000001),pk_h(B)))");
        invalid_ms("nj:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_b(andor(after(1),after(1),a:pk_k(A)),or_b(pk_h(B),after(1)))");
        invalid_ms("or_d(or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_d(after(1),after(1)))");
        invalid_ms("or_b(pk_h(A),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(or_c(after(1),after(1)),after(1)),pk_k(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),s:or_c(after(1),pk_k(D)))");
        invalid_ms("or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("andor(pk_h(A),j:multi(2,B,C,D),pk_k(E))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),s:multi(2,D,E,F))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_c(after(500000001),pk_k(A)),pk_k(B))");
        invalid_ms("js:or_c(after(500000001),after(500000001))");
        invalid_ms("ca:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("and_b(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),v:pk_h(B))");
        invalid_ms("cs:or_d(after(500000001),pk_k(A))");
        invalid_ms("d:pk_h(A)");
        invalid_ms("nv:after(500000001)");
        invalid_ms("c:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("thresh(2,after(1),j:multi(2,A,B,C),after(1))");
        invalid_ms("and_b(and_b(multi(2,A,B,C),after(1)),multi(2,D,E,F))");
        invalid_ms("or_b(thresh(2,after(1),pk_h(A),after(1)),pk_k(B))");
        invalid_ms("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(pk_k(A),or_b(after(1),pk_h(B)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(s:or_d(after(1),after(1)),after(1))");
        invalid_ms("or_d(or_b(multi(2,A,B,C),after(1)),pk_h(D))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),j:after(500000001))");
        invalid_ms("or_d(and_b(multi(2,A,B,C),after(1)),after(1))");
        invalid_ms("or_d(or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),pk_h(A))");
        invalid_ms("or_d(pk_k(A),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(1),pk_k(D)))");
        invalid_ms("or_b(d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_k(A))");
        invalid_ms("or_d(s:or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D))");
        invalid_ms("thresh(2,multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:multi(2,D,E,F))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),or_c(after(500000001),multi(2,D,E,F)))");
        invalid_ms("ns:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("c:or_b(after(500000001),pk_h(A))");
        invalid_ms("or_b(pk_h(A),s:or_c(after(500000001),multi(2,B,C,D)))");
        invalid_ms("da:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_d(thresh(2,after(500000001),pk_h(A),after(500000001)),pk_k(B))");
        invalid_ms("thresh(2,pk_k(A),d:pk_h(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("n:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:multi(2,A,B,C))");
        invalid_ms("thresh(2,c:after(500000001),after(500000001),pk_k(A))");
        invalid_ms("j:after(500000001)");
        invalid_ms("c:andor(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),pk_h(B))");
        invalid_ms("d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:pk_k(A))");
        invalid_ms("d:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:after(500000001),pk_k(A))");
        invalid_ms("or_b(pk_k(A),pk_h(B))");
        invalid_ms("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),pk_k(B))");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("thresh(2,or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_d(d:or_d(after(1),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("or_b(or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(after(1),vs:after(1))");
        invalid_ms("d:or_b(pk_k(A),after(500000001))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(ca:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(after(1),after(1)),aca:or_b(after(1),a:after(1)))");
        invalid_ms("c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:or_d(after(1),pk_h(A))");
        invalid_ms("and_b(after(1),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,after(500000001),pk_h(A),pk_h(B))");
        invalid_ms("or_b(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(1))");
        invalid_ms("or_d(s:or_b(pk_h(A),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,s:or_c(after(1),pk_k(A)))");
        invalid_ms("cs:or_d(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_b(pk_h(A),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)))");
        invalid_ms("thresh(2,c:after(500000001),after(500000001),multi(2,A,B,C))");
        invalid_ms("or_d(and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),pk_k(B))");
        invalid_ms("ca:or_c(after(1),pk_k(A))");
        invalid_ms("or_i(0,or_c(after(1),after(500000001)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_d(or_c(multi(2,A,B,C),j:or_b(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),pk_k(G))");
        invalid_ms("or_d(c:or_c(after(1),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_d(c:or_c(after(500000001),after(500000001)),multi(2,A,B,C))");
        invalid_ms("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),1)");
        invalid_ms("or_d(multi(2,A,B,C),or_c(j:multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,thresh(2,multi(2,A,B,C),thresh(2,pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,E,F,G))");
        invalid_ms("or_d(pk_k(A),sc:or_c(after(500000001),after(500000001)))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),thresh(2,or_c(after(1),pk_h(B)),sc:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(vs:or_b(pk_h(A),after(500000001)),pk_k(B))");
        invalid_ms("and_v(c:pk_k(A),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:pk_h(A))");
        invalid_ms("or_b(s:or_b(c:after(500000001),pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ns:or_d(after(500000001),after(500000001))");
        invalid_ms("thresh(2,pk_h(A),after(500000001),a:thresh(2,pk_k(B),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,C,D,E)))");
        invalid_ms("or_b(thresh(2,after(500000001),pk_k(A),after(500000001)),pk_k(B))");
        invalid_ms("dj:multi(2,A,B,C)");
        invalid_ms("c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),after(500000001))");
        invalid_ms("or_i(0,s:or_d(after(1),pk_k(A)))");
        invalid_ms("or_i(0,s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("or_d(or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:multi(2,B,C,D))");
        invalid_ms("ds:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_i(or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_h(A),after(500000001),pk_k(B))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("thresh(2,v:after(500000001),pk_k(A),pk_k(B))");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(or_c(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("or_b(multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,j:after(500000001),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(va:after(1),multi(2,A,B,C))");
        invalid_ms("dvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),s:multi(2,B,C,D))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001)))");
        invalid_ms("or_d(thresh(2,multi(2,A,B,C),after(500000001),after(500000001)),pk_k(D))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_i(0,s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("or_b(s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("jn:after(500000001)");
        invalid_ms("d:or_d(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("or_d(v:pk_h(A),vj:multi(2,B,C,D))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))");
        invalid_ms("d:or_c(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("thresh(2,or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F),pk_h(G))");
        invalid_ms("cs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_v(a:pk_k(A),1)");
        invalid_ms("or_d(pk_k(A),s:or_d(after(1),after(1)))");
        invalid_ms("or_d(a:or_b(or_d(after(1),after(1)),after(1)),after(1))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),multi(2,D,E,F),pk_k(G))");
        invalid_ms("or_d(or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(a:after(1),a:pk_h(A))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vc:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("andor(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:after(1))");
        invalid_ms("n:or_c(multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_d(multi(2,A,B,C),pk_k(D))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(500000001),after(500000001)))");
        invalid_ms("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),pk_k(A))");
        invalid_ms("or_d(or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(B))");
        invalid_ms("or_d(after(500000001),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sc:or_c(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_h(A),after(1),multi(2,B,C,D))");
        invalid_ms("or_b(s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))");
        invalid_ms("ds:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,or_c(j:multi(2,A,B,C),after(500000001)))");
        invalid_ms("or_b(s:or_c(after(500000001),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_d(pk_h(A),thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1)))");
        invalid_ms("c:or_d(or_d(after(500000001),after(500000001)),pk_k(A))");
        invalid_ms("c:andor(pk_h(A),after(1),after(1))");
        invalid_ms("thresh(2,after(500000001),csdcs:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("n:and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(or_c(j:pk_k(A),c:or_d(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),after(500000001))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(or_c(after(1),multi(2,A,B,C)),multi(2,D,E,F)))");
        invalid_ms("nv:and_v(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))");
        invalid_ms("nsva:or_c(a:after(500000001),a:after(500000001))");
        invalid_ms("or_b(after(500000001),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("d:and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(a:multi(2,A,B,C),after(500000001))");
        invalid_ms("or_d(after(1),c:or_c(after(500000001),multi(2,A,B,C)))");
        invalid_ms("thresh(2,or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sn:after(500000001)),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:pk_k(A))");
        invalid_ms("or_b(s:or_c(after(1),after(1)),multi(2,A,B,C))");
        invalid_ms("or_d(c:after(1),multi(2,A,B,C))");
        invalid_ms("or_d(or_d(pk_h(A),or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),pk_k(B))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(after(500000001),s:andor(after(500000001),after(500000001),after(500000001)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(after(500000001),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_h(A),thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1)))");
        invalid_ms("thresh(2,pk_k(A),or_b(after(1),pk_h(B)),pk_k(C))");
        invalid_ms("or_b(vs:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jj:and_v(pk_h(A),after(1))");
        invalid_ms("thresh(2,after(500000001),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(a:pk_h(A),after(500000001))");
        invalid_ms("or_b(multi(2,A,B,C),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(a:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,a:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))");
        invalid_ms("thresh(2,multi(2,A,B,C),or_b(pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_d(pk_h(A),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_b(or_d(after(500000001),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_b(or_d(after(500000001),pk_h(A)),multi(2,B,C,D))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("jcs:or_c(after(1),after(1))");
        invalid_ms("or_b(pk_k(A),or_b(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),or_b(multi(2,D,E,F),after(500000001)))");
        invalid_ms("d:or_b(multi(2,A,B,C),j:multi(2,D,E,F))");
        invalid_ms("or_d(or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))),pk_h(B))");
        invalid_ms("thresh(2,multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),multi(2,D,E,F))");
        invalid_ms("or_d(and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("j:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_i(0,s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),multi(2,B,C,D))");
        invalid_ms("or_d(j:multi(2,A,B,C),pk_h(D))");
        invalid_ms("or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),ns:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:or_b(multi(2,A,B,C),after(1)))");
        invalid_ms("or_d(j:multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),thresh(2,after(500000001),after(500000001),multi(2,A,B,C)))");
        invalid_ms("thresh(2,after(500000001),pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_b(ad:pk_k(A),after(500000001))");
        invalid_ms("or_d(multi(2,A,B,C),and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(c:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("thresh(2,j:thresh(2,after(500000001),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(thresh(2,after(1),pk_k(A),pk_k(B)),after(1))");
        invalid_ms("jcv:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_d(s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),multi(2,B,C,D))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cv:andor(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B))");
        invalid_ms("j:or_d(after(500000001),multi(2,A,B,C))");
        invalid_ms("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("dj:pk_k(A)");
        invalid_ms("or_d(or_c(or_b(pk_h(A),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))");
        invalid_ms("or_d(or_d(after(1),after(1)),cs:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(multi(2,A,B,C),s:or_c(after(500000001),multi(2,D,E,F)))");
        invalid_ms("thresh(2,c:thresh(2,after(500000001),after(500000001),c:pk_k(A)),or_i(after(500000001),0),after(500000001))");
        invalid_ms("ns:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_b(multi(2,A,B,C),after(500000001)),after(500000001))");
        invalid_ms("or_b(vs:after(1),pk_h(A))");
        invalid_ms("or_b(pk_k(A),thresh(2,multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,E,F,G)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_b(s:pk_k(A),v:after(1))");
        invalid_ms("or_b(pk_k(A),dv:after(1))");
        invalid_ms("andor(s:multi(2,A,B,C),multi(2,D,E,F),multi(2,G,I,J))");
        invalid_ms("or_b(pk_k(A),j:pk_h(B))");
        invalid_ms("or_b(j:multi(2,A,B,C),after(1))");
        invalid_ms("thresh(2,multi(2,A,B,C),j:multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ds:pk_h(A)");
        invalid_ms("or_b(multi(2,A,B,C),and_v(multi(2,D,E,F),after(1)))");
        invalid_ms("or_d(s:after(500000001),pk_h(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),thresh(2,after(1),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1)),pk_k(A))");
        invalid_ms("or_d(n:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),or_i(0,pk_k(B)))");
        invalid_ms("thresh(2,s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),after(500000001),after(500000001))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),av:and_b(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_d(after(500000001),or_b(after(500000001),multi(2,A,B,C)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:multi(2,A,B,C))");
        invalid_ms("ds:multi(2,A,B,C)");
        invalid_ms("or_b(pk_k(A),s:thresh(2,pk_k(B),multi(2,C,D,E),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:pk_h(A))");
        invalid_ms("thresh(2,after(1),pk_h(A),pk_h(B))");
        invalid_ms("or_d(pk_h(A),or_b(after(1),pk_h(B)))");
        invalid_ms("thresh(2,pk_k(A),pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),thresh(2,after(1),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("cs:or_c(after(1),pk_h(A))");
        invalid_ms("thresh(2,pk_h(A),multi(2,B,C,D),j:multi(2,E,F,G))");
        invalid_ms("or_d(or_d(after(1),after(500000001)),after(500000001))");
        invalid_ms("or_i(0,or_c(after(500000001),after(500000001)))");
        invalid_ms("thresh(2,v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(after(500000001),jc:pk_h(A))");
        invalid_ms("thresh(2,c:after(500000001),pk_h(A),pk_k(B))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:pk_h(A))");
        invalid_ms("or_b(svs:thresh(2,after(1),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(vs:or_c(after(1),after(1)),ns:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("ca:pk_h(A)");
        invalid_ms("cs:thresh(2,after(500000001),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cvs:or_d(after(1),pk_k(A))");
        invalid_ms("or_b(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C))");
        invalid_ms("or_b(multi(2,A,B,C),v:or_d(after(1),multi(2,D,E,F)))");
        invalid_ms("or_d(c:or_c(after(1),pk_k(A)),pk_k(B))");
        invalid_ms("d:thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_h(A))");
        invalid_ms("c:or_b(a:after(500000001),multi(2,A,B,C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(500000001),after(500000001)))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("andor(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),pk_h(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(s:multi(2,A,B,C),after(1)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A))");
        invalid_ms("or_i(0,or_c(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A)))");
        invalid_ms("cvscvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:or_b(after(500000001),multi(2,A,B,C)))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_d(after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_b(or_c(after(1),and_v(c:after(1),after(1))),pk_h(A))");
        invalid_ms("csv:or_i(or_c(after(1),pk_h(A)),0)");
        invalid_ms("j:and_v(pk_h(A),multi(2,B,C,D))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_h(D),j:pk_k(E))");
        invalid_ms("or_b(after(1),c:pk_k(A))");
        invalid_ms("jsv:pk_h(A)");
        invalid_ms("cs:or_c(after(500000001),pk_h(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(500000001),after(500000001))");
        invalid_ms("or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(D))");
        invalid_ms("or_d(or_c(a:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("dj:after(500000001)");
        invalid_ms("or_d(or_b(multi(2,A,B,C),after(1)),pk_k(D))");
        invalid_ms("or_b(pk_k(A),vs:or_b(multi(2,B,C,D),after(1)))");
        invalid_ms("da:multi(2,A,B,C)");
        invalid_ms("cs:thresh(2,multi(2,A,B,C),multi(2,D,E,F),after(500000001))");
        invalid_ms("c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:pk_h(A))");
        invalid_ms("cs:or_c(after(1),after(1))");
        invalid_ms("and_v(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),j:or_b(pk_k(B),v:pk_h(C)))");
        invalid_ms("or_d(multi(2,A,B,C),s:thresh(2,after(1),multi(2,D,E,F),after(1)))");
        invalid_ms("thresh(2,after(500000001),s:multi(2,A,B,C),after(500000001))");
        invalid_ms("c:or_i(after(1),0)");
        invalid_ms("csvavs:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(after(1),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(after(500000001),pk_k(D)))");
        invalid_ms("c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_b(or_c(after(500000001),after(500000001)),or_c(after(500000001),after(500000001)))");
        invalid_ms("or_d(pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_d(after(500000001),pk_k(A)),v:pk_k(B))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(multi(2,A,B,C),0))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(c:after(500000001),pk_h(A))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:multi(2,A,B,C),pk_h(D))");
        invalid_ms("cs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_b(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("dvs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cvs:or_b(multi(2,A,B,C),after(500000001))");
        invalid_ms("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("js:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(j:multi(2,A,B,C),1)");
        invalid_ms("or_b(after(500000001),s:pk_h(A))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),v:after(500000001))");
        invalid_ms("or_d(or_c(or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1)),pk_k(B))");
        invalid_ms("cs:or_b(after(1),pk_h(A))");
        invalid_ms("or_d(or_d(after(1),after(1)),pk_h(A))");
        invalid_ms("or_b(j:pk_h(A),multi(2,B,C,D))");
        invalid_ms("or_b(or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("or_d(pk_k(A),pk_h(B))");
        invalid_ms("or_d(c:after(500000001),v:multi(2,A,B,C))");
        invalid_ms("ja:pk_h(A)");
        invalid_ms("d:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("jd:pk_k(A)");
        invalid_ms("or_d(s:or_d(after(1),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,d:after(1),after(1),multi(2,A,B,C))");
        invalid_ms("or_b(pk_h(A),j:or_b(multi(2,B,C,D),after(1)))");
        invalid_ms("or_d(thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)),pk_h(E))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:pk_h(A))");
        invalid_ms("j:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vc:after(500000001))");
        invalid_ms("thresh(2,multi(2,A,B,C),j:multi(2,D,E,F),pk_h(G))");
        invalid_ms("or_d(after(500000001),v:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("andor(s:after(500000001),after(500000001),after(500000001))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("dcad:pk_h(A)");
        invalid_ms("or_d(vs:thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)),pk_h(G))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(1))");
        invalid_ms("or_b(thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),pk_k(B)),pk_k(C))");
        invalid_ms("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("andor(after(1),c:after(1),a:after(1))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_b(after(1),pk_h(B)))");
        invalid_ms("j:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jn:and_v(after(500000001),pk_h(A))");
        invalid_ms("or_b(pk_k(A),c:or_d(after(1),after(1)))");
        invalid_ms("or_d(after(1),vj:multi(2,A,B,C))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("thresh(2,a:after(500000001),multi(2,A,B,C),after(500000001))");
        invalid_ms("c:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("thresh(2,or_d(after(1),after(1)),after(1),pk_h(A))");
        invalid_ms("or_d(after(1),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,vs:or_c(after(1),after(1)))");
        invalid_ms("or_b(pk_h(A),s:or_c(after(500000001),after(500000001)))");
        invalid_ms("or_d(or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("andor(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),after(1))");
        invalid_ms("or_d(pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("or_b(j:multi(2,A,B,C),pk_k(D))");
        invalid_ms("or_b(multi(2,A,B,C),jn:pk_k(D))");
        invalid_ms("or_b(s:pk_k(A),after(1))");
        invalid_ms("or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("andor(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("djs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:after(500000001),after(500000001),pk_h(A))");
        invalid_ms("thresh(2,d:after(1),after(1),after(1))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)))");
        invalid_ms("thresh(2,multi(2,A,B,C),or_b(multi(2,D,E,F),after(1)),after(1))");
        invalid_ms("c:or_d(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("cvs:or_b(a:multi(2,A,B,C),after(500000001))");
        invalid_ms("or_b(pk_h(A),v:pk_k(B))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(1),a:after(1))");
        invalid_ms("dvsvs:or_d(after(1),pk_k(A))");
        invalid_ms("or_b(c:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("thresh(2,multi(2,A,B,C),j:pk_k(D),multi(2,E,F,G))");
        invalid_ms("c:or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_d(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_b(s:or_b(pk_h(A),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ns:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("jj:and_v(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jvs:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sc:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),a:pk_k(B))");
        invalid_ms("or_d(or_c(pk_k(A),or_c(after(500000001),pk_k(B))),multi(2,C,D,E))");
        invalid_ms("or_d(c:after(500000001),pk_h(A))");
        invalid_ms("or_d(s:or_d(or_c(after(1),pk_h(A)),after(1)),pk_k(B))");
        invalid_ms("or_d(pk_h(A),or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("cs:or_d(after(1),multi(2,A,B,C))");
        invalid_ms("or_d(after(500000001),pk_k(A))");
        invalid_ms("thresh(2,after(500000001),c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(pk_k(A),and_b(after(1),after(1)))");
        invalid_ms("or_b(or_b(multi(2,A,B,C),after(500000001)),d:after(500000001))");
        invalid_ms("j:or_c(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),pk_k(B))");
        invalid_ms("nsvs:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),1)");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),pk_k(B))");
        invalid_ms("or_b(multi(2,A,B,C),or_d(after(500000001),after(500000001)))");
        invalid_ms("ns:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("djd:multi(2,A,B,C)");
        invalid_ms("or_d(n:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(after(1),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("ds:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:multi(2,A,B,C),after(1),pk_k(D))");
        invalid_ms("dac:pk_h(A)");
        invalid_ms("or_d(or_d(s:or_c(after(1),after(1)),after(1)),after(1))");
        invalid_ms("or_b(n:pk_h(A),after(500000001))");
        invalid_ms("j:or_b(va:after(1),after(1))");
        invalid_ms("cs:or_c(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_b(thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),multi(2,D,E,F)),multi(2,G,I,J))");
        invalid_ms("j:or_d(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(vs:or_d(or_d(after(1),pk_k(A)),pk_h(B)),multi(2,C,D,E))");
        invalid_ms("or_b(s:or_b(multi(2,A,B,C),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:pk_k(A))");
        invalid_ms("or_d(c:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_d(pk_k(A),or_d(after(1),after(1)))");
        invalid_ms("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),a:after(1))");
        invalid_ms("or_b(vsvs:or_c(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vnvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("csvs:or_b(after(1),pk_h(A))");
        invalid_ms("or_d(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(D))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))");
        invalid_ms("or_b(j:after(500000001),or_c(after(1),pk_h(A)))");
        invalid_ms("or_d(or_i(multi(2,A,B,C),0),pk_k(D))");
        invalid_ms("and_b(pk_h(A),and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001))");
        invalid_ms("and_v(pk_h(A),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:thresh(2,after(500000001),multi(2,A,B,C),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_d(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("thresh(2,after(500000001),j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_b(multi(2,A,B,C),after(500000001)),pk_k(D))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D))");
        invalid_ms("ns:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:after(500000001))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),multi(2,A,B,C)))");
        invalid_ms("or_d(or_c(j:multi(2,A,B,C),after(1)),multi(2,D,E,F))");
        invalid_ms("or_b(or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),av:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sc:pk_h(A))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001)),after(500000001)))");
        invalid_ms("or_b(pk_h(A),s:thresh(2,after(500000001),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,pk_k(A),or_b(pk_k(B),after(500000001)),or_b(after(500000001),pk_k(C)))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(c:pk_h(A),after(1))");
        invalid_ms("or_b(pk_h(A),s:or_b(multi(2,B,C,D),after(500000001)))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001))");
        invalid_ms("or_b(or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_d(or_c(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),after(1)),after(1))");
        invalid_ms("d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_d(after(1),after(1)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("js:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_d(after(500000001),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_d(v:after(1),multi(2,A,B,C))");
        invalid_ms("or_b(c:pk_h(A),or_b(multi(2,B,C,D),after(500000001)))");
        invalid_ms("thresh(2,a:multi(2,A,B,C),after(500000001),after(500000001))");
        invalid_ms("or_b(pk_k(A),s:or_d(after(1),after(1)))");
        invalid_ms("or_b(s:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_d(vscs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),after(1))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:multi(2,A,B,C))");
        invalid_ms("n:or_d(after(1),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(pk_k(A),or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_v(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_d(thresh(2,multi(2,A,B,C),after(1),after(1)),pk_k(D))");
        invalid_ms("thresh(2,after(500000001),j:multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("j:or_d(multi(2,A,B,C),c:after(500000001))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(after(1),pk_h(A)))");
        invalid_ms("or_d(and_b(multi(2,A,B,C),vj:or_d(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,v:or_b(pk_h(A),or_b(pk_k(B),pk_k(C))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))");
        invalid_ms("thresh(2,c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_b(j:multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("cs:or_b(pk_h(A),after(1))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),c:after(1))");
        invalid_ms("or_d(vs:thresh(2,after(1),multi(2,A,B,C),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cv:or_i(s:or_c(after(500000001),multi(2,A,B,C)),0)");
        invalid_ms("thresh(2,or_b(after(500000001),multi(2,A,B,C)),multi(2,D,E,F),after(500000001))");
        invalid_ms("or_i(0,or_c(after(1),multi(2,A,B,C)))");
        invalid_ms("nsc:or_c(after(500000001),multi(2,A,B,C))");
        invalid_ms("andor(pk_k(A),multi(2,B,C,D),multi(2,E,F,G))");
        invalid_ms("ns:or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:or_b(multi(2,A,B,C),after(1))");
        invalid_ms("or_b(after(1),multi(2,A,B,C))");
        invalid_ms("na:after(1)");
        invalid_ms("or_d(s:or_d(after(1),after(1)),after(1))");
        invalid_ms("or_b(pk_k(A),v:multi(2,B,C,D))");
        invalid_ms("thresh(2,pk_k(A),after(1),after(1))");
        invalid_ms("or_b(a:pk_k(A),after(1))");
        invalid_ms("or_d(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("c:or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_i(0,after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_d(after(500000001),c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("thresh(2,n:after(1),after(1),multi(2,A,B,C))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:pk_k(A))");
        invalid_ms("thresh(2,a:pk_h(A),multi(2,B,C,D),after(1))");
        invalid_ms("or_d(svs:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))");
        invalid_ms("or_d(or_d(after(1),after(500000001)),after(1))");
        invalid_ms("or_d(pk_h(A),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("cs:multi(2,A,B,C)");
        invalid_ms("or_b(or_b(multi(2,A,B,C),after(1)),multi(2,D,E,F))");
        invalid_ms("or_b(multi(2,A,B,C),c:after(500000001))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,s:or_b(multi(2,A,B,C),after(1)))");
        invalid_ms("nvs:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A)))");
        invalid_ms("j:or_d(after(500000001),after(500000001))");
        invalid_ms("or_b(multi(2,A,B,C),or_d(multi(2,D,E,F),or_c(after(1),multi(2,G,I,J))))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),s:or_b(after(500000001),after(500000001)))");
        invalid_ms("or_d(or_b(multi(2,A,B,C),after(1)),pk_h(D))");
        invalid_ms("thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:after(1))");
        invalid_ms("or_d(after(500000001),sc:pk_k(A))");
        invalid_ms("or_b(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_d(c:after(500000001),pk_k(A)),pk_h(B))");
        invalid_ms("or_d(c:or_d(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:or_c(thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1)),after(1)),pk_k(A),after(1))");
        invalid_ms("or_d(ac:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("c:or_b(s:after(500000001),multi(2,A,B,C))");
        invalid_ms("or_b(after(1),c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_b(thresh(2,multi(2,A,B,C),after(500000001),after(500000001)),multi(2,D,E,F))");
        invalid_ms("js:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_d(after(1),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,and_b(and_v(or_d(after(500000001),after(500000001)),after(500000001)),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_b(pk_h(A),multi(2,B,C,D))");
        invalid_ms("or_b(pk_k(A),thresh(2,after(1),after(1),pk_k(B)))");
        invalid_ms("ns:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(500000001))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),svs:or_d(after(1),multi(2,A,B,C)))");
        invalid_ms("thresh(2,after(1),pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_c(after(1),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jjcs:after(500000001)");
        invalid_ms("or_d(or_c(after(1),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),d:multi(2,A,B,C))");
        invalid_ms("or_i(0,s:or_c(s:or_c(after(500000001),after(500000001)),after(500000001)))");
        invalid_ms("or_b(s:or_b(pk_h(A),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("nvs:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))");
        invalid_ms("da:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(500000001),after(500000001)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),multi(2,A,B,C))");
        invalid_ms("thresh(2,v:multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(thresh(2,pk_h(A),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(B))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("c:or_d(s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))),pk_k(B))");
        invalid_ms("and_b(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_b(after(1),multi(2,D,E,F)))");
        invalid_ms("cvs:or_b(pk_h(A),after(1))");
        invalid_ms("or_d(pk_k(A),multi(2,B,C,D))");
        invalid_ms("or_d(multi(2,A,B,C),v:or_b(multi(2,D,E,F),after(500000001)))");
        invalid_ms("andor(after(500000001),after(500000001),vs:or_d(after(500000001),after(500000001)))");
        invalid_ms("thresh(2,pk_k(A),thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),after(1))");
        invalid_ms("ns:or_c(after(500000001),after(500000001))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(pk_k(D),after(500000001)))");
        invalid_ms("andor(a:after(500000001),after(500000001),after(500000001))");
        invalid_ms("or_b(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:after(500000001),s:after(500000001))");
        invalid_ms("or_b(multi(2,A,B,C),vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),j:multi(2,D,E,F))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("thresh(2,or_c(after(500000001),after(500000001)),after(500000001),pk_k(A))");
        invalid_ms("thresh(2,after(1),pk_k(A),after(1))");
        invalid_ms("or_d(pk_k(A),thresh(2,after(1),after(1),pk_k(B)))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(c:or_c(after(1),multi(2,A,B,C)),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("cs:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),pk_k(A))");
        invalid_ms("or_b(pk_h(A),and_b(pk_h(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(after(1),j:after(1))");
        invalid_ms("or_i(0,s:or_c(after(1),multi(2,A,B,C)))");
        invalid_ms("or_b(vsc:or_d(after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),j:multi(2,B,C,D))");
        invalid_ms("thresh(2,c:after(1),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001),multi(2,A,B,C))");
        invalid_ms("or_b(or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D))");
        invalid_ms("thresh(2,multi(2,A,B,C),j:multi(2,D,E,F),multi(2,G,I,J))");
        invalid_ms("or_d(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(D))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),after(1))");
        invalid_ms("or_d(or_c(c:after(1),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("jc:andor(after(500000001),after(500000001),pk_k(A))");
        invalid_ms("or_b(multi(2,A,B,C),pk_h(D))");
        invalid_ms("c:thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(a:multi(2,A,B,C),after(1))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),thresh(2,after(1),pk_h(A),after(1)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("cd:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("thresh(2,pk_k(A),pk_k(B),c:after(500000001))");
        invalid_ms("or_d(vs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_b(pk_h(A),sj:multi(2,B,C,D))");
        invalid_ms("or_d(c:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ca:or_c(after(500000001),pk_k(A))");
        invalid_ms("or_d(after(1),cs:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("thresh(2,d:multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ca:and_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,v:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),pk_k(D))");
        invalid_ms("nsc:or_c(after(1),multi(2,A,B,C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_c(after(500000001),multi(2,D,E,F)))");
        invalid_ms("ns:or_b(multi(2,A,B,C),after(1))");
        invalid_ms("or_d(or_c(after(1),pk_k(A)),multi(2,B,C,D))");
        invalid_ms("or_d(or_c(after(500000001),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("c:andor(andor(after(500000001),pk_h(A),c:after(500000001)),c:after(500000001),after(500000001))");
        invalid_ms("or_d(pk_k(A),thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("or_d(a:after(500000001),pk_k(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1))");
        invalid_ms("and_b(v:pk_k(A),after(500000001))");
        invalid_ms("cs:or_c(after(1),multi(2,A,B,C))");
        invalid_ms("or_d(d:or_c(after(1),after(1)),pk_k(A))");
        invalid_ms("or_d(s:or_b(after(1),pk_h(A)),pk_h(B))");
        invalid_ms("thresh(2,pk_k(A),j:multi(2,B,C,D),pk_h(E))");
        invalid_ms("or_d(or_c(after(1),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(c:pk_h(A),after(500000001))");
        invalid_ms("or_d(or_d(after(500000001),after(500000001)),multi(2,A,B,C))");
        invalid_ms("thresh(2,pk_k(A),multi(2,B,C,D),after(500000001))");
        invalid_ms("thresh(2,or_b(d:and_v(after(1),after(1)),after(1)),after(1),pk_k(A))");
        invalid_ms("or_b(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)))");
        invalid_ms("ns:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,and_b(and_b(after(500000001),after(500000001)),c:after(500000001)),multi(2,A,B,C),after(500000001))");
        invalid_ms("jv:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("ca:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(1),v:after(1))");
        invalid_ms("andor(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(pk_k(A),or_b(pk_h(B),after(500000001)))");
        invalid_ms("c:and_v(multi(2,A,B,C),after(1))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:pk_h(A))");
        invalid_ms("jvs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vn:after(1))");
        invalid_ms("ds:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(pk_h(A),pk_k(B))");
        invalid_ms("thresh(2,multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),after(1))");
        invalid_ms("jvs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:pk_k(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:after(500000001))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),dv:pk_h(A))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),after(500000001)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,after(500000001),pk_k(A),s:pk_k(B))");
        invalid_ms("or_d(pk_k(A),or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:or_c(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_k(A))");
        invalid_ms("or_b(after(500000001),cs:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_c(after(500000001),pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:or_c(after(500000001),pk_h(A))");
        invalid_ms("or_d(s:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_k(A))");
        invalid_ms("or_d(or_c(multi(2,A,B,C),or_b(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),pk_k(G))");
        invalid_ms("js:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_i(0,or_c(after(500000001),multi(2,A,B,C))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,or_c(after(1),multi(2,A,B,C)))");
        invalid_ms("andor(pk_h(A),after(500000001),pk_k(B))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(pk_h(A),multi(2,B,C,D))");
        invalid_ms("thresh(2,pk_k(A),or_b(after(500000001),multi(2,B,C,D)),after(500000001))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),j:pk_k(D))");
        invalid_ms("or_d(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_d(s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("and_v(c:andor(after(500000001),after(500000001),and_b(after(500000001),after(500000001))),after(500000001))");
        invalid_ms("or_d(s:or_d(after(500000001),pk_k(A)),after(500000001))");
        invalid_ms("dc:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),v:pk_h(G))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("js:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(c:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(500000001),cvs:or_b(after(500000001),multi(2,A,B,C)))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,sc:or_d(after(1),after(1)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("or_b(pk_k(A),or_b(pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(s:or_d(after(1),after(1)),multi(2,A,B,C))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),pk_k(B))");
        invalid_ms("and_b(pk_k(A),pk_k(B))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(after(1),multi(2,A,B,C))");
        invalid_ms("cvs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("js:after(500000001)");
        invalid_ms("d:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("thresh(2,j:after(500000001),thresh(2,after(500000001),after(500000001),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_c(after(500000001),after(500000001)),pk_k(A))");
        invalid_ms("or_b(after(1),n:pk_k(A))");
        invalid_ms("or_b(thresh(2,after(500000001),after(500000001),cac:after(500000001)),after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:multi(2,A,B,C),pk_h(D))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("thresh(2,s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(s:or_d(after(500000001),after(500000001)),after(1))");
        invalid_ms("or_b(and_v(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D))");
        invalid_ms("j:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,n:pk_k(A),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("thresh(2,after(1),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("and_v(dj:after(1),1)");
        invalid_ms("thresh(2,after(500000001),c:pk_k(A),pk_h(B))");
        invalid_ms("d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(pk_k(A),after(500000001))");
        invalid_ms("d:thresh(2,or_c(v:multi(2,A,B,C),multi(2,D,E,F)),multi(2,G,I,J),multi(2,K,L,M))");
        invalid_ms("or_b(c:pk_k(A),or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(pk_k(A),pk_h(B)))");
        invalid_ms("or_d(pk_h(A),a:pk_k(B))");
        invalid_ms("c:or_c(after(500000001),a:multi(2,A,B,C))");
        invalid_ms("or_d(or_c(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),pk_k(B))");
        invalid_ms("or_d(s:or_d(after(1),after(500000001)),multi(2,A,B,C))");
        invalid_ms("thresh(2,after(1),pk_h(A),s:after(1))");
        invalid_ms("and_b(pk_k(A),v:and_b(and_b(multi(2,B,C,D),pk_k(E)),pk_k(F)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),pk_h(A))");
        invalid_ms("n:or_c(or_b(multi(2,A,B,C),after(500000001)),multi(2,D,E,F))");
        invalid_ms("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),pk_h(A))");
        invalid_ms("or_d(after(500000001),c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(s:or_c(after(1),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(ac:after(1),pk_k(A))");
        invalid_ms("nvs:thresh(2,after(1),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_b(pk_h(A),or_b(after(1),multi(2,B,C,D)))");
        invalid_ms("ds:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),multi(2,B,C,D))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),pk_h(A)))");
        invalid_ms("or_b(vs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_c(after(500000001),pk_k(D)))");
        invalid_ms("d:thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vj:pk_h(A))");
        invalid_ms("c:or_c(a:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,j:after(500000001),after(500000001),pk_h(A))");
        invalid_ms("or_b(and_b(after(500000001),after(500000001)),pk_k(A))");
        invalid_ms("or_d(pk_h(A),after(500000001))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:or_d(after(500000001),pk_k(A))");
        invalid_ms("cda:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1)),pk_h(A))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("andor(after(1),after(1),a:after(1))");
        invalid_ms("or_b(pk_k(A),vs:after(500000001))");
        invalid_ms("dca:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_d(or_d(or_d(after(1),after(1)),after(1)),after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),multi(2,B,C,D))");
        invalid_ms("thresh(2,pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)),pk_k(E))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("c:thresh(2,after(500000001),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(1),multi(2,A,B,C)))");
        invalid_ms("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("or_d(after(500000001),j:after(500000001))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_b(multi(2,A,B,C),v:pk_k(D))");
        invalid_ms("d:or_d(after(500000001),after(500000001))");
        invalid_ms("cs:or_c(after(500000001),after(500000001))");
        invalid_ms("nvs:or_b(after(500000001),pk_h(A))");
        invalid_ms("thresh(2,pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("nvs:or_d(after(1),after(1))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),pk_k(B))");
        invalid_ms("or_d(vs:or_d(after(1),after(1)),after(1))");
        invalid_ms("or_d(pk_k(A),thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))))");
        invalid_ms("andor(multi(2,A,B,C),after(1),pk_h(D))");
        invalid_ms("thresh(2,multi(2,A,B,C),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:after(1),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_b(after(500000001),dc:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),pk_k(B))");
        invalid_ms("thresh(2,s:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(after(1),multi(2,D,E,F)))");
        invalid_ms("thresh(2,or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)),after(500000001))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:or_c(after(1),after(1)))");
        invalid_ms("ns:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vsvs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("and_b(pk_h(A),multi(2,B,C,D))");
        invalid_ms("c:thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("dj:pk_h(A)");
        invalid_ms("or_d(vs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),pk_k(B))");
        invalid_ms("or_d(pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,vs:or_b(pk_h(A),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(500000001),after(500000001)))");
        invalid_ms("thresh(2,pk_h(A),pk_h(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),after(1))");
        invalid_ms("thresh(2,c:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("jj:or_c(after(500000001),multi(2,A,B,C))");
        invalid_ms("c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),s:or_d(or_c(after(1),multi(2,B,C,D)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),svs:thresh(2,pk_k(A),multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,or_c(after(500000001),multi(2,A,B,C)),multi(2,D,E,F),after(500000001))");
        invalid_ms("ns:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("j:or_b(multi(2,A,B,C),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,multi(2,A,B,C)))");
        invalid_ms("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("dv:multi(2,A,B,C)");
        invalid_ms("and_b(vc:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(pk_k(A),or_b(pk_k(B),after(500000001)))");
        invalid_ms("dscs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("jc:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_b(vsvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(after(1),cs:multi(2,A,B,C))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_b(pk_k(A),or_c(after(1),after(1)))");
        invalid_ms("or_b(pk_k(A),pk_k(B))");
        invalid_ms("or_b(pk_k(A),or_b(or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(B)))");
        invalid_ms("thresh(2,or_d(after(1),after(1)),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(500000001))");
        invalid_ms("or_b(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("c:or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),a:pk_h(B))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),pk_h(B))");
        invalid_ms("d:or_b(after(1),after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),v:after(500000001))");
        invalid_ms("or_d(thresh(2,after(1),after(1),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_d(and_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(B))");
        invalid_ms("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),multi(2,A,B,C))");
        invalid_ms("or_i(0,s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)))");
        invalid_ms("or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(j:multi(2,A,B,C),s:after(500000001))");
        invalid_ms("or_d(sc:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(after(500000001),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_d(a:or_d(or_d(after(500000001),after(500000001)),after(500000001)),after(500000001))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,pk_k(A),s:or_d(after(1),multi(2,B,C,D)),multi(2,E,F,G))");
        invalid_ms("or_d(c:pk_h(A),c:after(1))");
        invalid_ms("or_d(or_c(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(vs:or_b(or_b(multi(2,A,B,C),after(500000001)),pk_h(D)),multi(2,E,F,G))");
        invalid_ms("or_d(s:after(1),pk_k(A))");
        invalid_ms("or_d(or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(a:after(500000001),after(500000001))");
        invalid_ms("or_b(pk_h(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_h(A),multi(2,B,C,D),c:after(1))");
        invalid_ms("or_d(s:or_b(after(1),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jv:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("ds:or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:pk_k(A))");
        invalid_ms("or_d(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(v:pk_k(A),s:after(500000001))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(multi(2,A,B,C),vs:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:thresh(2,pk_k(A),multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,or_c(after(1),multi(2,A,B,C)),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),thresh(2,multi(2,A,B,C),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,B,C,D))");
        invalid_ms("or_d(s:or_c(after(1),after(1)),pk_h(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_k(D),after(1))");
        invalid_ms("or_b(or_b(after(500000001),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_d(after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_d(pk_k(A),or_b(pk_h(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:or_d(after(500000001),after(500000001)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,pk_k(A),after(1),vs:thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),pk_k(A))");
        invalid_ms("or_b(after(1),n:pk_h(A))");
        invalid_ms("d:andor(after(1),jjc:and_v(pk_h(A),after(1)),after(1))");
        invalid_ms("or_d(c:or_d(after(500000001),pk_h(A)),pk_h(B))");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D)))");
        invalid_ms("or_d(s:or_c(after(1),after(1)),after(1))");
        invalid_ms("or_b(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_c(after(1),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cv:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),multi(2,B,C,D))");
        invalid_ms("or_d(j:multi(2,A,B,C),pk_k(D))");
        invalid_ms("thresh(2,c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),after(500000001))");
        invalid_ms("or_b(c:or_b(after(500000001),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_b(s:pk_k(A),after(500000001))");
        invalid_ms("nvsns:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("j:and_v(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_d(jc:pk_k(A),pk_k(B))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_b(vsvs:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D))");
        invalid_ms("thresh(2,after(1),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),multi(2,B,C,D))");
        invalid_ms("or_b(or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("cs:thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("ca:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(after(500000001),1)");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),v:multi(2,B,C,D))");
        invalid_ms("or_d(after(500000001),c:pk_k(A))");
        invalid_ms("or_d(thresh(2,after(1),after(1),pk_h(A)),pk_h(B))");
        invalid_ms("or_b(pk_k(A),pk_k(B))");
        invalid_ms("or_i(0,s:or_b(pk_h(A),after(1)))");
        invalid_ms("or_b(pk_k(A),multi(2,B,C,D))");
        invalid_ms("or_i(0,s:or_c(after(1),multi(2,A,B,C)))");
        invalid_ms("thresh(2,after(500000001),or_c(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(1),n:after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),pk_k(A))");
        invalid_ms("or_d(or_c(after(500000001),pk_k(A)),multi(2,B,C,D))");
        invalid_ms("or_b(pk_k(A),after(500000001))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(pk_h(D),or_c(after(1),after(1))))");
        invalid_ms("d:or_c(after(1),multi(2,A,B,C))");
        invalid_ms("thresh(2,after(1),multi(2,A,B,C),a:after(1))");
        invalid_ms("cs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("js:or_c(after(1),after(1))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C))");
        invalid_ms("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),multi(2,B,C,D))");
        invalid_ms("or_d(a:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:after(1)");
        invalid_ms("or_b(j:multi(2,A,B,C),pk_h(D))");
        invalid_ms("andor(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(1))");
        invalid_ms("js:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))");
        invalid_ms("j:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_i(0,s:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),multi(2,B,C,D)))");
        invalid_ms("or_d(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("jd:multi(2,A,B,C)");
        invalid_ms("or_b(multi(2,A,B,C),or_b(pk_h(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))))");
        invalid_ms("jd:multi(2,A,B,C)");
        invalid_ms("cvs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("c:thresh(2,after(1),pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("cs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(v:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),multi(2,B,C,D))");
        invalid_ms("or_i(0,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("cs:thresh(2,after(500000001),pk_k(A),pk_k(B))");
        invalid_ms("or_d(multi(2,A,B,C),pk_h(D))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("cs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jd:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_b(pk_k(A),after(1))");
        invalid_ms("nvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(vs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:or_c(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sj:multi(2,A,B,C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001),multi(2,A,B,C))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001))");
        invalid_ms("or_b(or_c(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(j:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("n:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("or_b(or_b(vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jns:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("or_b(s:or_d(after(500000001),after(500000001)),pk_k(A))");
        invalid_ms("and_v(a:after(500000001),after(500000001))");
        invalid_ms("and_v(pk_k(A),after(1))");
        invalid_ms("thresh(2,d:multi(2,A,B,C),after(1),multi(2,D,E,F))");
        invalid_ms("or_d(d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_k(A),or_c(after(1),multi(2,B,C,D)),pk_h(E))");
        invalid_ms("thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_b(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))");
        invalid_ms("ca:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_d(or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(1),cs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(1),pk_k(D))");
        invalid_ms("or_d(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("dv:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:or_d(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_h(A),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_d(vs:or_d(after(500000001),after(500000001)),after(500000001)),after(500000001))");
        invalid_ms("j:and_b(after(1),after(1))");
        invalid_ms("or_d(or_c(or_d(after(1),pk_k(A)),pk_k(B)),after(500000001))");
        invalid_ms("ns:or_b(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_i(0,s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),c:after(1))");
        invalid_ms("or_d(after(1),a:after(1))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),ca:after(1))");
        invalid_ms("c:or_c(or_b(multi(2,A,B,C),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_b(multi(2,A,B,C),after(500000001)),pk_h(D))");
        invalid_ms("or_i(0,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),cs:after(1))");
        invalid_ms("or_d(pk_h(A),or_b(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),v:or_i(or_d(pk_h(B),after(500000001)),0))");
        invalid_ms("or_b(multi(2,A,B,C),cs:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cvsc:or_b(multi(2,A,B,C),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(c:after(500000001),j:after(500000001))");
        invalid_ms("nv:multi(2,A,B,C)");
        invalid_ms("or_b(or_c(after(500000001),after(500000001)),multi(2,A,B,C))");
        invalid_ms("js:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_b(pk_h(A),s:or_b(after(1),pk_h(B)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(pk_k(A),after(500000001)),after(500000001))");
        invalid_ms("thresh(2,multi(2,A,B,C),j:multi(2,D,E,F),multi(2,G,I,J))");
        invalid_ms("thresh(2,d:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))");
        invalid_ms("thresh(2,c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001))");
        invalid_ms("or_d(pk_h(A),or_d(after(1),after(1)))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),j:pk_k(G))");
        invalid_ms("n:or_d(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_b(or_b(pk_k(A),j:pk_k(B)),pk_k(C))");
        invalid_ms("thresh(2,c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),after(500000001))");
        invalid_ms("or_b(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("and_v(pk_k(A),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),av:after(500000001))");
        invalid_ms("c:or_c(after(500000001),pk_k(A))");
        invalid_ms("or_d(j:pk_h(A),pk_h(B))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001)))");
        invalid_ms("or_d(s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_i(or_i(0,c:after(1)),0)");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:pk_k(A))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),pk_h(B))");
        invalid_ms("and_b(pk_k(A),v:after(500000001))");
        invalid_ms("cs:thresh(2,pk_h(A),multi(2,B,C,D),after(1))");
        invalid_ms("or_d(pk_k(A),thresh(2,after(500000001),multi(2,B,C,D),after(500000001)))");
        invalid_ms("thresh(2,v:multi(2,A,B,C),multi(2,D,E,F),pk_h(G))");
        invalid_ms("or_d(after(500000001),c:or_b(after(500000001),multi(2,A,B,C)))");
        invalid_ms("or_b(pk_h(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),or_c(after(1),after(1)))");
        invalid_ms("and_v(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("cs:or_c(after(1),pk_k(A))");
        invalid_ms("n:or_c(after(500000001),after(500000001))");
        invalid_ms("and_b(vn:after(1),pk_k(A))");
        invalid_ms("or_d(pk_k(A),cs:or_d(after(1),after(1)))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:pk_k(D))");
        invalid_ms("or_b(multi(2,A,B,C),acs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(1),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("thresh(2,pk_k(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))");
        invalid_ms("nv:and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("d:or_d(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:after(500000001))");
        invalid_ms("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),pk_k(A))");
        invalid_ms("or_b(or_d(after(1),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_d(vs:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("n:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("c:or_b(pk_h(A),after(1))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("c:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("ca:or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(a:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_b(vn:pk_h(A),after(1))");
        invalid_ms("or_b(pk_k(A),or_b(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(cv:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("c:or_c(a:after(500000001),pk_k(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("nsvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(after(500000001),after(500000001))");
        invalid_ms("or_b(pk_h(A),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(after(500000001),or_b(after(500000001),after(500000001)))");
        invalid_ms("cv:andor(after(500000001),pk_k(A),after(500000001))");
        invalid_ms("or_i(0,and_b(after(1),after(1)))");
        invalid_ms("jn:pk_h(A)");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("c:or_d(after(500000001),multi(2,A,B,C))");
        invalid_ms("thresh(2,pk_k(A),multi(2,B,C,D),s:after(500000001))");
        invalid_ms("csvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:pk_k(A)),after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),pk_h(A)),multi(2,B,C,D))");
        invalid_ms("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),a:multi(2,D,E,F))");
        invalid_ms("or_d(multi(2,A,B,C),vs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),after(500000001))");
        invalid_ms("ns:thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("and_v(c:pk_k(A),multi(2,B,C,D))");
        invalid_ms("or_d(or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(B))");
        invalid_ms("or_d(or_b(pk_h(A),after(500000001)),pk_k(B))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001)),multi(2,A,B,C))");
        invalid_ms("or_b(multi(2,A,B,C),pk_k(D))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("djs:multi(2,A,B,C)");
        invalid_ms("or_d(after(1),c:or_c(after(1),pk_h(A)))");
        invalid_ms("d:or_c(after(500000001),multi(2,A,B,C))");
        invalid_ms("ds:after(1)");
        invalid_ms("or_b(or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(after(500000001),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))");
        invalid_ms("or_d(svs:or_b(after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),multi(2,A,B,C)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("or_b(or_c(after(1),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(thresh(2,after(500000001),multi(2,A,B,C),after(500000001)),pk_k(D))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_b(multi(2,A,B,C),or_c(or_b(after(500000001),multi(2,D,E,F)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("ja:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("thresh(2,after(500000001),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(after(1),pk_k(A)),multi(2,B,C,D))");
        invalid_ms("and_v(a:after(500000001),andor(or_c(a:after(500000001),a:after(500000001)),after(500000001),a:after(500000001)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A)))");
        invalid_ms("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),1)");
        invalid_ms("or_d(svs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),after(1))");
        invalid_ms("or_d(c:after(500000001),multi(2,A,B,C))");
        invalid_ms("c:or_d(s:after(500000001),multi(2,A,B,C))");
        invalid_ms("thresh(2,s:after(1),pk_k(A),multi(2,B,C,D))");
        invalid_ms("and_b(after(500000001),pk_k(A))");
        invalid_ms("ns:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(and_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_d(and_b(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),thresh(2,after(500000001),pk_k(A),after(500000001)))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(after(1),pk_k(D)))");
        invalid_ms("or_b(pk_h(A),s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(B)))");
        invalid_ms("andor(pk_k(A),j:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),after(500000001)))");
        invalid_ms("or_b(and_b(after(500000001),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("thresh(2,after(500000001),c:after(500000001),pk_k(A))");
        invalid_ms("n:or_c(after(1),multi(2,A,B,C))");
        invalid_ms("or_b(multi(2,A,B,C),n:after(500000001))");
        invalid_ms("andor(after(1),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_b(and_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_b(scvs:or_b(after(500000001),pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(1),multi(2,A,B,C),j:multi(2,D,E,F))");
        invalid_ms("or_d(pk_k(A),or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:or_d(after(1),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("thresh(2,pk_h(A),multi(2,B,C,D),s:thresh(2,multi(2,E,F,G),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(I)))");
        invalid_ms("or_b(vs:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),v:pk_h(D))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(500000001))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(cs:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,or_c(or_b(after(500000001),multi(2,A,B,C)),after(500000001)))");
        invalid_ms("jd:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),c:or_b(multi(2,D,E,F),after(1)))");
        invalid_ms("cs:or_d(after(500000001),pk_h(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,pk_h(A),after(1),c:after(1))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),after(1))");
        invalid_ms("c:or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),multi(2,A,B,C))");
        invalid_ms("and_b(or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_b(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_d(after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_b(pk_h(A),a:after(1))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(vsvs:or_c(after(500000001),after(500000001)),pk_k(A))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),after(500000001))");
        invalid_ms("na:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_v(a:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,c:after(1),pk_k(A),pk_h(B))");
        invalid_ms("or_b(s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(js:after(1),after(1))");
        invalid_ms("n:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_d(after(500000001),pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),vsvs:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(s:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),js:after(500000001))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:multi(2,B,C,D))");
        invalid_ms("or_b(multi(2,A,B,C),v:pk_h(D))");
        invalid_ms("or_d(or_d(c:after(500000001),pk_h(A)),pk_h(B))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),after(500000001))");
        invalid_ms("or_b(or_d(after(1),after(1)),pk_h(A))");
        invalid_ms("js:or_b(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_d(and_b(pk_k(A),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(pk_k(A),or_d(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_d(after(1),after(1)),pk_k(A))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(or_b(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_b(after(500000001),pk_h(A)),pk_k(B))");
        invalid_ms("ds:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),thresh(2,pk_k(D),after(1),multi(2,E,F,G)),pk_h(I))");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(c:after(500000001),pk_k(A))");
        invalid_ms("and_v(after(1),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(vs:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),multi(2,D,E,F)),pk_k(G))");
        invalid_ms("or_i(0,or_i(a:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),0))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_i(0,vs:or_i(0,or_c(after(1),multi(2,A,B,C))))");
        invalid_ms("c:or_d(after(1),pk_h(A))");
        invalid_ms("ns:thresh(2,multi(2,A,B,C),pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("djc:or_c(after(500000001),ac:pk_h(A))");
        invalid_ms("or_d(av:after(500000001),after(500000001))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(after(500000001),pk_k(D)))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(500000001),d:after(500000001))");
        invalid_ms("thresh(2,or_c(after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))");
        invalid_ms("or_d(v:or_d(or_d(multi(2,A,B,C),s:pk_h(D)),multi(2,E,F,G)),multi(2,I,J,K))");
        invalid_ms("or_b(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("j:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(or_d(multi(2,A,B,C),multi(2,D,E,F)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_i(0,or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),pk_k(A))");
        invalid_ms("thresh(2,j:pk_k(A),after(500000001),after(500000001))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_c(after(500000001),multi(2,D,E,F)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_d(pk_h(A),or_c(after(500000001),after(500000001)))");
        invalid_ms("or_d(and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)))");
        invalid_ms("dc:or_d(after(500000001),after(500000001))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("nsc:andor(after(500000001),after(500000001),0)");
        invalid_ms("ns:or_d(after(500000001),pk_k(A))");
        invalid_ms("cs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(or_c(after(1),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_b(and_b(pk_h(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(B))");
        invalid_ms("d:or_c(after(1),after(1))");
        invalid_ms("js:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("nvs:after(500000001)");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:after(500000001))");
        invalid_ms("ns:or_c(after(1),multi(2,A,B,C))");
        invalid_ms("d:or_b(after(1),pk_h(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),s:multi(2,D,E,F),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_c(after(500000001),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(after(1),cv:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(500000001),jn:after(500000001))");
        invalid_ms("or_d(after(500000001),va:after(500000001))");
        invalid_ms("or_b(pk_k(A),s:thresh(2,after(500000001),after(500000001),pk_k(B)))");
        invalid_ms("or_b(or_d(vs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(1)),pk_k(D))");
        invalid_ms("or_d(sc:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(1),pk_k(A))");
        invalid_ms("d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(or_d(after(1),pk_h(A)),pk_h(B))");
        invalid_ms("or_d(pk_h(A),multi(2,B,C,D))");
        invalid_ms("and_v(s:after(500000001),1)");
        invalid_ms("thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vn:after(500000001))");
        invalid_ms("and_v(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(ja:and_v(after(1),after(1)),after(500000001))");
        invalid_ms("c:or_c(after(500000001),after(500000001))");
        invalid_ms("or_d(pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:after(500000001)))");
        invalid_ms("and_b(s:after(500000001),pk_k(A))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_i(0,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("c:or_d(or_c(after(1),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("or_d(pk_k(A),thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_d(s:or_c(after(1),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(av:and_v(a:or_c(a:after(500000001),a:after(500000001)),a:after(500000001)),after(500000001))");
        invalid_ms("or_b(multi(2,A,B,C),vs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),ca:pk_k(A))");
        invalid_ms("thresh(2,s:after(1),after(1),pk_h(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(s:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))");
        invalid_ms("or_d(d:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(D))");
        invalid_ms("djca:multi(2,A,B,C)");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001),pk_k(A))");
        invalid_ms("java:pk_h(A)");
        invalid_ms("or_b(pk_h(A),or_b(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:or_i(or_c(after(1),pk_k(A)),0)");
        invalid_ms("c:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(500000001),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)))");
        invalid_ms("or_d(a:or_c(after(1),after(1)),a:after(500000001))");
        invalid_ms("dcs:after(500000001)");
        invalid_ms("or_b(after(500000001),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0))");
        invalid_ms("nvs:or_d(after(500000001),pk_h(A))");
        invalid_ms("thresh(2,after(500000001),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(1),andor(after(1),after(1),after(1)))");
        invalid_ms("or_d(vs:after(1),after(1))");
        invalid_ms("ns:or_b(pk_h(A),after(500000001))");
        invalid_ms("cs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(after(500000001),s:thresh(2,multi(2,A,B,C),after(500000001),after(500000001)))");
        invalid_ms("ds:multi(2,A,B,C)");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(pk_h(A),or_b(multi(2,B,C,D),after(500000001))))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(svs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),or_c(after(1),pk_k(D)))");
        invalid_ms("or_b(or_c(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),multi(2,A,B,C))");
        invalid_ms("thresh(2,d:after(1),after(1),pk_h(A))");
        invalid_ms("and_v(after(500000001),c:pk_h(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("or_b(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("c:thresh(2,after(500000001),after(500000001),multi(2,A,B,C))");
        invalid_ms("or_d(after(500000001),d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_d(pk_h(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(a:pk_h(A),after(1))");
        invalid_ms("andor(a:after(1),after(1),or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("n:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("jdn:pk_h(A)");
        invalid_ms("or_d(s:or_d(after(1),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(sc:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_h(A),thresh(2,pk_h(B),after(1),multi(2,C,D,E)))");
        invalid_ms("d:and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_h(A),or_b(after(500000001),pk_h(B)))");
        invalid_ms("and_b(or_d(after(1),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_d(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:multi(2,A,B,C))");
        invalid_ms("or_d(s:after(500000001),cnv:or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(vs:or_c(after(1),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("or_d(or_d(after(500000001),pk_k(A)),pk_h(B))");
        invalid_ms("or_b(s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),multi(2,A,B,C))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:andor(after(500000001),after(500000001),0)");
        invalid_ms("or_b(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(1))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_h(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_k(A),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(cs:after(1),multi(2,A,B,C))");
        invalid_ms("or_d(or_c(after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))),after(1))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_d(pk_k(A),or_b(multi(2,B,C,D),after(1)))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_d(after(1),multi(2,D,E,F)))");
        invalid_ms("or_b(c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_h(D),vs:thresh(2,after(1),after(1),multi(2,E,F,G)))");
        invalid_ms("cs:or_d(after(500000001),pk_k(A))");
        invalid_ms("thresh(2,after(500000001),pk_k(A),multi(2,B,C,D))");
        invalid_ms("or_d(s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(D))");
        invalid_ms("or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(j:multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("thresh(2,c:after(500000001),pk_h(A),multi(2,B,C,D))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:pk_k(A))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),multi(2,A,B,C))");
        invalid_ms("cs:or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ds:after(500000001)");
        invalid_ms("or_b(s:or_c(after(1),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_b(s:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(B))");
        invalid_ms("thresh(2,or_c(after(500000001),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B))");
        invalid_ms("j:or_b(after(500000001),after(500000001))");
        invalid_ms("ds:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("js:or_d(after(500000001),pk_k(A))");
        invalid_ms("or_d(after(500000001),after(500000001))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("ds:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(after(1),v:pk_h(A))");
        invalid_ms("jn:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_b(multi(2,A,B,C),after(500000001))");
        invalid_ms("cvs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("cavsvsv:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("andor(pk_k(A),j:multi(2,B,C,D),0)");
        invalid_ms("c:or_c(after(500000001),c:pk_k(A))");
        invalid_ms("dc:after(1)");
        invalid_ms("or_d(s:thresh(2,pk_h(A),after(1),after(1)),v:multi(2,B,C,D))");
        invalid_ms("and_v(multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),c:after(500000001))");
        invalid_ms("c:and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(1))");
        invalid_ms("or_d(or_d(after(1),after(1)),multi(2,A,B,C))");
        invalid_ms("or_d(after(1),after(1))");
        invalid_ms("or_d(or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_h(A),s:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_c(after(500000001),after(500000001)),pk_h(A))");
        invalid_ms("or_b(after(500000001),c:after(1))");
        invalid_ms("thresh(2,multi(2,A,B,C),c:after(500000001),after(500000001))");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_b(or_c(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(500000001),and_b(after(500000001),after(500000001)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(500000001),multi(2,A,B,C)))");
        invalid_ms("csc:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("cs:or_c(after(500000001),pk_k(A))");
        invalid_ms("d:or_b(vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_k(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),or_c(after(500000001),multi(2,B,C,D)))");
        invalid_ms("or_d(and_v(after(500000001),1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(cs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),after(1))");
        invalid_ms("andor(after(500000001),after(500000001),after(500000001))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(d:after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_b(pk_k(A),or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(j:after(1),after(1)))");
        invalid_ms("thresh(2,pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)),multi(2,C,D,E))");
        invalid_ms("or_d(or_d(after(1),a:after(1)),a:after(1))");
        invalid_ms("or_d(s:or_b(pk_h(A),after(500000001)),multi(2,B,C,D))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),1),pk_k(A))");
        invalid_ms("andor(s:or_d(after(1),after(1)),after(1),after(1))");
        invalid_ms("d:thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(B))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:multi(2,A,B,C),after(1))");
        invalid_ms("d:or_b(after(500000001),after(500000001))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,v:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_c(pk_h(A),or_b(after(500000001),pk_k(B))),pk_h(C))");
        invalid_ms("or_d(or_b(pk_k(A),after(1)),pk_k(B))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("cs:or_d(pk_h(A),after(500000001))");
        invalid_ms("or_d(pk_k(A),or_b(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(a:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(n:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_i(0,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_b(dc:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_b(after(500000001),multi(2,A,B,C))");
        invalid_ms("thresh(2,multi(2,A,B,C),or_b(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("ca:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("thresh(2,after(500000001),n:pk_h(A),after(500000001))");
        invalid_ms("or_b(multi(2,A,B,C),or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(pk_k(A),s:or_c(after(1),multi(2,B,C,D)))");
        invalid_ms("thresh(2,pk_h(A),c:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),pk_k(B))");
        invalid_ms("thresh(2,after(500000001),a:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:thresh(2,multi(2,A,B,C),after(500000001),multi(2,D,E,F))");
        invalid_ms("and_b(after(1),pk_h(A))");
        invalid_ms("or_b(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("nv:or_d(after(500000001),after(500000001))");
        invalid_ms("or_d(after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(vd:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(d:after(1),svs:or_d(after(1),after(1)))");
        invalid_ms("or_b(pk_k(A),or_d(s:or_c(after(500000001),after(500000001)),s:after(500000001)))");
        invalid_ms("or_b(d:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ns:thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1),pk_h(A))");
        invalid_ms("or_b(multi(2,A,B,C),ds:multi(2,D,E,F))");
        invalid_ms("or_b(pk_h(A),c:after(500000001))");
        invalid_ms("c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),multi(2,A,B,C))");
        invalid_ms("andor(multi(2,A,B,C),after(500000001),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,d:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(after(1),pk_k(A))");
        invalid_ms("csvs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("j:thresh(2,after(500000001),multi(2,A,B,C),pk_k(D))");
        invalid_ms("jj:after(500000001)");
        invalid_ms("or_b(s:multi(2,A,B,C),after(500000001))");
        invalid_ms("c:or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:or_b(multi(2,A,B,C),after(1)),multi(2,D,E,F))");
        invalid_ms("or_d(v:pk_k(A),multi(2,B,C,D))");
        invalid_ms("j:pk_h(A)");
        invalid_ms("thresh(2,d:pk_h(A),multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(after(500000001),c:or_d(after(500000001),multi(2,A,B,C)))");
        invalid_ms("n:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),svs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("cac:after(1)");
        invalid_ms("cs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(and_v(multi(2,A,B,C),pk_h(D)),multi(2,E,F,G))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_h(A),j:pk_h(B))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(500000001),multi(2,D,E,F))");
        invalid_ms("thresh(2,pk_k(A),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("j:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),thresh(2,after(500000001),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)),after(500000001)))");
        invalid_ms("ds:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(pk_k(A),s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)))");
        invalid_ms("or_d(or_b(s:multi(2,A,B,C),after(500000001)),pk_k(D))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(pk_k(A),after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),pk_k(D))");
        invalid_ms("and_v(and_v(pk_h(A),after(500000001)),after(500000001))");
        invalid_ms("c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(or_b(after(500000001),multi(2,A,B,C)),j:multi(2,D,E,F))");
        invalid_ms("cvs:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_d(after(500000001),after(500000001))");
        invalid_ms("thresh(2,after(500000001),s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),multi(2,B,C,D))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001),multi(2,A,B,C))");
        invalid_ms("or_d(multi(2,A,B,C),vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)))");
        invalid_ms("or_d(after(500000001),pk_k(A))");
        invalid_ms("thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("jv:after(1)");
        invalid_ms("or_d(pk_k(A),vs:or_c(after(1),multi(2,B,C,D)))");
        invalid_ms("or_d(after(1),j:multi(2,A,B,C))");
        invalid_ms("n:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(after(1),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("dj:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("andor(after(500000001),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_b(s:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(s:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),multi(2,A,B,C)))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_d(svs:or_d(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("cs:or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_d(or_c(v:pk_h(A),pk_h(B)),multi(2,C,D,E))");
        invalid_ms("or_d(vsva:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:thresh(2,multi(2,A,B,C),pk_h(D),after(500000001)),multi(2,E,F,G))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_h(A))");
        invalid_ms("c:or_d(after(1),after(1))");
        invalid_ms("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),pk_h(B))");
        invalid_ms("or_d(or_c(or_c(after(1),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(D))");
        invalid_ms("j:or_c(after(500000001),after(500000001))");
        invalid_ms("or_b(s:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(vs:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("thresh(2,after(500000001),or_b(or_i(0,multi(2,A,B,C)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("j:or_d(after(1),pk_h(A))");
        invalid_ms("dn:after(1)");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001))");
        invalid_ms("or_b(pk_k(A),s:or_c(after(500000001),multi(2,B,C,D)))");
        invalid_ms("or_d(or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_d(after(500000001),after(500000001)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),pk_k(A)))");
        invalid_ms("and_b(multi(2,A,B,C),pk_h(D))");
        invalid_ms("and_v(pk_k(A),or_b(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(or_b(after(1),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0)");
        invalid_ms("or_d(or_c(after(1),pk_k(A)),pk_h(B))");
        invalid_ms("thresh(2,after(1),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),pk_k(B))");
        invalid_ms("or_d(s:or_b(after(500000001),pk_h(A)),multi(2,B,C,D))");
        invalid_ms("thresh(2,multi(2,A,B,C),svs:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_d(svs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(after(500000001),multi(2,A,B,C)))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(j:multi(2,A,B,C),pk_h(D)),multi(2,E,F,G))");
        invalid_ms("or_d(s:multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("jv:multi(2,A,B,C)");
        invalid_ms("or_b(pk_k(A),s:or_b(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("d:and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),multi(2,B,C,D))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_b(or_c(after(500000001),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("thresh(2,v:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))");
        invalid_ms("c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(vs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(vsvs:or_d(after(1),after(1)),after(1))");
        invalid_ms("and_b(after(500000001),or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_i(0,s:or_b(after(1),multi(2,A,B,C)))");
        invalid_ms("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),pk_h(A))");
        invalid_ms("or_b(thresh(2,after(1),after(1),after(1)),after(500000001))");
        invalid_ms("or_b(or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))");
        invalid_ms("cn:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(500000001))");
        invalid_ms("or_b(pk_h(A),s:after(500000001))");
        invalid_ms("or_i(0,or_c(after(500000001),pk_k(A)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(pk_h(A),or_b(pk_h(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))))");
        invalid_ms("and_v(and_v(after(500000001),1),after(500000001))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(j:multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_d(s:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(B))");
        invalid_ms("or_d(thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_k(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_k(D),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,E,F,G)))");
        invalid_ms("or_d(c:or_d(after(500000001),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("or_b(after(500000001),or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("nvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cns:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cjd:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_b(pk_k(A),j:multi(2,B,C,D))");
        invalid_ms("or_d(pk_k(A),thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_d(v:multi(2,A,B,C),s:multi(2,D,E,F))");
        invalid_ms("or_d(s:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),multi(2,D,E,F),after(1))");
        invalid_ms("c:and_v(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),pk_h(A),c:pk_h(B))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vsvs:or_d(after(500000001),after(500000001)))");
        invalid_ms("or_d(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_d(or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A)),pk_k(B))");
        invalid_ms("c:or_d(or_c(after(1),pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),s:thresh(2,after(500000001),multi(2,B,C,D),pk_h(E)))");
        invalid_ms("c:or_c(multi(2,A,B,C),or_c(d:or_b(pk_k(D),after(500000001)),pk_h(E)))");
        invalid_ms("js:or_b(multi(2,A,B,C),after(500000001))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:and_v(or_b(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(and_v(after(1),1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),after(500000001))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(after(1),multi(2,A,B,C)))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_c(after(500000001),multi(2,D,E,F)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("nsv:pk_h(A)");
        invalid_ms("or_b(dv:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(pk_h(A),after(1))");
        invalid_ms("or_d(after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(s:after(1),after(1))");
        invalid_ms("or_b(or_c(after(1),pk_h(A)),pk_h(B))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),pk_k(B))");
        invalid_ms("cs:or_b(pk_h(A),or_c(after(1),after(1)))");
        invalid_ms("c:or_b(multi(2,A,B,C),after(500000001))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("csnvs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(vs:or_c(after(1),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),after(500000001))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("nava:or_c(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("thresh(2,after(500000001),c:pk_k(A),multi(2,B,C,D))");
        invalid_ms("or_d(c:after(1),pk_k(A))");
        invalid_ms("or_b(pk_k(A),or_b(after(500000001),multi(2,B,C,D)))");
        invalid_ms("or_d(or_c(after(500000001),multi(2,A,B,C)),after(500000001))");
        invalid_ms("or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:pk_k(A),after(500000001),multi(2,B,C,D))");
        invalid_ms("or_d(s:pk_k(A),or_c(after(500000001),after(500000001)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("dj:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(and_v(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("or_d(or_c(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(500000001))");
        invalid_ms("thresh(2,pk_k(A),pk_h(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),ns:after(500000001))");
        invalid_ms("jds:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),j:multi(2,D,E,F))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("andor(after(500000001),after(500000001),andor(after(500000001),after(500000001),after(500000001)))");
        invalid_ms("cs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_d(after(1),multi(2,A,B,C)))");
        invalid_ms("jvs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("c:or_c(s:or_d(after(500000001),pk_k(A)),after(500000001))");
        invalid_ms("ns:after(500000001)");
        invalid_ms("thresh(2,after(1),multi(2,A,B,C),s:after(1))");
        invalid_ms("or_d(svs:thresh(2,after(500000001),multi(2,A,B,C),pk_h(D)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),aj:multi(2,A,B,C))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),multi(2,A,B,C))");
        invalid_ms("or_d(j:multi(2,A,B,C),pk_h(D))");
        invalid_ms("n:or_c(after(500000001),pk_h(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("and_b(thresh(2,s:or_d(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),thresh(2,after(1),multi(2,D,E,F),after(1)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:multi(2,A,B,C))");
        invalid_ms("and_v(after(1),c:pk_k(A))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("n:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),or_b(s:multi(2,B,C,D),after(500000001)))");
        invalid_ms("or_d(after(1),c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(and_v(c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(svs:or_b(pk_h(A),after(1)),pk_k(B)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))");
        invalid_ms("or_b(c:or_d(after(1),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ns:or_b(multi(2,A,B,C),after(500000001))");
        invalid_ms("dv:pk_k(A)");
        invalid_ms("ns:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_i(or_c(or_i(or_c(v:pk_h(A),multi(2,B,C,D)),0),multi(2,E,F,G)),0),multi(2,I,J,K))");
        invalid_ms("and_b(pk_h(A),pk_k(B))");
        invalid_ms("and_v(pk_h(A),or_i(0,pk_k(B)))");
        invalid_ms("or_b(pk_h(A),n:or_c(multi(2,B,C,D),thresh(2,pk_k(E),multi(2,F,G,I),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))))");
        invalid_ms("jc:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C))");
        invalid_ms("cs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(pk_h(A),cs:or_c(after(1),multi(2,B,C,D)))");
        invalid_ms("or_b(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(or_i(pk_k(A),0),after(500000001))");
        invalid_ms("thresh(2,multi(2,A,B,C),s:pk_h(D),after(1))");
        invalid_ms("or_d(after(500000001),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001)),pk_k(A))");
        invalid_ms("or_d(ca:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(a:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),s:after(500000001))");
        invalid_ms("d:thresh(2,multi(2,A,B,C),pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("nv:pk_k(A)");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(500000001))");
        invalid_ms("csvs:thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:after(500000001))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(after(500000001),pk_h(A)))");
        invalid_ms("or_d(s:or_c(after(500000001),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_b(s:or_d(after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("thresh(2,or_c(after(500000001),after(500000001)),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),aj:multi(2,A,B,C))");
        invalid_ms("or_d(pk_h(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,after(1),multi(2,A,B,C),c:after(1))");
        invalid_ms("thresh(2,after(500000001),after(500000001),s:multi(2,A,B,C))");
        invalid_ms("or_b(sns:or_b(after(500000001),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("cn:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(1))");
        invalid_ms("or_b(multi(2,A,B,C),vs:or_d(after(1),multi(2,D,E,F)))");
        invalid_ms("or_b(a:after(1),pk_h(A))");
        invalid_ms("or_b(pk_h(A),c:after(1))");
        invalid_ms("or_b(multi(2,A,B,C),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B))");
        invalid_ms("c:or_c(or_d(after(1),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_b(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_b(s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,ac:or_d(after(500000001),after(500000001)))");
        invalid_ms("or_d(after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_b(after(1),multi(2,A,B,C))");
        invalid_ms("and_v(na:after(500000001),or_d(after(500000001),after(500000001)))");
        invalid_ms("ns:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(or_c(after(500000001),pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:thresh(2,after(500000001),after(500000001),multi(2,A,B,C))");
        invalid_ms("or_b(a:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(after(500000001),s:or_d(after(500000001),after(500000001)))");
        invalid_ms("d:andor(or_i(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),0),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(v:pk_h(A),multi(2,B,C,D))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(G)))");
        invalid_ms("thresh(2,j:pk_k(A),multi(2,B,C,D),pk_h(E))");
        invalid_ms("andor(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C),pk_k(D))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(1))");
        invalid_ms("thresh(2,after(500000001),pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("dc:after(500000001)");
        invalid_ms("or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(1),a:multi(2,A,B,C))");
        invalid_ms("or_b(thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(v:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:pk_k(A))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(1))");
        invalid_ms("or_b(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(D))");
        invalid_ms("or_b(and_v(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cac:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_h(A),or_c(after(500000001),multi(2,B,C,D)))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("cscs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_b(multi(2,A,B,C),vsvs:multi(2,D,E,F))");
        invalid_ms("js:thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("thresh(2,after(1),or_c(after(1),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ds:thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))");
        invalid_ms("thresh(2,s:multi(2,A,B,C),after(500000001),after(500000001))");
        invalid_ms("jc:or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_i(0,after(500000001)),after(500000001))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(multi(2,A,B,C),after(500000001))");
        invalid_ms("c:and_v(pk_k(A),pk_k(B))");
        invalid_ms("c:thresh(2,or_c(after(1),after(1)),pk_k(A),after(1))");
        invalid_ms("or_d(s:or_c(after(500000001),pk_h(A)),multi(2,B,C,D))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:or_d(after(500000001),after(500000001)))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:or_d(after(500000001),multi(2,A,B,C)))");
        invalid_ms("thresh(2,j:pk_k(A),multi(2,B,C,D),multi(2,E,F,G))");
        invalid_ms("na:multi(2,A,B,C)");
        invalid_ms("and_b(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("thresh(2,after(500000001),s:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cv:pk_h(A)");
        invalid_ms("or_d(vscs:or_c(after(1),multi(2,A,B,C)),after(1))");
        invalid_ms("cs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_b(or_b(multi(2,A,B,C),after(1)),multi(2,D,E,F))");
        invalid_ms("or_d(v:multi(2,A,B,C),pk_h(D))");
        invalid_ms("thresh(2,after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),or_b(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(thresh(2,after(1),multi(2,A,B,C),pk_h(D)),pk_h(E))");
        invalid_ms("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),pk_k(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(1),after(1)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C))");
        invalid_ms("thresh(2,after(1),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("thresh(2,after(1),after(1),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(after(500000001),pk_k(A)),multi(2,B,C,D))");
        invalid_ms("or_d(or_c(after(1),multi(2,A,B,C)),after(1))");
        invalid_ms("d:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cvn:pk_h(A)");
        invalid_ms("or_b(multi(2,A,B,C),or_b(after(1),multi(2,D,E,F)))");
        invalid_ms("or_d(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("n:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,after(500000001),j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("cvs:thresh(2,multi(2,A,B,C),after(500000001),pk_h(D))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,multi(2,A,B,C),or_b(after(500000001),pk_h(D)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("n:or_c(after(500000001),multi(2,A,B,C))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),v:pk_k(G))");
        invalid_ms("thresh(2,or_c(after(1),after(1)),after(1),pk_k(A))");
        invalid_ms("c:or_b(after(1),or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("d:thresh(2,pk_h(A),pk_k(B),n:after(500000001))");
        invalid_ms("c:thresh(2,a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:after(500000001),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:multi(2,A,B,C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),multi(2,B,C,D))");
        invalid_ms("c:or_i(or_c(after(500000001),pk_k(A)),0)");
        invalid_ms("or_i(0,or_c(c:after(500000001),pk_k(A)))");
        invalid_ms("cva:or_b(pk_h(A),after(500000001))");
        invalid_ms("or_d(or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(B))");
        invalid_ms("or_i(0,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("ja:andor(after(500000001),after(500000001),after(500000001))");
        invalid_ms("or_b(pk_h(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),s:after(1))");
        invalid_ms("and_v(and_b(after(1),after(1)),after(1))");
        invalid_ms("thresh(2,after(1),multi(2,A,B,C),or_c(after(1),after(1)))");
        invalid_ms("or_d(v:pk_h(A),multi(2,B,C,D))");
        invalid_ms("or_d(vs:or_c(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("na:pk_k(A)");
        invalid_ms("or_d(c:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),and_b(after(500000001),a:after(500000001)))");
        invalid_ms("c:or_c(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),after(500000001))");
        invalid_ms("c:or_c(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_d(sj:multi(2,A,B,C),after(500000001))");
        invalid_ms("or_b(j:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jn:or_c(after(500000001),after(500000001))");
        invalid_ms("or_b(or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(500000001),a:or_c(after(1),after(1)))");
        invalid_ms("or_b(pk_k(A),svs:thresh(2,after(1),after(1),pk_k(B)))");
        invalid_ms("or_d(multi(2,A,B,C),vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("or_d(pk_k(A),s:or_b(pk_h(B),after(500000001)))");
        invalid_ms("or_b(pk_k(A),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_d(pk_k(A),or_d(c:after(1),pk_k(B)))");
        invalid_ms("or_d(pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(j:multi(2,D,E,F),after(1)))");
        invalid_ms("or_d(s:or_d(after(1),multi(2,A,B,C)),after(1))");
        invalid_ms("or_d(pk_k(A),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("c:or_c(multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_b(or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D))");
        invalid_ms("or_d(pk_h(A),vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("c:or_c(after(1),pk_h(A))");
        invalid_ms("or_i(0,s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_d(pk_k(A),or_b(pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,c:after(1),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_b(s:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(cs:or_b(after(1),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(1),pk_k(A)),after(1))");
        invalid_ms("andor(after(500000001),c:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:or_d(after(500000001),pk_h(A)),multi(2,B,C,D))");
        invalid_ms("or_d(svs:or_d(after(500000001),pk_h(A)),pk_h(B))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001))");
        invalid_ms("or_d(pk_h(A),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("andor(multi(2,A,B,C),multi(2,D,E,F),vs:after(500000001))");
        invalid_ms("or_b(or_b(after(500000001),pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(thresh(2,after(500000001),after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(pk_k(A),vs:or_d(s:or_b(pk_h(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(a:after(500000001),a:after(500000001))");
        invalid_ms("thresh(2,after(1),s:after(1),multi(2,A,B,C))");
        invalid_ms("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_i(0,n:pk_h(A))");
        invalid_ms("or_b(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(B)))");
        invalid_ms("c:thresh(2,after(1),pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(thresh(2,after(500000001),after(500000001),pk_h(A)),pk_h(B))");
        invalid_ms("cs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_d(after(1),nv:after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(1),after(1)),after(1))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("and_v(s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),1)");
        invalid_ms("c:thresh(2,pk_k(A),after(1),vs:or_d(pk_k(B),thresh(2,after(1),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))))");
        invalid_ms("c:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),c:after(500000001))");
        invalid_ms("or_b(or_b(pk_h(A),after(500000001)),multi(2,B,C,D))");
        invalid_ms("cvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("d:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(s:or_c(after(500000001),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("thresh(2,pk_k(A),c:after(1),after(1))");
        invalid_ms("or_b(pk_h(A),or_b(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_b(after(500000001),after(500000001))");
        invalid_ms("d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_b(after(500000001),pk_h(D)))");
        invalid_ms("or_d(or_d(after(1),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),after(1),multi(2,D,E,F))");
        invalid_ms("or_b(c:or_c(after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:pk_k(A))");
        invalid_ms("d:or_b(after(1),multi(2,A,B,C))");
        invalid_ms("or_b(vs:or_d(after(500000001),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:or_d(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_d(after(1),c:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:andor(after(1),after(1),0)");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),n:pk_h(D))");
        invalid_ms("jj:after(1)");
        invalid_ms("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),pk_k(D))");
        invalid_ms("thresh(2,pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_b(and_v(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))");
        invalid_ms("and_v(pk_k(A),cs:or_c(after(1),after(1)))");
        invalid_ms("thresh(2,after(500000001),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_h(A),multi(2,B,C,D))");
        invalid_ms("j:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),j:pk_h(G))");
        invalid_ms("or_b(thresh(2,multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),ca:multi(2,A,B,C))");
        invalid_ms("or_d(pk_k(A),or_b(pk_k(B),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))))");
        invalid_ms("n:or_d(after(500000001),after(500000001))");
        invalid_ms("cac:pk_k(A)");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_k(A))");
        invalid_ms("js:or_c(after(500000001),pk_k(A))");
        invalid_ms("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(1),multi(2,A,B,C)))");
        invalid_ms("or_b(pk_k(A),or_d(after(500000001),multi(2,B,C,D)))");
        invalid_ms("d:or_b(multi(2,A,B,C),after(1))");
        invalid_ms("n:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:pk_k(A))");
        invalid_ms("and_v(or_d(after(1),after(1)),1)");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),v:after(500000001))");
        invalid_ms("or_b(or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_b(or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(500000001),s:multi(2,D,E,F))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),multi(2,A,B,C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_h(A),or_d(after(1),pk_h(B)))");
        invalid_ms("or_b(thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_d(pk_k(A),s:or_b(pk_h(B),after(1)))");
        invalid_ms("nvs:or_d(after(1),pk_k(A))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(a:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("ds:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:after(1))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),after(500000001)))");
        invalid_ms("or_b(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_k(D),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,multi(2,A,B,C),c:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_d(after(1),pk_k(A)),pk_k(B))");
        invalid_ms("thresh(2,c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),multi(2,A,B,C))");
        invalid_ms("j:and_v(multi(2,A,B,C),or_b(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,after(500000001),after(500000001),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(pk_k(A),after(1)))");
        invalid_ms("ds:thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("ds:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,after(1),j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,thresh(2,after(500000001),pk_k(A),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B))");
        invalid_ms("dcs:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_b(d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),multi(2,D,E,F)),multi(2,G,I,J)))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_d(pk_k(A),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(1))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("cva:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cs:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(1),s:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),or_b(after(1),pk_h(D)),pk_k(E))");
        invalid_ms("cs:thresh(2,after(1),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("and_v(and_v(after(500000001),pk_h(A)),after(500000001))");
        invalid_ms("or_d(s:after(500000001),c:or_b(vs:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("cv:or_i(or_c(after(500000001),after(500000001)),0)");
        invalid_ms("and_v(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_k(A),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:pk_h(A))");
        invalid_ms("or_d(pk_h(A),v:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("cs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("j:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("ca:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(s:or_d(after(500000001),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("dj:and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(j:multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("cs:or_b(after(500000001),after(500000001))");
        invalid_ms("jvs:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(thresh(2,pk_k(A),pk_h(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)))");
        invalid_ms("ds:or_d(after(1),after(1))");
        invalid_ms("or_d(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("nj:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_i(0,or_c(after(500000001),j:multi(2,A,B,C)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:pk_k(A)),after(500000001))");
        invalid_ms("nj:after(500000001)");
        invalid_ms("thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(dv:after(500000001),multi(2,A,B,C))");
        invalid_ms("or_d(or_c(or_d(after(500000001),after(500000001)),after(500000001)),after(500000001))");
        invalid_ms("or_d(n:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),after(500000001))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(s:or_b(multi(2,A,B,C),after(1)),multi(2,D,E,F))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(a:after(1),pk_h(A))");
        invalid_ms("njs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("and_v(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("c:or_c(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_d(multi(2,A,B,C),v:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(pk_h(A),s:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,multi(2,A,B,C),v:pk_h(D),multi(2,E,F,G))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),j:pk_h(G))");
        invalid_ms("or_b(pk_k(A),s:or_b(multi(2,B,C,D),after(500000001)))");
        invalid_ms("or_b(or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_d(pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(a:after(500000001),a:or_c(a:after(500000001),a:after(500000001)))");
        invalid_ms("or_b(d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:thresh(2,multi(2,A,B,C),pk_h(D),after(500000001))");
        invalid_ms("ja:or_b(pk_k(A),after(1))");
        invalid_ms("or_d(j:after(500000001),c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(multi(2,A,B,C),j:pk_h(D))");
        invalid_ms("or_b(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("and_v(and_v(after(500000001),after(500000001)),1)");
        invalid_ms("thresh(2,pk_k(A),after(500000001),after(500000001))");
        invalid_ms("and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),1)");
        invalid_ms("js:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cvd:after(500000001)");
        invalid_ms("thresh(2,pk_k(A),v:pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))");
        invalid_ms("or_d(s:or_b(multi(2,A,B,C),after(1)),after(1))");
        invalid_ms("or_b(s:or_c(after(1),after(1)),pk_h(A))");
        invalid_ms("d:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(vs:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_b(s:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(v:pk_k(A),ava:after(1))");
        invalid_ms("or_b(pk_h(A),or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("n:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,pk_h(A),or_b(multi(2,B,C,D),after(1)),multi(2,E,F,G))");
        invalid_ms("or_b(v:pk_k(A),multi(2,B,C,D))");
        invalid_ms("cs:after(500000001)");
        invalid_ms("ns:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("andor(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B))");
        invalid_ms("or_d(pk_h(A),or_d(after(500000001),j:after(500000001)))");
        invalid_ms("c:or_c(or_c(after(500000001),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(1),vs:or_c(after(1),after(1)),pk_k(A))");
        invalid_ms("or_i(0,or_c(after(500000001),after(1)))");
        invalid_ms("cvs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_b(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),pk_k(D))");
        invalid_ms("or_b(pk_h(A),or_b(or_b(after(1),after(1)),or_d(after(1),after(1))))");
        invalid_ms("or_d(or_c(after(500000001),j:multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("cs:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_c(after(500000001),pk_k(A)),pk_h(B))");
        invalid_ms("thresh(2,after(1),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("cv:pk_k(A)");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vj:multi(2,A,B,C))");
        invalid_ms("or_d(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:pk_k(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,thresh(2,after(500000001),multi(2,A,B,C),after(500000001)),pk_k(D),or_c(after(500000001),after(500000001)))");
        invalid_ms("thresh(2,multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("jjvs:multi(2,A,B,C)");
        invalid_ms("j:or_d(after(1),after(1))");
        invalid_ms("cscs:after(500000001)");
        invalid_ms("or_b(s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),multi(2,B,C,D))");
        invalid_ms("thresh(2,pk_h(A),after(500000001),c:after(500000001))");
        invalid_ms("c:or_i(0,or_c(after(500000001),after(500000001)))");
        invalid_ms("or_b(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)))");
        invalid_ms("andor(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001))");
        invalid_ms("or_b(or_b(after(500000001),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_d(after(1),after(1)))");
        invalid_ms("cs:or_c(pk_k(A),s:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("nvs:or_b(multi(2,A,B,C),after(1))");
        invalid_ms("d:or_c(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_c(after(500000001),multi(2,A,B,C)),c:after(500000001))");
        invalid_ms("or_d(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_d(s:or_c(after(1),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("or_d(a:after(500000001),pk_k(A))");
        invalid_ms("or_b(pk_k(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),pk_h(B))");
        invalid_ms("thresh(2,multi(2,A,B,C),multi(2,D,E,F),s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,G,I,J)))");
        invalid_ms("thresh(2,after(1),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ns:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cva:pk_h(A)");
        invalid_ms("or_d(or_b(after(500000001),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("thresh(2,multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("andor(a:after(1),after(1),after(1))");
        invalid_ms("dj:multi(2,A,B,C)");
        invalid_ms("and_v(v:pk_h(A),multi(2,B,C,D))");
        invalid_ms("or_d(or_c(after(1),multi(2,A,B,C)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("dj:pk_k(A)");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),s:multi(2,B,C,D))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),after(500000001))");
        invalid_ms("cs:or_b(pk_h(A),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("djv:or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_b(multi(2,A,B,C),and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001))");
        invalid_ms("or_d(pk_h(A),or_c(after(1),after(1)))");
        invalid_ms("or_b(or_b(multi(2,A,B,C),s:after(500000001)),multi(2,D,E,F))");
        invalid_ms("or_b(thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_b(pk_h(A),vs:pk_k(B))");
        invalid_ms("c:or_c(after(500000001),pk_k(A))");
        invalid_ms("or_d(and_v(pk_k(A),multi(2,B,C,D)),multi(2,E,F,G))");
        invalid_ms("or_d(multi(2,A,B,C),vs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)))");
        invalid_ms("or_d(d:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("thresh(2,after(1),after(1),multi(2,A,B,C))");
        invalid_ms("c:or_i(0,or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(pk_h(A),or_c(after(500000001),after(500000001)))");
        invalid_ms("thresh(2,pk_k(A),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_k(A),s:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),v:pk_h(D))");
        invalid_ms("thresh(2,pk_h(A),after(500000001),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_i(0,vs:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("ns:or_c(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_d(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:after(500000001))");
        invalid_ms("or_d(or_c(after(500000001),pk_k(A)),pk_k(B))");
        invalid_ms("cvsvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("and_b(and_v(after(1),after(1)),c:after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(1),after(1)))");
        invalid_ms("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(after(1),after(1)))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(500000001),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(n:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),and_b(dvc:after(1),after(1)))");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_d(after(1),multi(2,A,B,C))");
        invalid_ms("or_d(or_c(after(1),n:after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(a:after(1),after(1))");
        invalid_ms("or_b(v:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("j:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),v:multi(2,D,E,F))");
        invalid_ms("or_b(a:or_c(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("or_d(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:multi(2,D,E,F))");
        invalid_ms("and_b(a:after(500000001),after(500000001))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(sc:or_d(after(500000001),after(500000001)),after(1))");
        invalid_ms("thresh(2,pk_k(A),after(500000001),multi(2,B,C,D))");
        invalid_ms("cs:or_b(multi(2,A,B,C),after(500000001))");
        invalid_ms("or_d(after(500000001),s:or_d(after(500000001),after(500000001)))");
        invalid_ms("or_d(or_c(after(500000001),multi(2,A,B,C)),multi(2,D,E,F))");
        invalid_ms("or_b(multi(2,A,B,C),pk_h(D))");
        invalid_ms("or_b(multi(2,A,B,C),va:after(1))");
        invalid_ms("thresh(2,pk_k(A),after(1),pk_k(B))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_d(after(1),after(1)))");
        invalid_ms("cs:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("d:multi(2,A,B,C)");
        invalid_ms("or_d(multi(2,A,B,C),s:or_b(multi(2,D,E,F),after(1)))");
        invalid_ms("or_d(cv:pk_h(A),after(1))");
        invalid_ms("or_d(vsvs:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))");
        invalid_ms("or_d(or_b(pk_h(A),after(1)),pk_k(B))");
        invalid_ms("and_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_h(A),after(500000001))");
        invalid_ms("jd:after(1)");
        invalid_ms("thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),csv:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(c:pk_h(A),after(500000001))");
        invalid_ms("or_b(s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("dada:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),pk_k(B))");
        invalid_ms("dj:pk_h(A)");
        invalid_ms("andor(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ds:or_d(after(500000001),after(500000001))");
        invalid_ms("or_d(vs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)),after(500000001))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:pk_k(A))");
        invalid_ms("and_b(multi(2,A,B,C),after(1))");
        invalid_ms("or_b(j:multi(2,A,B,C),after(500000001))");
        invalid_ms("js:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(D))");
        invalid_ms("ds:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(s:or_b(pk_h(A),after(1)),after(1))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),v:pk_k(D))");
        invalid_ms("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,after(500000001),after(500000001),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(pk_h(A),svs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))");
        invalid_ms("or_b(dv:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_d(c:after(500000001),s:after(500000001))");
        invalid_ms("or_b(thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_h(A),or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D))");
        invalid_ms("or_b(multi(2,A,B,C),sv:pk_k(D))");
        invalid_ms("thresh(2,multi(2,A,B,C),s:multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,or_b(after(500000001),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),after(500000001),after(500000001))");
        invalid_ms("or_b(s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_k(A))");
        invalid_ms("thresh(2,c:after(1),after(1),pk_k(A))");
        invalid_ms("or_b(or_d(or_c(or_c(after(500000001),after(500000001)),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001)),pk_k(A))");
        invalid_ms("or_d(c:after(500000001),after(500000001))");
        invalid_ms("and_v(a:or_c(after(1),after(1)),after(1))");
        invalid_ms("or_d(or_c(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B))),multi(2,C,D,E))");
        invalid_ms("or_d(pk_k(A),or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)),after(500000001)))");
        invalid_ms("or_b(pk_k(A),c:after(1))");
        invalid_ms("cv:multi(2,A,B,C)");
        invalid_ms("or_b(d:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),or_b(after(1),pk_h(B)))");
        invalid_ms("thresh(2,s:after(500000001),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),or_d(pk_h(B),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(C))))");
        invalid_ms("or_d(or_c(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("or_b(multi(2,A,B,C),thresh(2,after(500000001),pk_k(D),pk_k(E)))");
        invalid_ms("jvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_c(multi(2,A,B,C),n:or_c(after(1),after(1)))");
        invalid_ms("or_b(pk_h(A),pk_k(B))");
        invalid_ms("c:or_c(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),j:or_b(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),d:multi(2,A,B,C))");
        invalid_ms("or_d(ajnacs:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_b(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),after(1))");
        invalid_ms("j:thresh(2,after(500000001),multi(2,A,B,C),pk_h(D))");
        invalid_ms("thresh(2,after(1),after(1),c:pk_k(A))");
        invalid_ms("c:or_i(0,or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:or_c(after(500000001),pk_h(A))");
        invalid_ms("thresh(2,pk_h(A),pk_k(B),v:pk_k(C))");
        invalid_ms("thresh(2,multi(2,A,B,C),v:pk_k(D),pk_k(E))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),d:multi(2,D,E,F))");
        invalid_ms("n:or_c(or_c(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A),multi(2,B,C,D))");
        invalid_ms("or_d(pk_k(A),or_b(after(500000001),multi(2,B,C,D)))");
        invalid_ms("jvs:or_b(pk_h(A),after(1))");
        invalid_ms("or_b(s:or_c(after(1),pk_k(A)),pk_k(B))");
        invalid_ms("thresh(2,c:after(1),after(1),thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1)))");
        invalid_ms("and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(after(500000001),after(500000001)))");
        invalid_ms("dac:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),pk_k(A))");
        invalid_ms("or_b(a:after(500000001),multi(2,A,B,C))");
        invalid_ms("j:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("j:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("and_b(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:after(1))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("or_d(s:after(1),multi(2,A,B,C))");
        invalid_ms("cj:after(1)");
        invalid_ms("or_d(vc:pk_h(A),multi(2,B,C,D))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_c(after(500000001),pk_h(D)))");
        invalid_ms("thresh(2,vsc:after(500000001),s:multi(2,A,B,C),d:multi(2,D,E,F))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("or_b(pk_h(A),s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(c:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("na:pk_h(A)");
        invalid_ms("or_d(s:or_c(after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),or_d(after(500000001),multi(2,B,C,D)))");
        invalid_ms("or_d(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(1),c:pk_h(A)))");
        invalid_ms("or_b(or_b(pk_k(A),after(1)),c:pk_h(B))");
        invalid_ms("or_b(d:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),s:or_b(multi(2,A,B,C),after(500000001)),multi(2,D,E,F))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_k(D),or_b(multi(2,E,F,G),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("andor(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),or_b(after(1),pk_k(B)))");
        invalid_ms("or_d(vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),multi(2,B,C,D))");
        invalid_ms("or_d(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("thresh(2,after(500000001),after(500000001),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("and_b(c:after(500000001),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:thresh(2,pk_k(A),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("da:pk_h(A)");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("js:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("and_b(and_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(B))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(after(500000001),multi(2,D,E,F)))");
        invalid_ms("or_d(a:after(500000001),c:after(500000001))");
        invalid_ms("or_d(pk_k(A),or_b(multi(2,B,C,D),after(1)))");
        invalid_ms("or_d(and_b(after(500000001),multi(2,A,B,C)),after(500000001))");
        invalid_ms("dsns:pk_h(A)");
        invalid_ms("cs:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_h(A),s:thresh(2,multi(2,B,C,D),after(1),after(1)))");
        invalid_ms("or_i(0,s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("thresh(2,d:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(vs:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("thresh(2,c:after(1),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("dn:pk_h(A)");
        invalid_ms("or_b(j:multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("and_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(c:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(500000001),nvs:or_b(multi(2,A,B,C),after(500000001)))");
        invalid_ms("cvs:thresh(2,after(500000001),pk_h(A),multi(2,B,C,D))");
        invalid_ms("thresh(2,pk_h(A),j:multi(2,B,C,D),multi(2,E,F,G))");
        invalid_ms("or_b(pk_h(A),pk_h(B))");
        invalid_ms("cs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("dj:pk_h(A)");
        invalid_ms("n:and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("ds:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(or_d(after(1),after(1)),after(1))");
        invalid_ms("or_d(pk_h(A),j:thresh(2,multi(2,B,C,D),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(500000001))");
        invalid_ms("and_b(pk_k(A),or_b(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(pk_h(A),or_b(multi(2,B,C,D),after(1)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),j:multi(2,D,E,F))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:after(1))");
        invalid_ms("thresh(2,pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ca:or_b(a:pk_h(A),after(1))");
        invalid_ms("or_b(after(1),pk_h(A))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),pk_h(D))");
        invalid_ms("or_i(0,vs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("thresh(2,after(1),c:after(1),multi(2,A,B,C))");
        invalid_ms("c:thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("or_d(pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_d(ns:or_c(after(1),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("dj:after(500000001)");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:pk_k(A),pk_h(B))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(500000001),multi(2,A,B,C))");
        invalid_ms("or_b(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:after(500000001)))");
        invalid_ms("or_b(multi(2,A,B,C),or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_b(or_c(after(1),after(1)),after(1))");
        invalid_ms("and_b(pk_k(A),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))");
        invalid_ms("or_b(pk_k(A),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("or_b(or_c(pk_h(A),or_c(after(500000001),multi(2,B,C,D))),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_h(A),j:multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),after(500000001),j:and_v(after(500000001),cv:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(after(500000001),multi(2,D,E,F)))");
        invalid_ms("thresh(2,v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_b(pk_h(A),j:multi(2,B,C,D))");
        invalid_ms("cs:or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(pk_k(A),or_b(after(500000001),pk_k(B))))");
        invalid_ms("or_d(or_c(after(500000001),after(500000001)),after(1))");
        invalid_ms("thresh(2,multi(2,A,B,C),c:pk_k(D),after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),j:multi(2,B,C,D))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:or_c(after(1),pk_h(A)))");
        invalid_ms("thresh(2,pk_k(A),pk_h(B),j:multi(2,C,D,E))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_i(0,or_d(s:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("dscs:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),c:after(500000001)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),after(500000001)))");
        invalid_ms("thresh(2,after(1),j:multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("thresh(2,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001),after(500000001))");
        invalid_ms("n:or_c(after(1),after(1))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))))");
        invalid_ms("thresh(2,after(500000001),pk_h(A),c:pk_k(B))");
        invalid_ms("d:or_b(d:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),s:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(1),after(1),n:pk_h(A))");
        invalid_ms("or_d(s:or_d(after(1),after(1)),pk_k(A))");
        invalid_ms("c:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,sc:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_c(pk_h(A),or_b(after(500000001),pk_h(B))),after(1))");
        invalid_ms("thresh(2,multi(2,A,B,C),c:after(1),after(1))");
        invalid_ms("and_b(after(500000001),multi(2,A,B,C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_c(after(500000001),multi(2,A,B,C)),j:pk_h(D))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(1))");
        invalid_ms("ns:or_c(after(1),pk_k(A))");
        invalid_ms("or_b(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("nj:after(1)");
        invalid_ms("js:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(B))");
        invalid_ms("or_b(cs:or_d(after(1),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_c(after(500000001),after(500000001)),pk_k(A))");
        invalid_ms("or_d(s:or_b(after(500000001),pk_h(A)),after(500000001))");
        invalid_ms("or_d(or_i(after(1),0),after(1))");
        invalid_ms("cvs:or_d(after(500000001),pk_k(A))");
        invalid_ms("or_d(and_v(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(B))");
        invalid_ms("or_d(s:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),pk_k(A)),multi(2,B,C,D))");
        invalid_ms("or_b(s:multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("or_d(or_c(after(1),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_h(A),or_b(pk_k(B),after(500000001)),multi(2,C,D,E))");
        invalid_ms("n:or_c(a:after(500000001),multi(2,A,B,C))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cv:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("thresh(2,a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),after(1))");
        invalid_ms("or_d(or_b(pk_k(A),after(1)),pk_k(B))");
        invalid_ms("thresh(2,pk_k(A),after(1),or_c(after(1),after(1)))");
        invalid_ms("andor(after(500000001),multi(2,A,B,C),c:after(500000001))");
        invalid_ms("or_d(pk_k(A),s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)))");
        invalid_ms("or_d(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(D))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),or_c(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),after(1)))");
        invalid_ms("and_b(after(1),or_d(after(1),pk_k(A)))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D)))");
        invalid_ms("j:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("ns:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(s:or_b(after(500000001),pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_h(A),after(500000001),after(500000001))");
        invalid_ms("or_b(multi(2,A,B,C),thresh(2,multi(2,D,E,F),pk_h(G),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_b(after(1),pk_k(A))");
        invalid_ms("cs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(c:or_c(or_c(pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("d:and_v(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(s:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_d(or_c(after(500000001),after(500000001)),avs:or_c(after(500000001),after(500000001)))");
        invalid_ms("or_d(or_c(after(500000001),multi(2,A,B,C)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("c:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(multi(2,A,B,C),c:or_b(after(500000001),multi(2,D,E,F)))");
        invalid_ms("csj:pk_k(A)");
        invalid_ms("c:or_d(multi(2,A,B,C),or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("j:and_v(after(500000001),after(500000001))");
        invalid_ms("c:or_b(after(500000001),multi(2,A,B,C))");
        invalid_ms("andor(pk_k(A),multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(j:multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_d(after(500000001),pk_h(A))");
        invalid_ms("csvs:thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),s:pk_h(A))");
        invalid_ms("ns:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(1),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_h(A),pk_k(B))");
        invalid_ms("thresh(2,pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)))");
        invalid_ms("js:or_b(multi(2,A,B,C),after(1))");
        invalid_ms("or_d(after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_d(multi(2,A,B,C),s:thresh(2,pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("jv:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),multi(2,B,C,D))");
        invalid_ms("or_b(pk_h(A),after(500000001))");
        invalid_ms("or_b(vs:or_c(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("or_b(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("d:pk_k(A)");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(vs:or_d(after(500000001),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("c:or_b(multi(2,A,B,C),after(1))");
        invalid_ms("or_d(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,after(500000001),after(500000001),pk_k(A))");
        invalid_ms("ns:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),or_b(multi(2,D,E,F),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_h(A),after(1))");
        invalid_ms("j:thresh(2,after(1),after(1),multi(2,A,B,C))");
        invalid_ms("or_d(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("thresh(2,v:after(500000001),multi(2,A,B,C),pk_h(D))");
        invalid_ms("and_b(after(500000001),or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),pk_h(B))");
        invalid_ms("c:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),pk_h(A))");
        invalid_ms("or_b(a:pk_k(A),after(500000001))");
        invalid_ms("thresh(2,after(500000001),pk_k(A),c:after(500000001))");
        invalid_ms("ca:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(s:or_c(after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(500000001),or_b(after(500000001),after(500000001)))");
        invalid_ms("cvs:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("cd:thresh(2,after(500000001),after(500000001),after(500000001))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("andor(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("js:pk_h(A)");
        invalid_ms("c:thresh(2,after(1),pk_k(A),after(1))");
        invalid_ms("d:or_d(or_c(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("and_v(n:after(500000001),1)");
        invalid_ms("thresh(2,or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))");
        invalid_ms("c:or_b(multi(2,A,B,C),after(1))");
        invalid_ms("or_b(s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(v:after(1),after(1))");
        invalid_ms("js:or_b(or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_b(av:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),pk_k(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:or_b(after(500000001),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))");
        invalid_ms("or_b(pk_k(A),s:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(after(500000001),jv:multi(2,A,B,C))");
        invalid_ms("thresh(2,after(500000001),or_c(s:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:after(500000001)),pk_h(A))");
        invalid_ms("cvs:or_d(after(1),multi(2,A,B,C))");
        invalid_ms("or_b(pk_h(A),or_b(s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_b(s:or_b(after(500000001),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("or_b(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(v:pk_k(A),v:pk_h(B))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(1),after(1)))");
        invalid_ms("or_b(pk_k(A),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),cs:multi(2,B,C,D))");
        invalid_ms("or_d(pk_h(A),pk_h(B))");
        invalid_ms("or_i(0,s:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(pk_h(A),or_b(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(vsvs:thresh(2,after(500000001),after(500000001),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("or_b(pk_h(A),or_d(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(s:or_d(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(v:pk_k(A),after(500000001))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_c(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(or_b(j:multi(2,A,B,C),after(500000001)),multi(2,D,E,F))");
        invalid_ms("or_d(a:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cvc:pk_h(A)");
        invalid_ms("or_d(c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_k(A))");
        invalid_ms("and_b(s:after(1),multi(2,A,B,C))");
        invalid_ms("cs:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),after(1))");
        invalid_ms("cs:and_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)),after(500000001))");
        invalid_ms("or_b(thresh(2,after(500000001),multi(2,A,B,C),sn:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),pk_k(D))");
        invalid_ms("csv:pk_k(A)");
        invalid_ms("jc:or_c(after(500000001),after(500000001))");
        invalid_ms("or_d(or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jd:pk_h(A)");
        invalid_ms("cs:or_d(after(1),pk_k(A))");
        invalid_ms("and_v(and_v(pk_h(A),pk_h(B)),after(500000001))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))))");
        invalid_ms("or_b(pk_k(A),or_d(or_c(after(1),after(1)),after(1)))");
        invalid_ms("or_d(vs:or_c(after(500000001),pk_k(A)),after(1))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_i(pk_h(A),0),after(500000001))");
        invalid_ms("or_b(thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),pk_k(B))");
        invalid_ms("or_d(after(1),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),after(500000001),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),after(500000001))");
        invalid_ms("j:thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(vs:after(500000001),after(500000001))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("cvs:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("cs:or_b(after(1),multi(2,A,B,C))");
        invalid_ms("and_v(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jvs:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,after(1),after(1),pk_h(A))");
        invalid_ms("cs:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_i(0,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_b(pk_h(A),s:or_c(vs:or_c(after(500000001),multi(2,B,C,D)),pk_k(E)))");
        invalid_ms("or_b(a:after(1),vc:and_b(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,pk_h(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(multi(2,A,B,C),multi(2,D,E,F)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:or_b(after(500000001),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jvs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("c:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_b(after(500000001),s:pk_k(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(pk_h(A),or_b(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))))");
        invalid_ms("csc:or_d(after(1),pk_h(A))");
        invalid_ms("or_d(pk_k(A),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(pk_h(A),j:multi(2,B,C,D))");
        invalid_ms("d:thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("cav:pk_k(A)");
        invalid_ms("or_b(or_c(after(500000001),after(500000001)),pk_k(A))");
        invalid_ms("andor(pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("cvs:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_i(pk_h(A),0),after(1))");
        invalid_ms("d:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("and_v(after(1),or_d(after(1),a:after(1)))");
        invalid_ms("thresh(2,c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(500000001),after(500000001)),after(500000001))");
        invalid_ms("n:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(pk_k(A),s:or_b(after(500000001),pk_h(B)))");
        invalid_ms("thresh(2,pk_k(A),v:pk_k(B),multi(2,C,D,E))");
        invalid_ms("thresh(2,multi(2,A,B,C),v:pk_k(D),multi(2,E,F,G))");
        invalid_ms("c:or_b(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(or_c(after(500000001),pk_h(A)),multi(2,B,C,D))");
        invalid_ms("j:or_b(pk_h(A),after(1))");
        invalid_ms("or_d(and_v(or_c(c:after(1),after(1)),and_b(after(1),after(1))),after(1))");
        invalid_ms("or_d(s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(j:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ja:or_c(after(500000001),a:after(500000001))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(s:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_b(multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("or_d(a:after(500000001),multi(2,A,B,C))");
        invalid_ms("cs:thresh(2,after(500000001),pk_h(A),after(500000001))");
        invalid_ms("or_d(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001)))");
        invalid_ms("thresh(2,pk_h(A),pk_k(B),after(500000001))");
        invalid_ms("and_v(pk_h(A),or_i(0,after(1)))");
        invalid_ms("thresh(2,d:after(500000001),pk_k(A),after(500000001))");
        invalid_ms("or_d(pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(pk_h(A),s:or_b(after(500000001),pk_h(B)))");
        invalid_ms("js:or_d(after(500000001),multi(2,A,B,C))");
        invalid_ms("or_b(multi(2,A,B,C),or_b(after(500000001),multi(2,D,E,F)))");
        invalid_ms("thresh(2,pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:after(1))");
        invalid_ms("j:and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(500000001))");
        invalid_ms("or_b(s:or_d(after(500000001),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("thresh(2,after(1),pk_k(A),j:multi(2,B,C,D))");
        invalid_ms("thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),vs:or_d(after(500000001),multi(2,A,B,C)))");
        invalid_ms("dv:pk_h(A)");
        invalid_ms("or_b(multi(2,A,B,C),and_v(multi(2,D,E,F),pk_h(G)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(a:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,pk_k(A),pk_h(B),after(1))");
        invalid_ms("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,s:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))");
        invalid_ms("js:thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jv:pk_k(A)");
        invalid_ms("thresh(2,after(500000001),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("or_b(a:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("d:or_b(pk_k(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(B)))");
        invalid_ms("j:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),a:multi(2,D,E,F))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_c(thresh(2,after(500000001),after(500000001),multi(2,D,E,F)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(j:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(multi(2,A,B,C),after(1)),pk_k(D))");
        invalid_ms("andor(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:after(500000001))");
        invalid_ms("or_d(ac:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("or_d(sv:pk_h(A),multi(2,B,C,D))");
        invalid_ms("or_d(c:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))");
        invalid_ms("thresh(2,after(1),after(1),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("thresh(2,s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),svs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_i(0,vs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("dj:after(1)");
        invalid_ms("or_d(after(500000001),c:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,j:multi(2,A,B,C),multi(2,D,E,F),multi(2,G,I,J))");
        invalid_ms("thresh(2,after(1),or_i(pk_k(A),0),after(1))");
        invalid_ms("and_b(a:multi(2,A,B,C),a:after(500000001))");
        invalid_ms("cs:after(1)");
        invalid_ms("or_b(pk_k(A),or_b(after(500000001),multi(2,B,C,D)))");
        invalid_ms("c:or_d(multi(2,A,B,C),after(500000001))");
        invalid_ms("or_d(after(500000001),v:after(500000001))");
        invalid_ms("j:and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(multi(2,A,B,C),or_c(after(500000001),after(500000001)))");
        invalid_ms("c:or_i(0,s:or_b(after(500000001),after(500000001)))");
        invalid_ms("andor(pk_h(A),multi(2,B,C,D),after(1))");
        invalid_ms("or_d(c:after(500000001),or_c(d:after(500000001),vc:after(500000001)))");
        invalid_ms("or_b(pk_k(A),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("or_b(svs:thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),pk_k(A),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),multi(2,A,B,C))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(or_d(after(500000001),multi(2,A,B,C)),pk_h(D))");
        invalid_ms("or_d(or_c(or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1)),after(1))");
        invalid_ms("or_d(a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_b(s:or_d(multi(2,A,B,C),or_c(after(500000001),after(500000001))),multi(2,D,E,F))");
        invalid_ms("or_i(0,ac:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("j:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,s:pk_h(A),after(500000001),multi(2,B,C,D))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),pk_h(B))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))");
        invalid_ms("thresh(2,pk_k(A),a:or_b(after(500000001),or_b(pk_k(B),after(500000001))),pk_h(C))");
        invalid_ms("j:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("c:thresh(2,after(1),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("csc:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("thresh(2,after(1),after(1),a:multi(2,A,B,C))");
        invalid_ms("or_b(pk_k(A),j:multi(2,B,C,D))");
        invalid_ms("or_b(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F),multi(2,G,I,J))");
        invalid_ms("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),ac:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)))");
        invalid_ms("or_b(j:multi(2,A,B,C),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))");
        invalid_ms("or_d(s:after(500000001),multi(2,A,B,C))");
        invalid_ms("or_b(pk_k(A),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(B)))");
        invalid_ms("or_d(s:or_c(s:or_d(after(1),after(1)),after(1)),after(1))");
        invalid_ms("ds:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)");
        invalid_ms("or_d(pk_h(A),c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))");
        invalid_ms("thresh(2,after(1),multi(2,A,B,C),pk_h(D))");
        invalid_ms("jvs:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(after(1),multi(2,A,B,C))");
        invalid_ms("andor(after(500000001),after(500000001),a:after(500000001))");
        invalid_ms("or_d(or_d(after(1),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("and_v(after(500000001),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("or_d(cvs:or_d(after(1),after(1)),after(1))");
        invalid_ms("or_d(or_c(after(1),pk_k(A)),j:multi(2,B,C,D))");
        invalid_ms("or_d(pk_k(A),thresh(2,pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("c:thresh(2,after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("thresh(2,pk_k(A),v:multi(2,B,C,D),pk_k(E))");
        invalid_ms("or_d(pk_k(A),svs:thresh(2,after(500000001),pk_h(B),after(500000001)))");
        invalid_ms("dva:or_c(after(500000001),after(500000001))");
        invalid_ms("or_d(s:multi(2,A,B,C),or_b(multi(2,D,E,F),after(500000001)))");
        invalid_ms("c:or_c(or_i(0,after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_k(A),after(1))");
        invalid_ms("c:or_i(or_c(after(500000001),after(500000001)),0)");
        invalid_ms("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),pk_h(B))");
        invalid_ms("andor(after(1),after(1),da:after(1))");
        invalid_ms("n:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))");
        invalid_ms("cs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("c:andor(multi(2,A,B,C),pk_h(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(vsvs:or_d(after(1),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_d(s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001)))");
        invalid_ms("thresh(2,n:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_d(after(500000001),after(500000001)),pk_k(A))");
        invalid_ms("c:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(500000001),multi(2,A,B,C),s:multi(2,D,E,F))");
        invalid_ms("cscvc:after(1)");
        invalid_ms("or_b(multi(2,A,B,C),svs:or_b(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("cscs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(pk_k(A),or_b(pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_b(pk_k(A),after(500000001)),pk_k(B))");
        invalid_ms("or_d(s:thresh(2,after(500000001),pk_h(A),after(500000001)),pk_k(B))");
        invalid_ms("ds:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,after(1),pk_h(A),after(1))");
        invalid_ms("cv:andor(after(500000001),after(500000001),pk_h(A))");
        invalid_ms("cs:thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(after(1),avsjc:or_b(pk_h(A),after(1)))");
        invalid_ms("thresh(2,c:pk_k(A),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(thresh(2,multi(2,A,B,C),pk_h(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,E,F,G))");
        invalid_ms("or_d(after(1),pk_h(A))");
        invalid_ms("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("jvs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("or_d(pk_k(A),v:or_d(after(500000001),multi(2,B,C,D)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
        invalid_ms("c:or_c(after(500000001),pk_h(A))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:thresh(2,after(500000001),multi(2,A,B,C),pk_h(D)))");
        invalid_ms("ns:thresh(2,after(500000001),multi(2,A,B,C),after(500000001))");
        invalid_ms("or_d(va:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(c:after(500000001),pk_h(D)))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),n:after(500000001))");
        invalid_ms("or_b(after(1),c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(after(1),after(1)),multi(2,A,B,C))");
        invalid_ms("or_b(pk_k(A),after(500000001))");
        invalid_ms("thresh(2,s:after(1),multi(2,A,B,C),after(1))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(v:pk_k(A),multi(2,B,C,D))");
        invalid_ms("or_d(pk_h(A),pk_h(B))");
        invalid_ms("da:after(1)");
        invalid_ms("andor(and_b(after(500000001),after(500000001)),after(500000001),and_v(after(500000001),after(500000001)))");
        invalid_ms("d:thresh(2,multi(2,A,B,C),after(500000001),vs:thresh(2,after(500000001),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("d:or_c(pk_k(A),or_c(or_c(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_d(or_d(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:pk_h(A))");
        invalid_ms("thresh(2,multi(2,A,B,C),s:multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("ds:pk_k(A)");
        invalid_ms("or_d(pk_h(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_k(A))");
        invalid_ms("or_b(c:after(1),j:after(1))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(multi(2,A,B,C),or_b(pk_k(D),after(500000001))))");
        invalid_ms("thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:after(500000001))");
        invalid_ms("cs:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("jc:after(1)");
        invalid_ms("or_b(or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(D))");
        invalid_ms("and_b(pk_h(A),pk_h(B))");
        invalid_ms("and_b(and_b(after(500000001),and_b(after(500000001),and_b(after(500000001),after(500000001)))),pk_k(A))");
        invalid_ms("or_i(0,vs:or_b(pk_h(A),after(500000001)))");
        invalid_ms("c:thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("n:or_c(after(500000001),pk_k(A))");
        invalid_ms("thresh(2,after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:after(1))");
        invalid_ms("or_d(pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(B)))");
        invalid_ms("or_b(vj:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(s:after(500000001),after(500000001))");
        invalid_ms("or_b(multi(2,A,B,C),multi(2,D,E,F))");
        invalid_ms("or_b(pk_h(A),or_i(or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),0))");
        invalid_ms("or_b(s:or_c(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(multi(2,A,B,C),s:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(D)))");
        invalid_ms("or_d(multi(2,A,B,C),or_c(after(1),pk_k(D)))");
        invalid_ms("csvs:or_d(after(500000001),pk_h(A))");
        invalid_ms("or_d(or_b(after(1),multi(2,A,B,C)),pk_k(D))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("cs:or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),or_d(after(500000001),multi(2,D,E,F)))");
        invalid_ms("or_i(0,or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_b(or_d(multi(2,A,B,C),after(1)),pk_h(D))");
        invalid_ms("c:thresh(2,multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),s:or_d(after(500000001),after(500000001)))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A)),multi(2,B,C,D))");
        invalid_ms("or_d(c:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("or_i(0,or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(s:or_b(pk_h(A),after(1)),pk_k(B))");
        invalid_ms("or_d(thresh(2,after(500000001),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(A))");
        invalid_ms("cn:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))");
        invalid_ms("or_b(pk_k(A),or_d(vs:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D)))");
        invalid_ms("or_b(or_d(pk_k(A),vs:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))),pk_h(B))");
        invalid_ms("jsc:or_c(after(1),after(1))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(multi(2,A,B,C),vsvs:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(D)))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("and_v(and_b(after(500000001),after(500000001)),1)");
        invalid_ms("j:or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:multi(2,A,B,C))");
        invalid_ms("cs:or_c(after(500000001),pk_h(A))");
        invalid_ms("or_b(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A))");
        invalid_ms("or_b(s:or_b(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_h(A),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(multi(2,A,B,C),na:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(after(1),or_i(0,pk_h(A)))");
        invalid_ms("and_v(pk_k(A),after(500000001))");
        invalid_ms("jcsc:thresh(2,pk_k(A),after(500000001),after(500000001))");
        invalid_ms("j:and_v(d:multi(2,A,B,C),1)");
        invalid_ms("or_b(after(500000001),c:or_b(after(500000001),after(500000001)))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(1),j:multi(2,D,E,F))");
        invalid_ms("and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),pk_k(B))");
        invalid_ms("c:andor(multi(2,A,B,C),after(500000001),0)");
        invalid_ms("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),c:after(1))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("thresh(2,v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),multi(2,B,C,D))");
        invalid_ms("jdj:multi(2,A,B,C)");
        invalid_ms("c:thresh(2,or_d(after(500000001),after(500000001)),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,pk_h(A),multi(2,B,C,D),after(500000001))");
        invalid_ms("thresh(2,after(500000001),pk_k(A),j:after(500000001))");
        invalid_ms("thresh(2,multi(2,A,B,C),pk_h(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(cs:or_c(after(1),after(1)),multi(2,A,B,C))");
        invalid_ms("or_b(vs:after(500000001),multi(2,A,B,C))");
        invalid_ms("d:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:pk_k(A))");
        invalid_ms("thresh(2,pk_h(A),j:multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(A))");
        invalid_ms("or_d(pk_k(A),s:or_d(after(500000001),after(500000001)))");
        invalid_ms("or_d(thresh(2,pk_h(A),after(500000001),after(500000001)),pk_k(B))");
        invalid_ms("or_d(or_c(after(500000001),after(500000001)),j:after(500000001))");
        invalid_ms("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(multi(2,A,B,C),after(1)))");
        invalid_ms("or_d(or_c(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))");
        invalid_ms("thresh(2,pk_k(A),pk_k(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_b(pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))");
        invalid_ms("js:or_b(after(1),pk_h(A))");
        invalid_ms("and_v(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),cs:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("c:or_b(after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_b(or_b(multi(2,A,B,C),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,multi(2,A,B,C),after(1),pk_k(D))");
        invalid_ms("thresh(2,pk_h(A),multi(2,B,C,D),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(pk_h(A),s:or_b(pk_h(B),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))");
        invalid_ms("or_b(c:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(after(1),after(1)))");
        invalid_ms("or_d(or_c(after(1),multi(2,A,B,C)),after(1))");
        invalid_ms("d:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))");
        invalid_ms("thresh(2,j:pk_k(A),pk_k(B),pk_k(C))");
        invalid_ms("or_d(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))");
        invalid_ms("thresh(2,a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),pk_k(D))");
        invalid_ms("or_b(or_b(multi(2,A,B,C),after(500000001)),multi(2,D,E,F))");
        invalid_ms("c:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A))");
    }
    #[test]
    #[cfg_attr(feature="cargo-fmt", rustfmt_skip)]
    fn mall_8f1e8_tests_from_alloy() {
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "B");
        ms_test("or_d(c:pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(1),after(1))", "oBf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))))", "Bfu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "B");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)),multi(2,B,C,D))", "Bdue");
        ms_test("andor(j:multi(2,A,B,C),multi(2,D,E,F),after(500000001))", "Bf");
        ms_test("c:and_v(v:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),pk_h(A))", "Bsu");
        ms_test("or_b(c:pk_h(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sc:pk_k(D))", "Bdue");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "B");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("or_d(andor(multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_i(0,multi(2,A,B,C)))", "Bd");
        ms_test("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),multi(2,A,B,C))", "Bdue");
        ms_test("or_d(or_i(multi(2,A,B,C),0),after(500000001))", "Bf");
        ms_test("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),1)", "Bfu");
        ms_test("and_v(v:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bu");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:and_v(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A)))", "Bu");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0))", "Bdue");
        ms_test("or_b(or_d(multi(2,A,B,C),multi(2,D,E,F)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(D)))", "Bdue");
        ms_test("andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_d(j:multi(2,A,B,C),j:multi(2,D,E,F))", "Bdus");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("and_b(and_b(after(1),a:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1))),a:after(1))", "Bfu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),n:after(500000001)))", "Bfu");
        ms_test("or_d(multi(2,A,B,C),or_b(multi(2,D,E,F),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(1))", "B");
        ms_test("or_b(jc:pk_k(A),a:multi(2,B,C,D))", "sdueB");
        ms_test("n:or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bfu");
        ms_test("or_d(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C),after(500000001))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,multi(2,A,B,C)),after(1))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_k(A))", "Bdu");
        ms_test("andor(dv:after(500000001),after(500000001),after(500000001))", "oBf");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0)", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),after(500000001))", "Bf");
        ms_test("c:andor(j:multi(2,A,B,C),pk_h(D),pk_h(E))", "sdueB");
        ms_test("thresh(2,multi(2,A,B,C),a:multi(2,D,E,F),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdus");
        ms_test("or_b(jc:pk_k(A),a:multi(2,B,C,D))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),multi(2,D,E,F)),after(1))", "B");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_d(multi(2,A,B,C),after(500000001)))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(c:pk_h(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(or_d(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),jc:pk_k(A))", "Bd");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C))", "Bdue");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001),after(1))", "Bf");
        ms_test("or_d(dv:after(500000001),after(500000001))", "oBf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),jc:pk_h(A))", "Bd");
        ms_test("andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),c:pk_h(D))", "Bdus");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),c:pk_k(B))", "Bdus");
        ms_test("and_v(v:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),1)", "oBfu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),dv:after(1))", "Bdue");
        ms_test("or_d(or_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),n:after(1))", "Bu");
        ms_test("or_d(or_d(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F),after(1))", "B");
        ms_test("or_d(or_i(0,c:pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("andor(j:multi(2,A,B,C),c:pk_h(D),multi(2,E,F,G))", "sdueB");
        ms_test("or_d(or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bu");
        ms_test("or_d(or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(or_d(multi(2,A,B,C),j:multi(2,D,E,F)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1),after(1))", "Bf");
        ms_test("andor(multi(2,A,B,C),multi(2,D,E,F),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("and_b(after(500000001),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bu");
        ms_test("n:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "oBfu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),c:pk_h(B))", "Bdus");
        ms_test("andor(j:multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(500000001)))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(500000001)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(500000001))", "Bu");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),a:multi(2,D,E,F))", "Bdus");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C)))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1)))", "Bf");
        ms_test("andor(multi(2,A,B,C),multi(2,D,E,F),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(and_b(after(1),a:after(1)),a:after(1)),after(1))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),dv:after(1))", "Bdue");
        ms_test("and_b(after(1),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_d(jc:pk_k(A),c:pk_k(B))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(500000001),a:multi(2,A,B,C)))", "Bu");
        ms_test("and_b(after(500000001),a:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),and_b(after(500000001),a:after(500000001))))", "Bfu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)))", "Bdue");
        ms_test("andor(jc:pk_h(A),multi(2,B,C,D),multi(2,E,F,G))", "sdueB");
        ms_test("or_b(c:pk_h(A),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),j:multi(2,B,C,D))", "Bdus");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(or_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)))", "Bdue");
        ms_test("or_d(or_i(c:pk_h(A),0),after(500000001))", "Bf");
        ms_test("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(1)),1)", "oBfu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),n:after(500000001))", "oBf");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))", "sdueB");
        ms_test("or_d(or_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),dv:after(500000001))", "Bdu");
        ms_test("or_d(or_i(multi(2,A,B,C),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(500000001)),after(500000001))", "Bf");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)),after(1))", "Bf");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),after(500000001))", "Bu");
        ms_test("or_d(dv:after(500000001),after(500000001))", "oBf");
        ms_test("or_d(and_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_h(A),multi(2,B,C,D)))", "Bdue");
        ms_test("or_d(or_i(multi(2,A,B,C),0),multi(2,D,E,F))", "sdueB");
        ms_test("or_d(multi(2,A,B,C),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),a:multi(2,D,E,F))", "Bdue");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),c:pk_h(B))", "Bdus");
        ms_test("or_d(or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(c:pk_h(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(jc:pk_k(A),c:pk_h(B))", "sdueB");
        ms_test("or_b(j:multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))", "Bdu");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),after(1))", "Bu");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),after(500000001))", "Bf");
        ms_test("c:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),pk_k(B))", "Bdus");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdue");
        ms_test("or_d(or_i(0,multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(jc:pk_k(A),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(500000001))", "Bfu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_b(c:pk_h(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(or_d(multi(2,A,B,C),multi(2,D,E,F)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(c:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),pk_k(B)),after(500000001))", "Bf");
        ms_test("andor(j:multi(2,A,B,C),after(500000001),multi(2,D,E,F))", "Bdse");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))", "Bdue");
        ms_test("thresh(2,multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(j:multi(2,A,B,C),n:after(500000001),multi(2,D,E,F))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(1),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(500000001)))", "Bf");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sc:pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("and_v(v:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),multi(2,A,B,C))", "Bsu");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:after(500000001))", "Bu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_i(multi(2,A,B,C),0),after(1))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),multi(2,D,E,F)))", "Bdu");
        ms_test("or_b(jc:pk_h(A),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_b(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:multi(2,D,E,F))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),after(1))", "Bf");
        ms_test("andor(jc:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("n:or_d(jc:pk_k(A),after(1))", "oBfu");
        ms_test("or_d(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(v:after(1),after(1)))", "oBf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(500000001),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bu");
        ms_test("andor(jc:pk_h(A),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(multi(2,D,E,F),multi(2,G,I,J)))", "Bdus");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_k(D))", "Bdue");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(D))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),or_d(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(1),a:multi(2,A,B,C)))", "Bu");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),multi(2,D,E,F))", "Bdue");
        ms_test("andor(jc:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))", "sdueB");
        ms_test("and_b(multi(2,A,B,C),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "nBsu");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "B");
        ms_test("andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))", "Bdue");
        ms_test("or_b(or_d(multi(2,A,B,C),multi(2,D,E,F)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),n:after(1))", "oBf");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:multi(2,A,B,C))", "Bdus");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(and_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_b(multi(2,A,B,C),ajc:pk_k(D))", "sdueB");
        ms_test("or_d(or_i(multi(2,A,B,C),0),multi(2,D,E,F))", "sdueB");
        ms_test("andor(multi(2,A,B,C),after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_i(c:pk_k(A),0),multi(2,B,C,D))", "sdueB");
        ms_test("c:andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_h(D),pk_k(E))", "Bdus");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bdue");
        ms_test("andor(or_i(multi(2,A,B,C),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdue");
        ms_test("thresh(2,multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "Bf");
        ms_test("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),multi(2,D,E,F)),after(1))", "B");
        ms_test("and_v(v:after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "oBf");
        ms_test("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C))", "Bdu");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdue");
        ms_test("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "nBdu");
        ms_test("c:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),pk_h(B))", "Bdus");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),ajc:pk_h(A))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),jc:pk_k(A))", "Bd");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),multi(2,A,B,C))", "Bdue");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "B");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)),after(500000001))", "Bf");
        ms_test("andor(j:multi(2,A,B,C),after(500000001),c:pk_h(D))", "Bdse");
        ms_test("or_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(or_d(c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "B");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),c:pk_h(D))", "Bdue");
        ms_test("or_d(c:pk_h(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),multi(2,A,B,C))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "B");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(D)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))))", "B");
        ms_test("andor(j:multi(2,A,B,C),after(500000001),c:pk_k(D))", "Bdse");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(jc:pk_h(A),a:multi(2,B,C,D))", "sdueB");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),and_b(after(1),a:after(1)))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A))", "Bdu");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:after(500000001))", "Bu");
        ms_test("andor(jc:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),c:pk_k(D))", "Bdus");
        ms_test("or_d(j:multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("or_d(or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(or_i(0,multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "Bf");
        ms_test("or_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),multi(2,A,B,C))", "Bde");
        ms_test("or_d(multi(2,A,B,C),or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(jc:pk_k(A),n:after(500000001))", "oBfu");
        ms_test("or_d(or_d(multi(2,A,B,C),j:multi(2,D,E,F)),after(1))", "Bf");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),j:multi(2,D,E,F))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),multi(2,D,E,F))", "Bdus");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0))", "Bdu");
        ms_test("or_d(or_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(j:multi(2,A,B,C),multi(2,D,E,F)))", "Bdue");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(1))", "Bu");
        ms_test("thresh(2,multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("and_b(multi(2,A,B,C),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "sduBn");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),multi(2,A,B,C))", "Bde");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(after(1),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))", "Bdue");
        ms_test("and_b(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),and_b(after(1),a:after(1))),a:after(1))", "Bfu");
        ms_test("or_b(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("and_b(and_b(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1)),a:after(1)),a:after(1))", "Bfu");
        ms_test("c:andor(j:multi(2,A,B,C),pk_h(D),pk_k(E))", "sdueB");
        ms_test("or_b(multi(2,A,B,C),aj:multi(2,D,E,F))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A))", "Bdu");
        ms_test("and_v(v:after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:multi(2,A,B,C))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(jc:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "Bf");
        ms_test("andor(j:multi(2,A,B,C),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_i(multi(2,A,B,C),0),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))", "Bdu");
        ms_test("n:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "oBfu");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),after(500000001))", "B");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A))", "Bdue");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)),multi(2,B,C,D))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),after(500000001))", "Bf");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F),after(500000001))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),c:pk_h(B))", "Bdus");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(1),c:pk_h(A))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),c:pk_h(D))", "Bdus");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(1)))", "Bfu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:multi(2,A,B,C))", "Bdue");
        ms_test("n:or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bfu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bde");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001),multi(2,D,E,F))", "Bde");
        ms_test("c:andor(j:multi(2,A,B,C),pk_h(D),pk_h(E))", "sdueB");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:multi(2,A,B,C))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(v:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bu");
        ms_test("or_d(or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),multi(2,A,B,C))", "Bde");
        ms_test("andor(jc:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(D)))", "Bdue");
        ms_test("andor(jc:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))", "sdueB");
        ms_test("or_d(jc:pk_h(A),n:after(500000001))", "Bfu");
        ms_test("andor(jc:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "Bf");
        ms_test("or_d(multi(2,A,B,C),or_b(multi(2,D,E,F),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:multi(2,A,B,C))", "Bdue");
        ms_test("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "Bu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),c:pk_k(D))", "Bdue");
        ms_test("andor(jc:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),n:after(500000001))", "Bu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)))", "Bdue");
        ms_test("or_d(or_d(c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D))", "Bdue");
        ms_test("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),after(500000001))", "Bf");
        ms_test("andor(jc:pk_k(A),multi(2,B,C,D),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "B");
        ms_test("andor(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),after(500000001))", "Bf");
        ms_test("or_d(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(multi(2,A,B,C),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(dv:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "B");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "B");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(multi(2,A,B,C),a:multi(2,D,E,F)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(multi(2,A,B,C),0),multi(2,D,E,F))", "Bdus");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(v:after(500000001),1))", "oBfu");
        ms_test("or_b(c:pk_h(A),aj:multi(2,B,C,D))", "sdueB");
        ms_test("andor(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),c:pk_k(D)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(1),after(500000001))", "oBf");
        ms_test("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "nBdu");
        ms_test("or_d(or_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "Bdue");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_k(D))", "Bdue");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001),multi(2,A,B,C))", "Bde");
        ms_test("or_b(dv:after(1),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),multi(2,A,B,C),after(500000001))", "B");
        ms_test("or_d(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_b(multi(2,A,B,C),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(or_i(multi(2,A,B,C),0),multi(2,D,E,F))", "sdueB");
        ms_test("or_d(j:multi(2,A,B,C),c:pk_k(D))", "sdueB");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)),after(1))", "Bf");
        ms_test("or_d(andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),c:pk_h(D)))", "Bdue");
        ms_test("or_b(jc:pk_k(A),a:multi(2,B,C,D))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),dv:after(1))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bde");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(1)))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_i(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),0)", "Bde");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_b(c:pk_h(A),aj:multi(2,B,C,D))", "sdueB");
        ms_test("or_b(jc:pk_k(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_b(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("n:andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "Bfu");
        ms_test("or_d(or_i(0,multi(2,A,B,C)),after(1))", "Bf");
        ms_test("or_d(or_i(0,multi(2,A,B,C)),c:pk_h(D))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_k(A))", "Bdu");
        ms_test("or_d(or_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "Bdue");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_b(multi(2,A,B,C),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("andor(j:multi(2,A,B,C),c:pk_h(D),after(500000001))", "Bf");
        ms_test("or_d(or_i(multi(2,A,B,C),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),j:multi(2,B,C,D))", "Bdus");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),c:pk_h(D)))", "Bdue");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_k(D))", "Bdue");
        ms_test("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C))", "Bdu");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(D)))", "Bdue");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),ajc:pk_h(A))", "Bdue");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)),after(1))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),c:pk_k(B))", "Bdus");
        ms_test("or_d(or_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),a:multi(2,D,E,F))", "Bdue");
        ms_test("andor(j:multi(2,A,B,C),after(500000001),after(1))", "Bf");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_h(A),multi(2,B,C,D))", "Bdus");
        ms_test("andor(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "B");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),1))", "Bfu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),after(500000001))", "B");
        ms_test("or_i(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),0)", "Bde");
        ms_test("andor(jc:pk_k(A),multi(2,B,C,D),multi(2,E,F,G))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "B");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),multi(2,A,B,C))", "Bdue");
        ms_test("andor(or_i(multi(2,A,B,C),0),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),c:pk_h(A))", "Bde");
        ms_test("or_b(multi(2,A,B,C),ajc:pk_h(D))", "sdueB");
        ms_test("and_b(after(500000001),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bu");
        ms_test("or_d(andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "Bdue");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),c:pk_h(A))", "Bde");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),after(1))", "B");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("n:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "oBfu");
        ms_test("or_i(0,or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)))", "Bdu");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "B");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("and_b(after(500000001),a:and_b(after(500000001),a:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001))))", "Bfu");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(jc:pk_k(A),c:pk_h(B))", "sdueB");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("and_v(v:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "B");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(multi(2,A,B,C),multi(2,D,E,F),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(multi(2,A,B,C),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "B");
        ms_test("or_b(c:pk_k(A),aj:multi(2,B,C,D))", "sdueB");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)),after(1))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(500000001)))", "B");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(1)),multi(2,D,E,F))", "Bde");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdue");
        ms_test("c:andor(j:multi(2,A,B,C),pk_h(D),pk_h(E))", "sdueB");
        ms_test("or_b(c:pk_k(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("c:andor(j:multi(2,A,B,C),pk_k(D),pk_h(E))", "sdueB");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sc:pk_k(A))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(1))", "Bu");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),a:multi(2,D,E,F))", "Bdue");
        ms_test("and_v(v:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(multi(2,A,B,C),a:multi(2,D,E,F)))", "Bdue");
        ms_test("or_d(multi(2,A,B,C),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(and_b(after(500000001),a:after(500000001)),a:after(500000001)),after(1))", "Bf");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdue");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),n:after(1)))", "Bfu");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,c:pk_h(A)))", "Bdu");
        ms_test("or_d(j:multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))", "Bdue");
        ms_test("or_d(and_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_d(or_d(c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(jc:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),multi(2,D,E,F)))", "Bdu");
        ms_test("or_b(multi(2,A,B,C),sjc:pk_k(D))", "sdueB");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(D)))", "Bdue");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(j:multi(2,A,B,C),multi(2,D,E,F),multi(2,G,I,J))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),c:pk_k(D))", "Bdus");
        ms_test("or_d(jc:pk_h(A),multi(2,B,C,D))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),ac:pk_k(A)))", "Bdue");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),ac:pk_k(A))", "Bdue");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sjc:pk_k(A))", "Bdue");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),multi(2,B,C,D))", "Bdus");
        ms_test("or_d(andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),j:multi(2,A,B,C))", "Bd");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(500000001)),multi(2,D,E,F))", "Bde");
        ms_test("andor(j:multi(2,A,B,C),after(500000001),after(1))", "Bf");
        ms_test("or_d(or_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "Bdue");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("n:or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)),after(500000001))", "Bfu");
        ms_test("andor(j:multi(2,A,B,C),c:pk_h(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "Bu");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(multi(2,A,B,C),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(multi(2,D,E,F),multi(2,G,I,J)))", "Bdus");
        ms_test("or_d(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdue");
        ms_test("c:andor(j:multi(2,A,B,C),pk_k(D),pk_h(E))", "sdueB");
        ms_test("andor(multi(2,A,B,C),after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))", "Bde");
        ms_test("andor(jc:pk_h(A),multi(2,B,C,D),after(1))", "Bf");
        ms_test("or_d(or_d(c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D))", "Bdue");
        ms_test("or_b(j:multi(2,A,B,C),aj:multi(2,D,E,F))", "sdueB");
        ms_test("or_d(andor(multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_i(after(500000001),0))", "Bdue");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("c:andor(j:multi(2,A,B,C),pk_k(D),pk_k(E))", "sdueB");
        ms_test("and_v(v:after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "oBf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(dv:after(500000001),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),after(500000001))", "B");
        ms_test("n:or_d(j:multi(2,A,B,C),after(1))", "Bfu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(and_b(after(500000001),a:after(500000001)),a:after(500000001)),after(500000001))", "Bf");
        ms_test("or_d(c:pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(v:multi(2,A,B,C),multi(2,D,E,F)))", "Bfu");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdue");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_d(j:multi(2,A,B,C),multi(2,D,E,F)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,multi(2,A,B,C)),after(500000001))", "B");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C),a:multi(2,D,E,F))", "Bdus");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)))", "Bdue");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "B");
        ms_test("or_d(or_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),after(1))", "B");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdue");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F),after(500000001))", "B");
        ms_test("or_d(andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),and_b(and_b(after(500000001),a:after(500000001)),a:after(500000001)))", "Bf");
        ms_test("or_d(or_i(multi(2,A,B,C),0),after(500000001))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),c:pk_h(A))", "Bde");
        ms_test("or_b(c:pk_h(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(jc:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_i(0,or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bde");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),or_d(multi(2,A,B,C),multi(2,D,E,F)))", "Bde");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),ac:pk_h(A))", "Bdue");
        ms_test("or_d(or_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(and_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(dv:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(jc:pk_h(A),after(500000001),after(1))", "Bf");
        ms_test("n:or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(500000001))", "Bfu");
        ms_test("andor(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)),after(500000001))", "Bf");
        ms_test("thresh(2,multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "B");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(dv:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(j:multi(2,A,B,C),multi(2,D,E,F),j:multi(2,G,I,J))", "Bdus");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(1)))", "Bf");
        ms_test("c:andor(j:multi(2,A,B,C),pk_h(D),pk_h(E))", "sdueB");
        ms_test("andor(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(j:multi(2,A,B,C),after(500000001)))", "Bf");
        ms_test("or_d(and_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "nBdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(j:multi(2,A,B,C),multi(2,D,E,F)))", "Bdue");
        ms_test("or_d(j:multi(2,A,B,C),jc:pk_h(D))", "Bdus");
        ms_test("or_d(or_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001),after(500000001))", "Bf");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("and_b(multi(2,A,B,C),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "nBsu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_h(A),after(500000001)))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(500000001),after(1))", "oBf");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_b(j:multi(2,A,B,C),ac:pk_h(D))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A))", "Bdue");
        ms_test("or_d(j:multi(2,A,B,C),or_d(multi(2,D,E,F),after(1)))", "Bf");
        ms_test("and_b(multi(2,A,B,C),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))", "sduBn");
        ms_test("or_b(j:multi(2,A,B,C),sc:pk_k(D))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),multi(2,D,E,F))", "Bdus");
        ms_test("andor(j:multi(2,A,B,C),c:pk_h(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(j:multi(2,A,B,C),after(1),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,multi(2,A,B,C)),multi(2,D,E,F))", "Bdus");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),multi(2,A,B,C),after(500000001))", "B");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,c:pk_h(A)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),after(500000001))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),c:pk_h(B))", "Bdus");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("and_b(multi(2,A,B,C),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))", "sduBn");
        ms_test("andor(j:multi(2,A,B,C),multi(2,D,E,F),multi(2,G,I,J))", "sdueB");
        ms_test("andor(or_i(0,multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "Bf");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),after(500000001))", "Bu");
        ms_test("andor(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "B");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(jc:pk_k(A),multi(2,B,C,D),multi(2,E,F,G))", "sdueB");
        ms_test("or_d(j:multi(2,A,B,C),j:multi(2,D,E,F))", "Bdus");
        ms_test("andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("or_i(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),0)", "Bd");
        ms_test("andor(multi(2,A,B,C),multi(2,D,E,F),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,G,I,J)))", "Bdue");
        ms_test("or_d(or_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_i(0,after(500000001)))", "Bdue");
        ms_test("or_b(j:multi(2,A,B,C),a:multi(2,D,E,F))", "sdueB");
        ms_test("and_b(after(500000001),a:and_b(after(500000001),a:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001))))", "Bfu");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "Bf");
        ms_test("andor(jc:pk_h(A),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),multi(2,A,B,C))", "Bdue");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdue");
        ms_test("n:andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "Bfu");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bfu");
        ms_test("c:andor(j:multi(2,A,B,C),pk_k(D),pk_h(E))", "sdueB");
        ms_test("or_d(or_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(500000001),a:and_b(after(500000001),a:after(500000001))),after(500000001))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(500000001),a:multi(2,A,B,C)))", "Bu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_d(multi(2,A,B,C),after(1)))", "Bf");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)),after(500000001))", "Bf");
        ms_test("andor(j:multi(2,A,B,C),multi(2,D,E,F),after(500000001))", "Bf");
        ms_test("or_d(multi(2,A,B,C),or_d(j:multi(2,D,E,F),after(1)))", "Bf");
        ms_test("or_b(or_i(0,after(1)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(or_i(0,multi(2,A,B,C)),multi(2,D,E,F))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C))", "Bdu");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_h(D))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdu");
        ms_test("or_d(c:pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001),after(500000001))", "Bf");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))", "sdueB");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,D,E,F)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),n:after(500000001))", "oBf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(or_i(multi(2,A,B,C),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(1),a:multi(2,A,B,C)))", "Bu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_k(A))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),0))", "Bdu");
        ms_test("andor(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),dv:after(1))", "Bdue");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),ac:pk_k(A))", "Bdue");
        ms_test("or_i(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),0)", "Bde");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bdue");
        ms_test("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "nBu");
        ms_test("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(j:multi(2,A,B,C),aj:multi(2,D,E,F))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(500000001),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bu");
        ms_test("or_d(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(dv:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("and_b(after(500000001),a:and_b(after(500000001),a:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001))))", "Bfu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),dv:after(500000001))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:after(1)))", "Bu");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001))", "oBfu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),c:pk_k(A))", "Bde");
        ms_test("or_d(j:multi(2,A,B,C),after(1))", "Bf");
        ms_test("c:andor(j:multi(2,A,B,C),pk_h(D),pk_h(E))", "sdueB");
        ms_test("andor(j:multi(2,A,B,C),j:multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(after(500000001),0))", "Bd");
        ms_test("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(multi(2,A,B,C),after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("and_v(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "nBu");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),0),after(1))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),multi(2,D,E,F)),multi(2,G,I,J))", "Bdus");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bu");
        ms_test("or_b(multi(2,A,B,C),a:or_d(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(D)))", "Bdue");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)),multi(2,B,C,D))", "Bdue");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,D,E,F)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_k(A))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),0)", "Bdus");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_h(A))", "Bdu");
        ms_test("andor(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("or_d(multi(2,A,B,C),or_b(multi(2,D,E,F),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_b(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),aj:multi(2,A,B,C))", "Bdue");
        ms_test("or_b(multi(2,A,B,C),a:or_d(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(D))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),j:multi(2,D,E,F)))", "Bdu");
        ms_test("or_d(or_i(c:pk_k(A),0),after(500000001))", "Bf");
        ms_test("or_b(j:multi(2,A,B,C),ac:pk_h(D))", "sdueB");
        ms_test("andor(jc:pk_h(A),multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(1)),multi(2,D,E,F))", "Bde");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_h(A),after(1))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),j:multi(2,A,B,C))", "Bd");
        ms_test("or_b(j:multi(2,A,B,C),a:multi(2,D,E,F))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),ajc:pk_k(A))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A))", "Bdu");
        ms_test("andor(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)),multi(2,G,I,J))", "sdueB");
        ms_test("andor(j:multi(2,A,B,C),after(1),multi(2,D,E,F))", "Bdse");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_d(multi(2,A,B,C),after(500000001)))", "Bf");
        ms_test("or_d(jc:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),after(1))", "B");
        ms_test("or_d(or_i(multi(2,A,B,C),0),after(1))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(j:multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(jc:pk_k(A),multi(2,B,C,D),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),multi(2,D,E,F))", "Bdus");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdue");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sc:pk_k(D))", "Bdue");
        ms_test("andor(jc:pk_h(A),after(500000001),after(500000001))", "Bf");
        ms_test("or_d(jc:pk_k(A),c:pk_k(B))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),multi(2,D,E,F)),after(1))", "B");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(multi(2,A,B,C),multi(2,D,E,F),after(1)))", "Bf");
        ms_test("n:or_d(jc:pk_h(A),after(500000001))", "Bfu");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),multi(2,D,E,F))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),jc:pk_h(D))", "Bdus");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(multi(2,A,B,C),after(1),after(500000001)))", "Bf");
        ms_test("or_b(or_i(after(1),0),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F),after(1))", "B");
        ms_test("or_b(dv:after(500000001),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(500000001))", "B");
        ms_test("andor(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),c:pk_h(D)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A))", "Bdu");
        ms_test("or_d(multi(2,A,B,C),or_d(j:multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),multi(2,D,E,F)),after(1))", "B");
        ms_test("andor(dv:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(jc:pk_h(A),after(1),multi(2,B,C,D))", "Bdse");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),after(1))", "Bf");
        ms_test("or_d(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("and_v(v:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),after(500000001))", "oBf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),n:after(1))", "Bu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bdue");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "Bu");
        ms_test("andor(jc:pk_h(A),multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_h(A),after(500000001)))", "Bf");
        ms_test("n:andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "Bfu");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bu");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(multi(2,A,B,C),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(j:multi(2,A,B,C),n:after(500000001))", "Bfu");
        ms_test("or_d(or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(jc:pk_h(A),after(1))", "Bf");
        ms_test("or_d(jc:pk_h(A),after(500000001))", "Bf");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "B");
        ms_test("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bsu");
        ms_test("or_d(jc:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(1))", "Bf");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F),after(500000001))", "B");
        ms_test("and_b(after(500000001),a:and_b(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001)),a:after(500000001)))", "Bfu");
        ms_test("or_d(or_b(c:pk_k(A),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(or_i(0,multi(2,A,B,C)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),multi(2,D,E,F)),after(500000001))", "B");
        ms_test("or_d(or_i(c:pk_h(A),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("or_d(or_d(multi(2,A,B,C),j:multi(2,D,E,F)),after(500000001))", "Bf");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:after(500000001)))", "Bu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bu");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("andor(j:multi(2,A,B,C),multi(2,D,E,F),after(1))", "Bf");
        ms_test("or_d(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)),after(500000001))", "Bf");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_h(D))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),multi(2,D,E,F)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_b(c:pk_k(A),aj:multi(2,B,C,D))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(1)))", "Bfu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(or_c(multi(2,A,B,C),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bu");
        ms_test("or_d(multi(2,A,B,C),or_d(j:multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(after(1),0))", "Bde");
        ms_test("c:andor(j:multi(2,A,B,C),pk_h(D),pk_k(E))", "sdueB");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),j:multi(2,A,B,C))", "Bdu");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),adv:after(1))", "Bdue");
        ms_test("or_d(or_i(0,multi(2,A,B,C)),after(1))", "Bf");
        ms_test("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),j:multi(2,A,B,C))", "Bd");
        ms_test("andor(j:multi(2,A,B,C),after(1),multi(2,D,E,F))", "Bdse");
        ms_test("n:or_d(j:multi(2,A,B,C),after(1))", "Bfu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),and_b(after(500000001),a:and_b(after(500000001),a:after(500000001))))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(v:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bu");
        ms_test("or_b(jc:pk_h(A),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(multi(2,D,E,F),multi(2,G,I,J)))", "Bdus");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,D,E,F)))", "Bdu");
        ms_test("or_b(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("c:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),pk_k(B))", "Bdus");
        ms_test("andor(jc:pk_h(A),multi(2,B,C,D),after(500000001))", "Bf");
        ms_test("or_d(or_i(0,multi(2,A,B,C)),after(1))", "Bf");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("and_v(v:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(500000001))", "Bf");
        ms_test("or_b(multi(2,A,B,C),ajc:pk_k(D))", "sdueB");
        ms_test("or_d(c:pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))", "Bdue");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),after(500000001))", "Bf");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)),multi(2,B,C,D))", "Bdue");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(1))", "Bf");
        ms_test("or_d(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_h(A))", "Bdue");
        ms_test("or_i(0,or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(or_i(0,multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_i(0,or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(j:multi(2,A,B,C),j:multi(2,D,E,F),after(1))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),multi(2,D,E,F)))", "Bdue");
        ms_test("thresh(2,multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,D,E,F))", "Bdus");
        ms_test("or_d(andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("andor(jc:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "Bf");
        ms_test("andor(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_h(A),multi(2,B,C,D))", "Bdus");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0))", "Bd");
        ms_test("or_d(j:multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),c:pk_k(B))", "Bdus");
        ms_test("or_d(j:multi(2,A,B,C),n:after(500000001))", "Bfu");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "B");
        ms_test("or_d(or_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_i(c:pk_k(A),0),multi(2,B,C,D))", "sdueB");
        ms_test("or_b(dv:after(1),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(or_d(c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(1),after(1))", "oBf");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),after(500000001))", "Bf");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)))", "Bdue");
        ms_test("andor(j:multi(2,A,B,C),c:pk_h(D),multi(2,E,F,G))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),multi(2,D,E,F))", "Bdus");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_b(jc:pk_k(A),a:multi(2,B,C,D))", "sdueB");
        ms_test("or_d(multi(2,A,B,C),or_d(j:multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_b(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:multi(2,D,E,F))", "Bdue");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1),after(500000001))", "Bf");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "B");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0))", "Bdue");
        ms_test("or_d(c:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),pk_h(B)),after(500000001))", "Bf");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_h(A))", "Bdue");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_h(A))", "Bdu");
        ms_test("andor(jc:pk_h(A),after(500000001),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),c:pk_h(B))", "Bdus");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C))", "Bdue");
        ms_test("and_b(and_b(after(500000001),a:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001))),a:after(500000001))", "Bfu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),c:pk_k(D))", "Bdus");
        ms_test("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_b(j:multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)),multi(2,B,C,D))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(jc:pk_k(A),j:multi(2,B,C,D))", "Bdus");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(1)))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(after(500000001),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_i(0,or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bde");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),multi(2,B,C,D))", "Bdus");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)))", "Bdue");
        ms_test("or_d(c:pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))", "Bdue");
        ms_test("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(500000001)),multi(2,A,B,C))", "Bsu");
        ms_test("or_b(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C)))", "Bdu");
        ms_test("n:andor(j:multi(2,A,B,C),multi(2,D,E,F),after(500000001))", "Bfu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),multi(2,A,B,C))", "Bde");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,after(1)),multi(2,A,B,C))", "Bd");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),c:pk_k(A))", "Bde");
        ms_test("or_d(or_i(multi(2,A,B,C),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("thresh(2,multi(2,A,B,C),a:multi(2,D,E,F),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdus");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "Bf");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F),multi(2,G,I,J))", "Bdus");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),after(1))", "B");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),j:multi(2,D,E,F)))", "Bdu");
        ms_test("or_d(or_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(500000001))", "oBf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("or_d(dvn:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(and_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),n:after(1))", "Bu");
        ms_test("or_d(or_d(j:multi(2,A,B,C),multi(2,D,E,F)),after(1))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_b(j:multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_i(0,andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bd");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1))", "oBfu");
        ms_test("or_d(j:multi(2,A,B,C),dv:after(500000001))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(500000001),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bu");
        ms_test("or_d(or_i(c:pk_h(A),0),multi(2,B,C,D))", "sdueB");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdue");
        ms_test("or_b(c:pk_h(A),aj:multi(2,B,C,D))", "sdueB");
        ms_test("or_i(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),0)", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),multi(2,A,B,C))", "Bde");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),multi(2,D,E,F))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_k(A))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("andor(j:multi(2,A,B,C),after(1),multi(2,D,E,F))", "Bdse");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sc:pk_k(A))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(jc:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("or_d(or_i(0,multi(2,A,B,C)),multi(2,D,E,F))", "sdueB");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(D))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(and_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "sdueB");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),ac:pk_h(A))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),c:pk_k(D)))", "Bdue");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdu");
        ms_test("andor(j:multi(2,A,B,C),after(1),after(1))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),after(500000001))", "Bf");
        ms_test("c:andor(j:multi(2,A,B,C),pk_h(D),pk_h(E))", "sdueB");
        ms_test("andor(j:multi(2,A,B,C),c:pk_k(D),multi(2,E,F,G))", "sdueB");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(D))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,j:multi(2,A,B,C)))", "Bdu");
        ms_test("or_d(or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_d(c:pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("or_d(multi(2,A,B,C),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bdu");
        ms_test("andor(or_i(0,multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "Bf");
        ms_test("andor(j:multi(2,A,B,C),j:multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_i(0,after(1)))", "Bdue");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),c:pk_h(D))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),c:pk_k(A))", "Bde");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),j:multi(2,A,B,C))", "Bdu");
        ms_test("or_d(multi(2,A,B,C),or_b(multi(2,D,E,F),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("andor(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),multi(2,A,B,C),after(500000001))", "B");
        ms_test("or_d(c:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),pk_h(B)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(1)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("c:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),pk_h(B))", "Bdus");
        ms_test("or_d(multi(2,A,B,C),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(500000001)))", "Bf");
        ms_test("or_d(or_d(multi(2,A,B,C),j:multi(2,D,E,F)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("or_d(or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D))", "Bdue");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("or_d(or_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "Bdue");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:multi(2,A,B,C))", "Bdue");
        ms_test("or_b(c:pk_h(A),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("or_d(or_i(c:pk_k(A),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(jc:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("and_v(v:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,multi(2,A,B,C)))", "Bdu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(500000001))", "Bf");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(D))", "sdueB");
        ms_test("or_d(jc:pk_h(A),c:pk_k(B))", "sdueB");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdue");
        ms_test("c:andor(j:multi(2,A,B,C),pk_h(D),pk_k(E))", "sdueB");
        ms_test("or_d(or_i(multi(2,A,B,C),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("and_b(after(500000001),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(and_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("or_d(and_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(j:multi(2,A,B,C),multi(2,D,E,F)))", "Bdue");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("or_d(jc:pk_k(A),c:pk_h(B))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A))", "Bdue");
        ms_test("andor(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C)),after(1))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,D,E,F)))", "Bdue");
        ms_test("andor(j:multi(2,A,B,C),after(500000001),after(1))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),after(1))", "B");
        ms_test("and_v(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "nBf");
        ms_test("andor(jc:pk_h(A),multi(2,B,C,D),after(500000001))", "Bf");
        ms_test("andor(multi(2,A,B,C),after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("or_d(jc:pk_k(A),c:pk_h(B))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),multi(2,B,C,D))", "Bdus");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_h(A),after(1)))", "Bf");
        ms_test("andor(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)),after(1))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "B");
        ms_test("thresh(2,multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),adv:after(1))", "Bdue");
        ms_test("andor(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("andor(j:multi(2,A,B,C),after(1),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),n:after(1))", "Bu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(multi(2,A,B,C),after(500000001),after(500000001)))", "Bf");
        ms_test("and_b(after(1),a:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1)))", "Bfu");
        ms_test("andor(or_i(0,multi(2,A,B,C)),multi(2,D,E,F),after(500000001))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("and_v(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "nBu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(1))", "oBf");
        ms_test("andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))", "Bdue");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(c:pk_h(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_b(c:pk_k(A),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(j:multi(2,A,B,C),j:multi(2,D,E,F),after(500000001))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "B");
        ms_test("andor(jc:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))", "sdueB");
        ms_test("or_d(jc:pk_h(A),after(500000001))", "Bf");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),0)", "oBde");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "B");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),c:pk_h(D))", "Bdus");
        ms_test("or_d(or_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_i(and_v(v:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),0)", "Bdu");
        ms_test("andor(or_i(multi(2,A,B,C),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "Bf");
        ms_test("and_b(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001)),a:and_b(after(500000001),a:after(500000001)))", "Bfu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),after(500000001))", "B");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(j:multi(2,A,B,C),multi(2,D,E,F),c:pk_h(G))", "sdueB");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(1)),after(1))", "Bf");
        ms_test("or_d(jc:pk_k(A),c:pk_k(B))", "sdueB");
        ms_test("or_d(or_d(c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),after(500000001))", "Bf");
        ms_test("andor(jc:pk_h(A),after(1),after(500000001))", "Bf");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),a:multi(2,A,B,C))", "Bsu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_i(after(500000001),0))", "Bd");
        ms_test("andor(j:multi(2,A,B,C),multi(2,D,E,F),multi(2,G,I,J))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,multi(2,A,B,C)))", "Bdu");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1),after(1))", "Bf");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("n:or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bfu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),dv:after(500000001))", "Bdue");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_k(D))", "Bdue");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F),after(500000001))", "B");
        ms_test("or_d(c:pk_h(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),multi(2,D,E,F)))", "Bdue");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),c:pk_h(D))", "Bdue");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "Bf");
        ms_test("thresh(2,multi(2,A,B,C),a:multi(2,D,E,F),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdus");
        ms_test("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bu");
        ms_test("or_d(and_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("and_v(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "nBu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,after(500000001)))", "Bde");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,multi(2,A,B,C)),after(500000001))", "B");
        ms_test("thresh(2,multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,D,E,F))", "Bdus");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_h(A))", "Bdu");
        ms_test("or_d(or_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_h(D))", "Bdue");
        ms_test("c:andor(j:multi(2,A,B,C),pk_k(D),pk_k(E))", "sdueB");
        ms_test("or_d(and_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "B");
        ms_test("or_d(or_d(j:multi(2,A,B,C),multi(2,D,E,F)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),multi(2,B,C,D))", "Bdus");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(or_d(c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(1),a:and_b(after(1),a:after(1))),after(1))", "Bf");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),adv:after(500000001))", "Bdue");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(or_i(multi(2,A,B,C),0),after(500000001),after(500000001))", "Bf");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),after(500000001))", "Bf");
        ms_test("or_d(j:multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "Bdue");
        ms_test("or_d(c:pk_h(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(500000001),c:pk_k(A))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),j:multi(2,D,E,F))", "Bdus");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(500000001)),after(1))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "B");
        ms_test("andor(multi(2,A,B,C),multi(2,D,E,F),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(jc:pk_h(A),c:pk_h(B))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_b(jc:pk_h(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sc:pk_k(A)),multi(2,B,C,D))", "Bdue");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(dv:after(500000001),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(j:multi(2,A,B,C),c:pk_k(D))", "sdueB");
        ms_test("andor(j:multi(2,A,B,C),c:pk_k(D),after(1))", "Bf");
        ms_test("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0))", "Bdu");
        ms_test("andor(jc:pk_k(A),after(1),multi(2,B,C,D))", "Bdse");
        ms_test("c:andor(j:multi(2,A,B,C),pk_h(D),pk_k(E))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(multi(2,A,B,C),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),dv:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(c:pk_h(A),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,multi(2,A,B,C)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),c:pk_k(D))", "Bdus");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),j:multi(2,A,B,C))", "Bd");
        ms_test("or_d(or_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(D)))", "Bdue");
        ms_test("and_v(v:after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bu");
        ms_test("andor(j:multi(2,A,B,C),after(500000001),after(500000001))", "Bf");
        ms_test("andor(jc:pk_k(A),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_h(D))", "Bdue");
        ms_test("c:andor(j:multi(2,A,B,C),pk_k(D),pk_k(E))", "sdueB");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1))", "oBf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(c:pk_k(A),0))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_h(A))", "Bdue");
        ms_test("or_b(multi(2,A,B,C),sjc:pk_k(D))", "sdueB");
        ms_test("or_b(or_i(after(500000001),0),a:or_i(0,after(1)))", "Bdue");
        ms_test("or_d(n:or_i(after(500000001),0),after(500000001))", "oBf");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))", "Bdue");
        ms_test("or_d(jc:pk_h(A),c:pk_k(B))", "sdueB");
        ms_test("or_i(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),0)", "Bdu");
        ms_test("andor(j:multi(2,A,B,C),after(1),0)", "Bdse");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),dv:after(500000001))", "Bdu");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),j:multi(2,A,B,C))", "Bdu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),c:pk_k(A))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),j:multi(2,D,E,F))", "Bdus");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(multi(2,A,B,C),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(jc:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "Bf");
        ms_test("andor(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "B");
        ms_test("andor(jc:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "B");
        ms_test("andor(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),multi(2,A,B,C),after(500000001))", "B");
        ms_test("or_b(or_d(multi(2,A,B,C),multi(2,D,E,F)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_h(A))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(1))", "B");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "B");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_b(jc:pk_h(A),a:multi(2,B,C,D))", "sdueB");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_b(c:pk_k(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0)", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),multi(2,D,E,F))", "Bdus");
        ms_test("andor(j:multi(2,A,B,C),j:multi(2,D,E,F),after(1))", "Bf");
        ms_test("thresh(2,multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(dv:after(1),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(dv:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(1)),1)", "oBfu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),multi(2,D,E,F)),after(500000001))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),after(1))", "B");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_k(A),after(1)))", "Bf");
        ms_test("andor(j:multi(2,A,B,C),j:multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),jc:pk_k(D))", "Bdus");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,D,E,F))", "Bdus");
        ms_test("andor(j:multi(2,A,B,C),j:multi(2,D,E,F),after(500000001))", "Bf");
        ms_test("and_v(v:after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bu");
        ms_test("andor(j:multi(2,A,B,C),c:pk_h(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(or_i(multi(2,A,B,C),0),multi(2,D,E,F),multi(2,G,I,J))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(500000001)),multi(2,D,E,F))", "Bde");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),c:pk_h(D))", "Bdue");
        ms_test("or_d(j:multi(2,A,B,C),j:multi(2,D,E,F))", "Bdus");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(j:multi(2,A,B,C),after(1),after(500000001))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),j:multi(2,D,E,F)))", "Bdu");
        ms_test("or_i(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)),0)", "Bdu");
        ms_test("or_d(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(jc:pk_h(A),after(500000001),after(500000001))", "Bf");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdue");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_k(A),after(500000001)))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(multi(2,A,B,C),a:after(500000001)))", "Bu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("or_d(j:multi(2,A,B,C),j:multi(2,D,E,F))", "Bdus");
        ms_test("or_i(0,or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bde");
        ms_test("or_d(or_i(0,c:pk_h(A)),after(500000001))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(v:after(500000001),after(500000001)))", "oBf");
        ms_test("or_d(jc:pk_k(A),multi(2,B,C,D))", "sdueB");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(multi(2,A,B,C),0))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_i(multi(2,A,B,C),0)),after(500000001))", "Bf");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "Bu");
        ms_test("or_b(multi(2,A,B,C),ajc:pk_k(D))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))))", "Bdu");
        ms_test("or_d(and_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "sdueB");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),ajc:pk_k(A))", "Bdue");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(D)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),j:multi(2,B,C,D))", "Bdus");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(500000001),a:after(500000001)),and_b(after(1),a:after(1)))", "Bfu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "B");
        ms_test("or_b(jc:pk_k(A),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bu");
        ms_test("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),n:after(500000001))", "Bfu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(multi(2,A,B,C),0))", "Bdu");
        ms_test("and_b(and_b(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001)),a:after(500000001)),a:after(500000001))", "Bfu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),c:pk_h(D)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_k(A),multi(2,B,C,D)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(500000001),multi(2,A,B,C))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),n:after(1))", "Bu");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(1))", "Bf");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),adv:after(500000001))", "Bdue");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("andor(jc:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(500000001)))", "Bf");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_d(or_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "Bdue");
        ms_test("or_d(j:multi(2,A,B,C),c:pk_h(D))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),after(1))", "B");
        ms_test("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(j:multi(2,A,B,C),ac:pk_h(D))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("and_b(after(500000001),a:and_b(after(500000001),a:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001))))", "Bfu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),j:multi(2,D,E,F))", "Bdus");
        ms_test("or_d(j:multi(2,A,B,C),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C)))", "Bdu");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(j:multi(2,A,B,C),c:pk_h(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(and_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),c:pk_h(D)))", "Bdue");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)),multi(2,B,C,D))", "Bdue");
        ms_test("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C),a:multi(2,D,E,F))", "Bdus");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0))", "Bd");
        ms_test("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(500000001)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),jc:pk_k(D))", "Bdus");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bfu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_k(A),multi(2,B,C,D))", "Bdus");
        ms_test("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),after(1))", "Bf");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("and_b(after(1),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),c:pk_h(A))", "Bde");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),n:after(1))", "oBf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdue");
        ms_test("or_b(or_d(multi(2,A,B,C),multi(2,D,E,F)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "B");
        ms_test("or_d(or_d(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("c:andor(j:multi(2,A,B,C),pk_k(D),pk_k(E))", "sdueB");
        ms_test("and_b(multi(2,A,B,C),a:or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,D,E,F)))", "sduBn");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("and_v(v:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001)),multi(2,A,B,C))", "Bsu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0))", "Bdu");
        ms_test("or_d(c:pk_h(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_k(A),after(1)))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(multi(2,A,B,C),0),after(1))", "B");
        ms_test("or_d(or_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001)))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),c:pk_k(D))", "Bdus");
        ms_test("c:andor(j:multi(2,A,B,C),pk_h(D),pk_k(E))", "sdueB");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,after(1)))", "Bde");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),dvn:after(500000001))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(multi(2,A,B,C),a:multi(2,D,E,F)))", "Bdue");
        ms_test("or_b(multi(2,A,B,C),a:or_d(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("and_v(v:after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C)))", "Bdu");
        ms_test("or_d(multi(2,A,B,C),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(j:multi(2,A,B,C),j:multi(2,D,E,F),after(500000001))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),c:pk_k(D)))", "Bdue");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "Bdue");
        ms_test("and_b(after(1),a:and_b(after(1),a:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1))))", "Bfu");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),after(1))", "Bf");
        ms_test("or_d(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(andor(multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(dv:after(1),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(j:multi(2,A,B,C),c:pk_h(D),after(500000001))", "Bf");
        ms_test("or_d(or_i(0,c:pk_k(A)),after(1))", "Bf");
        ms_test("or_b(jc:pk_k(A),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(j:multi(2,A,B,C),j:multi(2,D,E,F),after(1))", "Bf");
        ms_test("or_d(multi(2,A,B,C),or_b(multi(2,D,E,F),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),j:multi(2,D,E,F)))", "Bdu");
        ms_test("or_d(or_i(c:pk_h(A),0),multi(2,B,C,D))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(after(500000001),0))", "Bde");
        ms_test("or_d(or_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_b(jc:pk_k(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_d(j:multi(2,A,B,C),dv:after(1))", "Bdue");
        ms_test("or_b(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(j:multi(2,A,B,C),j:multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F),multi(2,G,I,J))", "Bdus");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_k(A))", "Bdu");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001),after(500000001))", "Bf");
        ms_test("or_d(or_d(j:multi(2,A,B,C),multi(2,D,E,F)),multi(2,G,I,J))", "sdueB");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(1))", "Bf");
        ms_test("or_d(or_i(0,multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("c:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),pk_h(B))", "Bdus");
        ms_test("n:or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)),after(500000001))", "Bfu");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_d(multi(2,A,B,C),multi(2,D,E,F)))", "Bdue");
        ms_test("or_d(multi(2,A,B,C),or_b(multi(2,D,E,F),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("and_b(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("c:andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),pk_k(D),pk_h(E))", "Bdus");
        ms_test("or_d(j:multi(2,A,B,C),c:pk_h(D))", "sdueB");
        ms_test("or_d(j:multi(2,A,B,C),n:after(1))", "Bfu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(500000001),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,after(500000001)))", "Bde");
        ms_test("or_d(jc:pk_k(A),c:pk_h(B))", "sdueB");
        ms_test("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bu");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),c:pk_k(B))", "Bdus");
        ms_test("andor(dv:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),after(500000001))", "B");
        ms_test("or_b(multi(2,A,B,C),a:or_d(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,after(500000001)))", "Bd");
        ms_test("andor(jc:pk_h(A),multi(2,B,C,D),after(500000001))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),multi(2,A,B,C))", "Bdue");
        ms_test("or_d(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,D,E,F)))", "Bdue");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_b(j:multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("andor(j:multi(2,A,B,C),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("andor(j:multi(2,A,B,C),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),ac:pk_k(A))", "Bdue");
        ms_test("or_d(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(jc:pk_k(A),multi(2,B,C,D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(500000001))", "B");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)),after(1))", "Bf");
        ms_test("or_d(dv:after(500000001),after(1))", "oBf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(500000001),a:after(500000001)))", "Bfu");
        ms_test("or_d(j:multi(2,A,B,C),or_i(0,or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),c:pk_h(B))", "Bdus");
        ms_test("andor(j:multi(2,A,B,C),c:pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(dv:after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(jc:pk_k(A),after(1),after(500000001))", "oBf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(after(1),0),multi(2,A,B,C))", "Bd");
        ms_test("or_d(multi(2,A,B,C),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("or_b(j:multi(2,A,B,C),ac:pk_k(D))", "sdueB");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),multi(2,D,E,F)),after(500000001))", "Bf");
        ms_test("or_d(andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(j:multi(2,A,B,C),multi(2,D,E,F),0)", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(j:multi(2,A,B,C),multi(2,D,E,F),after(1))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),after(1))", "B");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("andor(dv:after(1),after(1),after(1))", "oBf");
        ms_test("or_b(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_h(A),after(1)))", "Bf");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),after(500000001))", "Bu");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(n:or_i(0,after(500000001)),after(500000001))", "oBf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),c:pk_k(D))", "Bdus");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(j:multi(2,A,B,C),multi(2,D,E,F)))", "Bdue");
        ms_test("thresh(2,multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("c:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_h(A),pk_k(B))", "Bdus");
        ms_test("or_b(j:multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),after(500000001)))", "B");
        ms_test("and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "nBu");
        ms_test("andor(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)),after(1))", "Bf");
        ms_test("or_d(or_i(multi(2,A,B,C),0),multi(2,D,E,F))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),j:multi(2,B,C,D))", "Bdus");
        ms_test("andor(jc:pk_k(A),after(500000001),after(500000001))", "oBf");
        ms_test("or_d(multi(2,A,B,C),or_d(j:multi(2,D,E,F),multi(2,G,I,J)))", "sdueB");
        ms_test("andor(multi(2,A,B,C),after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_k(A),after(1)))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(multi(2,A,B,C),a:after(500000001)))", "Bu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("andor(multi(2,A,B,C),after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("or_d(j:multi(2,A,B,C),after(500000001))", "Bf");
        ms_test("andor(multi(2,A,B,C),multi(2,D,E,F),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,multi(2,A,B,C)))", "Bdu");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),multi(2,A,B,C))", "Bdue");
        ms_test("or_i(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)),0)", "Bdu");
        ms_test("andor(j:multi(2,A,B,C),c:pk_h(D),after(1))", "Bf");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),after(1))", "Bu");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("thresh(2,multi(2,A,B,C),a:multi(2,D,E,F),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdus");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(1),multi(2,D,E,F))", "Bde");
        ms_test("or_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_k(A))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(D))", "sdueB");
        ms_test("and_v(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "nBf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,after(1)))", "Bde");
        ms_test("andor(j:multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(1),multi(2,A,B,C))", "Bdue");
        ms_test("and_b(after(1),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),1))", "Bfu");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_d(multi(2,A,B,C),multi(2,D,E,F)))", "Bdue");
        ms_test("or_d(or_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "Bdue");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),after(500000001))", "Bu");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))", "sdueB");
        ms_test("andor(jc:pk_h(A),after(1),multi(2,B,C,D))", "Bdse");
        ms_test("andor(or_i(multi(2,A,B,C),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "Bf");
        ms_test("or_d(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,D,E,F)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),c:pk_k(A))", "Bde");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("thresh(2,multi(2,A,B,C),a:multi(2,D,E,F),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdus");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_h(A),after(500000001))", "B");
        ms_test("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(or_d(c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),c:pk_h(D))", "Bdus");
        ms_test("or_b(multi(2,A,B,C),aj:multi(2,D,E,F))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),j:multi(2,D,E,F))", "Bdus");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(multi(2,A,B,C),after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("or_d(c:pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("n:andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "Bfu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),j:multi(2,A,B,C))", "Bd");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(jc:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "Bf");
        ms_test("andor(j:multi(2,A,B,C),multi(2,D,E,F),after(1))", "Bf");
        ms_test("or_b(multi(2,A,B,C),ajc:pk_h(D))", "sdueB");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_h(A))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(500000001)))", "Bfu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),j:multi(2,B,C,D))", "Bdus");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_k(D))", "Bdue");
        ms_test("and_b(after(1),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(500000001))", "Bf");
        ms_test("or_d(multi(2,A,B,C),or_d(multi(2,D,E,F),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,G,I,J))))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "B");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(1))", "oBfu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdu");
        ms_test("or_d(jc:pk_h(A),j:multi(2,B,C,D))", "Bdus");
        ms_test("or_d(dv:after(1),after(500000001))", "oBf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,after(500000001)))", "Bde");
        ms_test("andor(j:multi(2,A,B,C),after(500000001),multi(2,D,E,F))", "Bdse");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(1),j:multi(2,A,B,C))", "Bdu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),j:multi(2,A,B,C))", "Bdu");
        ms_test("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:multi(2,A,B,C)),1)", "Bfu");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),after(1))", "Bu");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:multi(2,A,B,C))", "Bdus");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_k(A))", "Bdue");
        ms_test("andor(or_i(multi(2,A,B,C),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "Bf");
        ms_test("andor(j:multi(2,A,B,C),multi(2,D,E,F),after(1))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_i(0,c:pk_h(A)),multi(2,B,C,D))", "sdueB");
        ms_test("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),c:pk_h(A))", "Bdue");
        ms_test("or_b(or_d(multi(2,A,B,C),multi(2,D,E,F)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdu");
        ms_test("or_b(c:pk_h(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("c:andor(j:multi(2,A,B,C),pk_k(D),pk_k(E))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),0)", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(1)),multi(2,D,E,F))", "Bde");
        ms_test("or_b(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_i(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),0)", "Bd");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "B");
        ms_test("or_d(jc:pk_h(A),after(500000001))", "Bf");
        ms_test("andor(jc:pk_k(A),multi(2,B,C,D),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),after(500000001))", "B");
        ms_test("or_d(or_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "Bdue");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),multi(2,D,E,F))", "sdueB");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:after(1))", "Bu");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),a:multi(2,D,E,F))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(1),a:and_b(after(1),a:after(1))),after(1))", "Bf");
        ms_test("or_d(c:pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_i(0,or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)))", "Bdu");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bdue");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))", "Bdue");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "B");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(multi(2,A,B,C),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("and_b(after(500000001),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bfu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)),after(500000001))", "Bf");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_k(A),after(500000001))", "B");
        ms_test("andor(j:multi(2,A,B,C),multi(2,D,E,F),multi(2,G,I,J))", "sdueB");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(jc:pk_k(A),c:pk_k(B))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),n:after(1))", "oBf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),multi(2,D,E,F)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C))", "Bdu");
        ms_test("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(1)),multi(2,A,B,C))", "Bsu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),multi(2,D,E,F)),after(500000001))", "B");
        ms_test("or_b(multi(2,A,B,C),ajc:pk_k(D))", "sdueB");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(j:multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(j:multi(2,A,B,C),multi(2,D,E,F),c:pk_h(G))", "sdueB");
        ms_test("and_v(v:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),1)", "oBfu");
        ms_test("or_d(or_d(or_d(multi(2,A,B,C),multi(2,D,E,F)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,G,I,J))", "Bdue");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(multi(2,A,B,C),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(j:multi(2,A,B,C),after(500000001))", "Bf");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)),after(500000001))", "Bf");
        ms_test("and_b(or_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:multi(2,D,E,F))", "Bdus");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),after(1))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C)),after(500000001))", "Bf");
        ms_test("or_d(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(jc:pk_k(A),multi(2,B,C,D),after(1))", "Bf");
        ms_test("andor(jc:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(j:multi(2,A,B,C),aj:multi(2,D,E,F))", "sdueB");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_h(A))", "Bdue");
        ms_test("or_d(j:multi(2,A,B,C),n:after(1))", "Bfu");
        ms_test("andor(j:multi(2,A,B,C),after(500000001),0)", "Bdse");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)),multi(2,B,C,D))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,c:pk_k(A)))", "Bdu");
        ms_test("c:andor(j:multi(2,A,B,C),pk_h(D),pk_h(E))", "sdueB");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)))", "Bdue");
        ms_test("or_d(j:multi(2,A,B,C),c:pk_h(D))", "sdueB");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "Bf");
        ms_test("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sc:pk_k(A))", "Bdu");
        ms_test("or_i(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),0)", "Bdu");
        ms_test("or_d(multi(2,A,B,C),or_d(j:multi(2,D,E,F),after(500000001)))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C))", "Bdue");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(jc:pk_h(A),c:pk_h(B))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_k(A),after(500000001)))", "Bf");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_h(A))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(j:multi(2,A,B,C),multi(2,D,E,F)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("thresh(2,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(j:multi(2,A,B,C),aj:multi(2,D,E,F))", "sdueB");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),multi(2,D,E,F))", "sdueB");
        ms_test("or_d(jc:pk_h(A),c:pk_k(B))", "sdueB");
        ms_test("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),after(500000001))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(1),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))))", "Bf");
        ms_test("and_b(after(1),a:and_b(after(1),a:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(1))))", "Bfu");
        ms_test("or_d(or_i(j:multi(2,A,B,C),0),after(500000001))", "Bf");
        ms_test("or_d(or_i(0,multi(2,A,B,C)),after(1))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(and_b(after(500000001),a:after(500000001)),a:after(500000001)),after(500000001))", "Bf");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))", "Bdu");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sc:pk_k(A))", "Bdue");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001),multi(2,D,E,F))", "Bde");
        ms_test("andor(j:multi(2,A,B,C),c:pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(c:pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(1),multi(2,A,B,C))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(after(500000001),0))", "Bd");
        ms_test("andor(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),after(1))", "B");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_k(A))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_d(c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(500000001)),after(500000001))", "oBf");
        ms_test("or_b(c:pk_k(A),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(v:after(500000001),multi(2,A,B,C)))", "Bu");
        ms_test("or_d(or_d(j:multi(2,A,B,C),multi(2,D,E,F)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)),multi(2,B,C,D))", "Bdue");
        ms_test("or_d(multi(2,A,B,C),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F),after(500000001)))", "B");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),after(500000001))", "Bf");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),after(1))", "Bf");
        ms_test("and_b(and_b(after(500000001),a:after(500000001)),a:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001)))", "Bfu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bde");
        ms_test("or_b(jc:pk_h(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(jc:pk_h(A),multi(2,B,C,D))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "B");
        ms_test("andor(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "B");
        ms_test("and_b(after(500000001),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C))", "Bdu");
        ms_test("or_b(j:multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)))", "Bdue");
        ms_test("andor(multi(2,A,B,C),after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("or_b(multi(2,A,B,C),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),c:pk_k(B))", "Bdus");
        ms_test("or_d(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,D,E,F)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),after(1))", "B");
        ms_test("or_d(j:multi(2,A,B,C),c:pk_h(D))", "sdueB");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_d(or_i(0,c:pk_h(A)),multi(2,B,C,D))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),n:after(1))", "Bu");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),adv:after(500000001))", "Bdue");
        ms_test("or_d(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),after(500000001))", "Bu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),multi(2,D,E,F)))", "Bdue");
        ms_test("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:multi(2,A,B,C)),multi(2,D,E,F))", "Bsu");
        ms_test("or_d(or_d(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("andor(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "B");
        ms_test("or_d(j:multi(2,A,B,C),j:multi(2,D,E,F))", "Bdus");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_h(A),multi(2,B,C,D)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,multi(2,A,B,C)))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C))", "Bdu");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("c:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),pk_k(B))", "Bdus");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),a:multi(2,A,B,C))", "Bsu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(500000001)))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(multi(2,A,B,C),0),after(500000001))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("or_d(dv:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(j:multi(2,A,B,C),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bde");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(1),a:after(1)))", "Bfu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),n:after(500000001))", "Bu");
        ms_test("or_d(or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C)),after(1))", "Bf");
        ms_test("or_b(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("and_b(after(500000001),a:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),after(500000001)))", "Bfu");
        ms_test("andor(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)),after(1))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(500000001),a:after(500000001)),and_b(after(500000001),a:after(500000001)))", "Bfu");
        ms_test("or_d(or_i(multi(2,A,B,C),0),after(1))", "Bf");
        ms_test("or_d(jc:pk_h(A),multi(2,B,C,D))", "sdueB");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_k(A))", "Bdu");
        ms_test("or_d(or_d(j:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(multi(2,A,B,C),or_b(multi(2,D,E,F),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_i(after(500000001),0))", "Bdue");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdue");
        ms_test("n:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bu");
        ms_test("or_d(jc:pk_h(A),c:pk_h(B))", "sdueB");
        ms_test("or_d(or_d(or_d(or_d(or_d(j:multi(2,A,B,C),multi(2,D,E,F)),multi(2,G,I,J)),multi(2,K,L,M)),multi(2,N,O,P)),multi(2,Q,R,S))", "sdueB");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_d(or_b(multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("thresh(2,multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,D,E,F))", "Bdue");
        ms_test("or_b(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(j:multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(0,or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "B");
        ms_test("andor(j:multi(2,A,B,C),after(1),after(1))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(multi(2,D,E,F),after(500000001)))", "B");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_h(A))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)))", "Bdue");
        ms_test("andor(j:multi(2,A,B,C),j:multi(2,D,E,F),after(500000001))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)))", "Bdue");
        ms_test("or_b(j:multi(2,A,B,C),a:multi(2,D,E,F))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0))", "Bd");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(after(500000001),a:and_b(after(500000001),a:after(500000001))),after(500000001))", "Bf");
        ms_test("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(500000001)),after(500000001))", "oBf");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),j:multi(2,B,C,D))", "Bdus");
        ms_test("or_d(or_i(0,multi(2,A,B,C)),multi(2,D,E,F))", "sdueB");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(multi(2,A,B,C),a:after(1)))", "Bu");
        ms_test("andor(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(dv:after(500000001),after(500000001),after(500000001))", "oBf");
        ms_test("and_b(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A),andor(c:pk_h(B),after(500000001),after(500000001))),a:after(500000001))", "Bu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C))", "Bdu");
        ms_test("or_d(or_d(c:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_i(0,after(500000001)))", "Bdue");
        ms_test("andor(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,A,B,C),after(500000001))", "B");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(after(500000001),0))", "Bde");
        ms_test("or_d(or_i(multi(2,A,B,C),0),multi(2,D,E,F))", "sdueB");
        ms_test("or_b(j:multi(2,A,B,C),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(multi(2,A,B,C),a:multi(2,D,E,F)))", "Bdue");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_b(multi(2,A,B,C),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))", "Bdue");
        ms_test("or_d(j:multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(or_i(multi(2,A,B,C),0),c:pk_h(D))", "sdueB");
        ms_test("or_b(jc:pk_h(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),after(1))", "Bu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),after(1))", "Bu");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),multi(2,D,E,F))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(c:pk_h(A),after(500000001)))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:after(1)))", "Bu");
        ms_test("andor(j:multi(2,A,B,C),j:multi(2,D,E,F),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "B");
        ms_test("or_d(jc:pk_k(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),c:pk_h(D))", "Bdus");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(multi(2,A,B,C),or_b(multi(2,D,E,F),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("andor(dv:after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "B");
        ms_test("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bu");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_i(after(1),0))", "Bdue");
        ms_test("andor(multi(2,A,B,C),multi(2,D,E,F),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),j:multi(2,D,E,F))", "Bdus");
        ms_test("and_v(v:after(1),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "oBf");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:or_i(after(1),0))", "Bdue");
        ms_test("or_d(or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_h(B))", "Bdue");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("and_v(or_c(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),v:after(1)),after(1))", "oBf");
        ms_test("andor(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "B");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),after(500000001))", "oBfu");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("or_d(jc:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(and_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(j:multi(2,A,B,C),c:pk_k(D),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("andor(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1),after(500000001))", "Bf");
        ms_test("andor(multi(2,A,B,C),after(500000001),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "oBf");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),c:pk_h(D))", "Bdue");
        ms_test("or_d(jc:pk_h(A),c:pk_k(B))", "sdueB");
        ms_test("or_d(and_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_d(or_i(0,c:pk_k(A)),multi(2,B,C,D))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(multi(2,A,B,C),0),multi(2,D,E,F))", "Bdus");
        ms_test("or_d(or_b(multi(2,A,B,C),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("or_d(or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_k(A))", "Bdue");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),after(500000001))", "Bu");
        ms_test("or_d(c:pk_k(A),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),multi(2,D,E,F)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),andor(multi(2,A,B,C),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bd");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)),multi(2,A,B,C))", "Bde");
        ms_test("or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),c:pk_k(D))", "Bdus");
        ms_test("or_d(j:multi(2,A,B,C),n:after(1))", "Bfu");
        ms_test("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),after(1))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("n:or_d(or_d(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bfu");
        ms_test("andor(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_i(0,multi(2,A,B,C)),after(500000001))", "Bf");
        ms_test("n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001))", "Bu");
        ms_test("or_d(andor(multi(2,A,B,C),multi(2,D,E,F),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001))", "Bf");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sdv:after(500000001))", "Bdue");
        ms_test("thresh(2,multi(2,A,B,C),a:multi(2,D,E,F),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdus");
        ms_test("or_d(c:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),pk_k(A),pk_h(B)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)),after(500000001))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(500000001)))", "Bf");
        ms_test("or_b(jc:pk_h(A),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),after(500000001))", "B");
        ms_test("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),after(500000001))", "Bf");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),and_v(v:multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bfu");
        ms_test("or_d(j:multi(2,A,B,C),c:pk_k(D))", "sdueB");
        ms_test("andor(or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1))", "B");
        ms_test("or_d(multi(2,A,B,C),or_b(multi(2,D,E,F),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:or_i(0,sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_b(multi(2,A,B,C),a:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))", "Bdue");
        ms_test("andor(andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(j:multi(2,A,B,C),multi(2,D,E,F),multi(2,G,I,J))", "sdueB");
        ms_test("or_d(multi(2,A,B,C),or_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdue");
        ms_test("or_d(jc:pk_k(A),multi(2,B,C,D))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C))", "Bdu");
        ms_test("or_d(n:andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("n:or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001)))", "Bfu");
        ms_test("or_d(multi(2,A,B,C),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F),after(1)))", "B");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),after(1))", "Bf");
        ms_test("andor(j:multi(2,A,B,C),j:multi(2,D,E,F),after(1))", "Bf");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_h(A))", "Bdu");
        ms_test("or_b(jc:pk_h(A),s:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(multi(2,A,B,C),a:multi(2,D,E,F)))", "Bdue");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_h(A))", "Bdu");
        ms_test("or_d(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),c:pk_h(A))", "Bdue");
        ms_test("or_b(multi(2,A,B,C),sjc:pk_k(D))", "sdueB");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)))", "Bf");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(500000001),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bd");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C),or_d(multi(2,D,E,F),after(1)))", "B");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(1)))", "Bfu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),after(1),j:multi(2,A,B,C))", "Bd");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(500000001))", "Bu");
        ms_test("andor(or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_b(multi(2,A,B,C),a:multi(2,D,E,F)))", "Bdue");
        ms_test("or_b(c:pk_h(A),aj:multi(2,B,C,D))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_k(A),multi(2,B,C,D))", "Bdus");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),j:multi(2,A,B,C),after(500000001))", "B");
        ms_test("or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),n:after(500000001)))", "Bfu");
        ms_test("andor(j:multi(2,A,B,C),after(500000001),after(500000001))", "Bf");
        ms_test("andor(jc:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,B,C,D))", "sdueB");
        ms_test("andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),c:pk_k(A))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_i(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),0),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_d(c:pk_h(A),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),multi(2,B,C,D))", "Bdue");
        ms_test("or_d(multi(2,A,B,C),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(D)))", "Bdue");
        ms_test("and_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,A,B,C)),a:after(1))", "Bu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),after(1)))", "B");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_k(A),multi(2,B,C,D))", "Bdus");
        ms_test("or_d(jc:pk_h(A),after(1))", "Bf");
        ms_test("or_b(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),a:sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdue");
        ms_test("or_d(multi(2,A,B,C),andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)))", "Bdu");
        ms_test("andor(multi(2,A,B,C),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),multi(2,D,E,F)))", "Bdue");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(or_d(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),c:pk_h(A)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("or_d(and_b(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),a:multi(2,A,B,C)),sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc))", "Bdu");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),or_d(multi(2,A,B,C),multi(2,D,E,F)),after(1))", "B");
        ms_test("or_b(j:multi(2,A,B,C),a:multi(2,D,E,F))", "sdueB");
        ms_test("c:andor(j:multi(2,A,B,C),pk_k(D),pk_k(E))", "sdueB");
        ms_test("andor(sha256(926a54995ca48600920a19bf7bc502ca5f2f7d07e6f804c4f00ebf0325084dbc),jc:pk_h(A),sha256(926a54995c