//! Run code in virtual machines (sandboxes)
//!
//! # Summary
//!
//! This crate provides an abstract API for the Rust programming language to
//! allow executing scripting language code snippets in a sandbox.
//!
//! The scripting language (or VM which executes code written in that scripting
//! language) is not specified by this crate. Instead, other crates may use
//! this crate to provide a scripting-language-independent interface to execute
//! certain functions or provide callbacks from the VM to Rust.
//!
//! Central trait is a [`Machine`], which is capable to compile certain code
//! into a function (see trait [`Compile`] for compiling code into functions,
//! and see associated type [`Machine::Function`] and trait [`Function`] for
//! executing these functions).
//!
//! Passing values from Rust to the VM or from the VM to Rust is done through a
//! "datum" type that is specific to each particular VM used (see associated
//! type [`Machine::Datum`]). Some VMs may support creating closures in Rust
//! which can then be converted into a "datum" and passed to the VM (see trait
//! [`Callback`]).
//!
//! VMs may also support setting global variables or variables in modules (see
//! traits [`Globals`] and [`HasModules`]).
//!
//! As different VMs may have "datum" types with different features (e.g. some
//! scripting languages might support integers while others only know strings,
//! a [`Machine::Datum`] can implement different traits in the [`types`] module
//! which determine which functionality is supported
//! (e.g. [`types::MaybeString`] if a datum can be a string).

#![feature(generic_associated_types)]
#![feature(associated_type_defaults)]
#![warn(missing_docs)]

pub mod errors;
pub mod types;

/// Unnameable traits to be included via wildcard syntax
pub mod traits {
    pub use super::types::{
        HasArray as _, HasStringMap as _, MaybeArray as _, MaybeBinary as _, MaybeBoolean as _,
        MaybeFloat as _, MaybeFunction as _, MaybeInteger as _, MaybeOpaque as _, MaybeString as _,
        MaybeStringMap as _, Nullable as _,
    };
    pub use super::{
        Callback as _, Compile as _, Function as _, Globals as _, HasModules as _, Machine as _,
        Module as _,
    };
}

/// Prelude that (currently) re-exports everything
pub mod prelude {
    pub use super::{
        errors::*, types::*, Callback, Compile, Function, Globals, HasModules, Machine, Module,
    };
}

pub use errors::*;

use std::error::Error;

/// Virtual machine to execute code (with inner mutability)
///
/// # Lifetimes
///
/// The lifetime argument `'a` is a lower bound for closures passed to the
/// machine (see [`Callback`]), which will usually be inferred automatically.
///
/// The generic associated types have a lifetime argument `'b`, which is
/// usually equal to the lifetime of the shared reference to the machine that
/// is being worked with. The lifetime argument `'c` of [`Machine::Datum`] is
/// usually `'static` but may be shorter for temporary values that have been
/// created from a `&str`, for example.
///
/// # Example
///
/// ```
/// use sandkiste::prelude::*;
///
/// fn say_hello<'a, 'b, M, C>(
///     machine: &'b M,
///     code: C,
/// ) -> Result<String, MachineError>
/// where
///     M: Machine<'a> + Compile<'a, C>,
///     for<'c> <M as Machine<'a>>::Datum<'b, 'c>: MaybeString<'c>,
/// {
///     Ok(machine
///         .compile(None, code)?
///         .call_1ret(["Hi there! Who are you?".into()])?
///         .try_into()?
///     )
/// }
/// ```
pub trait Machine<'a> {
    /// Data type representing values passed to or returned from machine
    type Datum<'b, 'c>
    where
        Self: 'b;
    /// Function (for example returned by [`Compile::compile`])
    /// which can be executed by the machine
    type Function<'b>: for<'c> Function<Datum<'c> = Self::Datum<'b, 'c>>
    where
        Self: 'b;
}

/// Ability of [`Machine`]s to compile provided code (of a certain type, e.g. `&str` or `String`)
///
/// # Owned vs borrowed code
///
/// Implementations should provide an implementation both for the owned and the
/// borrowed variant of the type representing the code (e.g. [`String`] and
/// [`&str`](str), or `Vec<u8>` and `&[u8]`). The `chunk_name` is always passed
/// as owned `String`, as it will usually be stored in the returned
/// [`Function`].
///
/// # Example
///
/// See [`Machine`].
pub trait Compile<'a, C>: Machine<'a> {
    /// Compiles code, returns the compiled function or a compilation error
    ///
    /// `chunk_name` is an optional chunk name for diagnostic output.
    fn compile(
        &self,
        chunk_name: Option<String>,
        code: C,
    ) -> Result<Self::Function<'_>, MachineError>;
}

fn expect_cardinality<F>(
    function: &F,
    retvals: Vec<F::Datum<'static>>,
    count: usize,
    strict: bool,
) -> Result<Vec<F::Datum<'static>>, MachineError>
where
    F: Function + ?Sized,
{
    let len = retvals.len();
    if (strict && len == count) || (!strict && len >= count) {
        Ok(retvals)
    } else {
        Err(MachineError {
            kind: MachineErrorKind::Data,
            message: Some(format!(
                "expected {} {} but got {} return value{}",
                if strict { "exactly" } else { "at least" },
                count,
                retvals.len(),
                if retvals.len() != 1 { "s" } else { "" }
            )),
            chunk_name: function.chunk_name().map(|x| x.to_string()),
            ..Default::default()
        })
    }
}

/// Basic interface of function that runs in a [`Machine`]
///
/// # Example
///
/// ```
/// use sandkiste::prelude::*;
///
/// fn perform_numeric_operation<F>(func: F, a: i32, b: i32) -> i32
/// where
///     F: Function,
///     for<'c> <F as Function>::Datum<'c>: MaybeInteger,
/// {
///     func
///         .call_1ret([a.into(), b.into()])
///         .expect("runtime error")
///         .try_as_i32().expect("type error")
/// }
/// ```
pub trait Function {
    /// Type used for arguments and return values
    type Datum<'c>;
    /// Chunk name which was used to create function (if known)
    fn chunk_name(&self) -> Option<&str>;
    /// Call the function
    fn call<'c, A>(&self, args: A) -> Result<Vec<Self::Datum<'static>>, MachineError>
    where
        A: IntoIterator<Item = Self::Datum<'c>>,
        <A as IntoIterator>::IntoIter: ExactSizeIterator;
    /// Same as [`Function::call`] but expects a fixed number of return values
    fn call_expect_retvals<'c, A>(
        &self,
        count: usize,
        args: A,
    ) -> Result<Vec<Self::Datum<'static>>, MachineError>
    where
        A: IntoIterator<Item = Self::Datum<'c>>,
        <A as IntoIterator>::IntoIter: ExactSizeIterator,
    {
        expect_cardinality(self, self.call(args)?, count, true)
    }
    /// Same as [`Function::call`] but expects a minimum number of return values
    fn call_expect_min_retvals<'c, A>(
        &self,
        count: usize,
        args: A,
    ) -> Result<Vec<Self::Datum<'static>>, MachineError>
    where
        A: IntoIterator<Item = Self::Datum<'c>>,
        <A as IntoIterator>::IntoIter: ExactSizeIterator,
    {
        expect_cardinality(self, self.call(args)?, count, false)
    }
    /// Same as [`Function::call`] but expects exactly one return value
    fn call_1ret<'c, A>(&self, args: A) -> Result<Self::Datum<'static>, MachineError>
    where
        A: IntoIterator<Item = Self::Datum<'c>>,
        <A as IntoIterator>::IntoIter: ExactSizeIterator,
    {
        Ok(self
            .call_expect_retvals(1, args)?
            .into_iter()
            .next()
            .unwrap())
    }
    /// Same as [`Function::call`] but expects exactly two return values
    fn call_2ret<'c, A>(
        &self,
        args: A,
    ) -> Result<(Self::Datum<'static>, Self::Datum<'static>), MachineError>
    where
        A: IntoIterator<Item = Self::Datum<'c>>,
        <A as IntoIterator>::IntoIter: ExactSizeIterator,
    {
        let mut retvals = self.call_expect_retvals(2, args)?.into_iter();
        Ok((retvals.next().unwrap(), retvals.next().unwrap()))
    }
    /// Same as [`Function::call`] but expects exactly three return values
    fn call_3ret<'c, A>(
        &self,
        args: A,
    ) -> Result<
        (
            Self::Datum<'static>,
            Self::Datum<'static>,
            Self::Datum<'static>,
        ),
        MachineError,
    >
    where
        A: IntoIterator<Item = Self::Datum<'c>>,
        <A as IntoIterator>::IntoIter: ExactSizeIterator,
    {
        let mut retvals = self.call_expect_retvals(3, args)?.into_iter();
        Ok((
            retvals.next().unwrap(),
            retvals.next().unwrap(),
            retvals.next().unwrap(),
        ))
    }
}

/// Ability of [`Machine`]s to call provided callbacks
///
/// # Example
///
/// ```
/// use std::rc::Rc;
/// use std::cell::RefCell;
/// use sandkiste::prelude::*;
///
/// fn collect_output<'a, 'b, M, C>(
///     machine: &'b M,
///     setup: C,
///     run: C,
/// ) -> Result<String, MachineError>
/// where
///     M: Machine<'a> + Compile<'a, C> + Callback<'a>,
///     for<'c> <M as Machine<'a>>::Datum<'b, 'c>: MaybeString<'c>,
/// {
///     let output_cell = RefCell::new(String::new());
///     let output_rc = Rc::new(output_cell);
///     let output_weak = Rc::downgrade(&output_rc);
///
///     let my_print = machine
///         .callback_1arg(move |s| {
///             output_weak
///                 .upgrade()
///                 .ok_or("closure expired")?
///                 .borrow_mut()
///                 .push_str(s.try_as_str()?);
///             Ok([])
///         })?;
///
///     machine.compile(None, setup)?.call([my_print])?;
///     machine.compile(None, run)?.call([])?;
///
///     Ok(Rc::try_unwrap(output_rc).unwrap().into_inner())
/// }
/// ```
pub trait Callback<'a>: Machine<'a> {
    /// Create a [`Machine::Datum`] representing a callback (which invokes the `func` closure)
    fn callback<'b, 'c, F, R>(&'b self, func: F) -> Result<Self::Datum<'b, 'static>, MachineError>
    where
        R: IntoIterator<Item = Self::Datum<'b, 'c>>,
        <R as IntoIterator>::IntoIter: ExactSizeIterator,
        F: 'a + Fn(Vec<Self::Datum<'b, 'static>>) -> Result<R, Box<dyn Error>>;
    /// Same as [`Callback::callback`] but expects a fixed number of arguments
    fn callback_expect_args<'b, 'c, F, R>(
        &'b self,
        argc: usize,
        func: F,
    ) -> Result<Self::Datum<'b, 'static>, MachineError>
    where
        R: IntoIterator<Item = Self::Datum<'b, 'c>>,
        <R as IntoIterator>::IntoIter: ExactSizeIterator,
        F: 'a + Fn(Vec<Self::Datum<'b, 'static>>) -> Result<R, Box<dyn Error>>,
    {
        self.callback(move |args| {
            if args.len() != argc {
                Err(format!(
                    "expected exactly {} but got {} argument{}",
                    argc,
                    args.len(),
                    if args.len() != 1 { "s" } else { "" }
                ))?;
            }
            func(args)
        })
    }
    /// Same as [`Callback::callback`] but expects a minimum number of arguments
    fn callback_expect_min_args<'b, 'c, F, R>(
        &'b self,
        argc: usize,
        func: F,
    ) -> Result<Self::Datum<'b, 'static>, MachineError>
    where
        R: IntoIterator<Item = Self::Datum<'b, 'c>>,
        <R as IntoIterator>::IntoIter: ExactSizeIterator,
        F: 'a + Fn(Vec<Self::Datum<'b, 'static>>) -> Result<R, Box<dyn Error>>,
    {
        self.callback(move |args| {
            if args.len() < argc {
                Err(format!(
                    "expected at least {} but got {} argument{}",
                    argc,
                    args.len(),
                    if args.len() != 1 { "s" } else { "" }
                ))?;
            }
            func(args)
        })
    }
    /// Same as [`Callback::callback`] but expects exactly one argument
    fn callback_1arg<'b, 'c, F, R>(
        &'b self,
        func: F,
    ) -> Result<Self::Datum<'b, 'static>, MachineError>
    where
        R: IntoIterator<Item = Self::Datum<'b, 'c>>,
        <R as IntoIterator>::IntoIter: ExactSizeIterator,
        F: 'a + Fn(Self::Datum<'b, 'static>) -> Result<R, Box<dyn Error>>,
    {
        self.callback_expect_args(1, move |args| {
            let mut args = args.into_iter();
            func(args.next().unwrap())
        })
    }
    /// Same as [`Callback::callback`] but expects exactly two arguments
    fn callback_2arg<'b, 'c, F, R>(
        &'b self,
        func: F,
    ) -> Result<Self::Datum<'b, 'static>, MachineError>
    where
        R: IntoIterator<Item = Self::Datum<'b, 'c>>,
        <R as IntoIterator>::IntoIter: ExactSizeIterator,
        F: 'a + Fn(Self::Datum<'b, 'static>, Self::Datum<'b, 'static>) -> Result<R, Box<dyn Error>>,
    {
        self.callback_expect_args(2, move |args| {
            let mut args = args.into_iter();
            func(args.next().unwrap(), args.next().unwrap())
        })
    }
    /// Same as [`Callback::callback`] but expects exactly three arguments
    fn callback_3arg<'b, 'c, F, R>(
        &'b self,
        func: F,
    ) -> Result<Self::Datum<'b, 'static>, MachineError>
    where
        R: IntoIterator<Item = Self::Datum<'b, 'c>>,
        <R as IntoIterator>::IntoIter: ExactSizeIterator,
        F: 'a
            + Fn(
                Self::Datum<'b, 'static>,
                Self::Datum<'b, 'static>,
                Self::Datum<'b, 'static>,
            ) -> Result<R, Box<dyn Error>>,
    {
        self.callback_expect_args(3, move |args| {
            let mut args = args.into_iter();
            func(
                args.next().unwrap(),
                args.next().unwrap(),
                args.next().unwrap(),
            )
        })
    }
    /// Same as [`Callback::callback`] but expects exactly four arguments
    fn callback_4arg<'b, 'c, F, R>(
        &'b self,
        func: F,
    ) -> Result<Self::Datum<'b, 'static>, MachineError>
    where
        R: IntoIterator<Item = Self::Datum<'b, 'c>>,
        <R as IntoIterator>::IntoIter: ExactSizeIterator,
        F: 'a
            + Fn(
                Self::Datum<'b, 'static>,
                Self::Datum<'b, 'static>,
                Self::Datum<'b, 'static>,
                Self::Datum<'b, 'static>,
            ) -> Result<R, Box<dyn Error>>,
    {
        self.callback_expect_args(4, move |args| {
            let mut args = args.into_iter();
            func(
                args.next().unwrap(),
                args.next().unwrap(),
                args.next().unwrap(),
                args.next().unwrap(),
            )
        })
    }
    /// Same as [`Callback::callback`] but expects exactly five arguments
    fn callback_5arg<'b, 'c, F, R>(
        &'b self,
        func: F,
    ) -> Result<Self::Datum<'b, 'static>, MachineError>
    where
        R: IntoIterator<Item = Self::Datum<'b, 'c>>,
        <R as IntoIterator>::IntoIter: ExactSizeIterator,
        F: 'a
            + Fn(
                Self::Datum<'b, 'static>,
                Self::Datum<'b, 'static>,
                Self::Datum<'b, 'static>,
                Self::Datum<'b, 'static>,
                Self::Datum<'b, 'static>,
            ) -> Result<R, Box<dyn Error>>,
    {
        self.callback_expect_args(5, move |args| {
            let mut args = args.into_iter();
            func(
                args.next().unwrap(),
                args.next().unwrap(),
                args.next().unwrap(),
                args.next().unwrap(),
                args.next().unwrap(),
            )
        })
    }
}

/// [`Machine`]s that support global variables
///
/// # Example
///
/// ```
/// use sandkiste::prelude::*;
///
/// fn call_func_with_globals<'a, 'b, M, F>(machine: &'b M, func: F, a: i32, b: i32) -> i32
/// where
///     M: Machine<'a> + Globals<'a>,
///     for<'c> <M as Machine<'a>>::Datum<'b, 'c>: MaybeInteger,
///     F: Function,
/// {
///     machine.set("a", a.into());
///     machine.set("b", b.into());
///     func.call([]).expect("runtime error");
///     machine.get("c").expect("runtime error").try_as_i32().expect("type error")
/// }
/// ```
pub trait Globals<'a>: Machine<'a> {
    /// Get global variable
    fn get<'b>(&'b self, name: &str) -> Result<Self::Datum<'b, 'static>, MachineError>;
    /// Set global variable
    fn set<'b, 'c>(&'b self, name: &str, datum: Self::Datum<'b, 'c>) -> Result<(), MachineError>;
}

/// Ability to load language-independent modules into a [`Machine`]
///
/// # Example
///
/// ```
/// use sandkiste::prelude::*;
///
/// fn load_my_module<'a, 'b, M>(machine: &'b M) -> Result<(), MachineError>
/// where
///     M: Machine<'a> + HasModules<'a> + Callback<'a>,
///     for<'c> M::Datum<'b, 'c>: MaybeFloat,
/// {
///     let mymod = machine.module("mymod")?;
///     mymod.set("pi_approx", 3.14.into())?;
///     mymod.set("double", machine.callback_1arg(|x| {
///         Ok([(x.try_as_f64()? * 2.0).into()])
///     })?)?;
///     Ok(())
/// }
/// ```
pub trait HasModules<'a> {
    /// Type of modules and sub-modules
    type Module<'b>: Module<'a, 'b, Machine = Self>
    where
        Self: 'b;
    /// Open a module (create if non-existent)
    fn module<'b>(&'b self, name: &str) -> Result<Self::Module<'b>, MachineError>;
}

/// Module of [`Machine`] where values can be read and set (access with [`HasModules::module`])
pub trait Module<'a, 'b>: Sized {
    /// Machine type which the module belongs to
    type Machine: Machine<'a>;
    /// Open a sub-module (create if non-existent)
    fn module(&self, name: &str) -> Result<Self, MachineError>;
    /// Get value from module
    fn get(
        &self,
        name: &str,
    ) -> Result<<Self::Machine as Machine<'a>>::Datum<'b, 'static>, MachineError>;
    /// Set value in module
    fn set<'c>(
        &self,
        name: &str,
        datum: <Self::Machine as Machine<'a>>::Datum<'b, 'c>,
    ) -> Result<(), MachineError>;
}
