//! Error traits and error kinds

#[cfg(doc)]
use super::*;

use std::error::Error;
use std::fmt;

/// Kinds of [`MachineError`]s
#[derive(Copy, Clone, PartialEq, Eq, Hash, Debug)]
pub enum MachineErrorKind {
    /// Out-of-memory error
    Memory,
    /// Syntax error during compilation (see [`Compile::compile`])
    Syntax,
    /// Execution limit error
    ExecLimit,
    /// Generic runtime error (see [`Function::call`])
    Runtime,
    /// Unexpected value(s) (e.g. wrong type or wrong count of return values)
    Data,
    /// Error kind unknown
    Unknown,
}

impl Default for MachineErrorKind {
    fn default() -> Self {
        MachineErrorKind::Unknown
    }
}

/// Runtime or compiler errors that may refer to a specific position in code
#[derive(Clone, Default)]
pub struct MachineError {
    /// Kind of runtime error
    pub kind: MachineErrorKind,
    /// Error message
    pub message: Option<String>,
    /// Backtrace (multiple lines)
    pub machine_backtrace: Option<String>,
    /// Chunk name already included in message?
    pub message_incl_chunk_name: bool,
    /// Chunk name
    pub chunk_name: Option<String>,
    /// Position already included in message?
    pub message_incl_pos: bool,
    /// Line number
    pub line: Option<usize>,
    /// Column number (in UTF-8 codepoints)
    pub column: Option<usize>,
    /// Byte position
    pub position: Option<usize>,
}

impl Error for MachineError {}

impl fmt::Display for MachineError {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self.kind {
            MachineErrorKind::Memory => write!(f, "VM memory exhausted")?,
            MachineErrorKind::Syntax => write!(f, "syntax error in code for VM")?,
            MachineErrorKind::ExecLimit => write!(f, "VM execution limit exhausted")?,
            MachineErrorKind::Runtime => write!(f, "runtime error in VM")?,
            MachineErrorKind::Data => write!(f, "unexpected data from VM")?,
            _ => write!(f, "VM error")?,
        }
        let mut chunk_name_written = false;
        if !self.message_incl_chunk_name {
            if let Some(ref chunk_name) = self.chunk_name {
                write!(f, " in chunk \"{}\"", chunk_name)?;
                chunk_name_written = true;
            }
        }
        if !self.message_incl_pos {
            if let Some(line) = self.line {
                write!(
                    f,
                    "{} line {}",
                    if chunk_name_written { ", " } else { " in" },
                    line
                )?;
                if let Some(column) = self.column {
                    write!(f, ", column {}", column)?;
                }
            } else if let Some(position) = self.position {
                write!(f, ", at byte position {}", position)?;
            }
        }
        if let Some(msg) = &self.message {
            write!(f, ": {}", msg)?;
        }
        Ok(())
    }
}

impl fmt::Debug for MachineError {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        f.debug_struct("MachineError")
            .field("kind", &self.kind)
            .field("message", &self.message)
            .field("message_incl_pos", &self.message_incl_pos)
            .field("chunk_name", &self.chunk_name)
            .field("line", &self.line)
            .field("column", &self.column)
            .field("position", &self.position)
            .finish_non_exhaustive()
    }
}

/// Error when viewing [`Machine::Datum`] as a particular type
///
/// See `Maybe…` traits in module [`types`].
///
/// For convenience, this error type can be converted into a [`MachineError`].
#[derive(Debug)]
pub struct DatumViewError {
    /// Error description
    pub details: &'static str,
}

impl DatumViewError {
    /// Create new `DatumViewError` with static error description
    pub fn new(details: &'static str) -> Self {
        DatumViewError { details }
    }
}

impl fmt::Display for DatumViewError {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f, "{}", self.details)
    }
}

impl Error for DatumViewError {}

impl From<DatumViewError> for MachineError {
    fn from(datum_err: DatumViewError) -> Self {
        MachineError {
            kind: MachineErrorKind::Data,
            message: Some(datum_err.to_string()),
            ..Default::default()
        }
    }
}

/// Error when converting [`Machine::Datum`] into a particular type
///
/// See `Maybe…` traits in module [`types`].
///
/// For convenience, this error type can be converted into a [`MachineError`].
pub struct DatumConversionError<D> {
    /// Original datum that could not be converted
    pub original: D,
    /// Error description
    pub details: &'static str,
}

impl<D> DatumConversionError<D> {
    /// Create new `DatumConversionError` with static error description
    pub fn new(original: D, details: &'static str) -> Self {
        DatumConversionError { original, details }
    }
}

impl<D> fmt::Display for DatumConversionError<D> {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f, ": {}", self.details)
    }
}

impl<D> fmt::Debug for DatumConversionError<D> {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        f.debug_struct("DatumConversionError")
            .field("details", &self.details)
            .finish_non_exhaustive()
    }
}

impl<D> Error for DatumConversionError<D> {}

impl<D> From<DatumConversionError<D>> for MachineError {
    fn from(datum_err: DatumConversionError<D>) -> Self {
        MachineError {
            kind: MachineErrorKind::Data,
            message: Some(datum_err.to_string()),
            ..Default::default()
        }
    }
}
