//! Traits to be implemented by [`Machine::Datum`] to support specific types
//!
//! # Converting from and to VM-specific data types
//!
//! ## From VM-specific data types
//!
//! Scalar data types have `try_as_…` methods, which attempt to interpret a
//! datum as certain type, e.g. [`MaybeInteger::try_as_i32`] or
//! [`MaybeString`::`try_as_str`]. These methods return a [`DatumViewError`]
//! if the type doesn't match.
//!
//! Unicode and binary strings must furthermore implement [`TryInto`] (ideally
//! by implementing [`TryFrom`]) with [`DatumConversionError`] as error type to
//! allow conversion into an owned `String` or `Vec<u8>`. This conversion is
//! made available as [`try_into_string`](MaybeString::try_into_string) and
//! [`try_into_binary`](MaybeBinary::try_into_binary) to specify the type
//! through the method call (as [`TryInto::try_into`] isn't turbofishable).
//!
//! Collections are accessed through various methods in the [`MaybeArray`] and
//! [`MaybeStringMap`] traits of which some may fail due to runtime errors in
//! the VM and thus can return [`MachineError`]s on failure.
//!
//! ## To VM-specific data types
//!
//! Scalar data types support [`From`] which allows creation directly from
//! standard Rust data types (such as `&str`, `String`, or `i32`).
//!
//! Collections are created by trait methods that are implemented by the
//! machine, e.g. [`HasArray::new_array`].

use super::*;

use std::hash::Hash;
use std::num::TryFromIntError;
use std::ops::Deref;

/// Types that can be null
pub trait Nullable
where
    Self: Sized,
{
    /// Datum representing a null value
    fn null() -> Self;
    /// Can the datum be interpreted as a null value?
    fn is_null(&self) -> bool;
}

/// Types that can be boolean
pub trait MaybeBoolean
where
    Self: From<bool>,
{
    /// Try to interpret datum as boolean value
    fn try_as_bool(&self) -> Result<bool, DatumViewError>;
}

/// Types that can be a function
pub trait MaybeFunction {
    /// Function type (see [`Machine::Function`])
    type Function: Function;
    /// Reference or smart pointer to [`MaybeFunction::Function`] as returned
    /// by [`try_as_function`](MaybeFunction::try_as_function)
    type FunctionRef<'a>: Deref<Target = Self::Function>
    where
        Self: 'a,
    = &'a Self::Function;
    /// Convert from `Machine::Function`
    fn from_function(value: Self::Function) -> Self;
    /// Try to interpret datum as function
    fn try_as_function(&self) -> Result<Self::FunctionRef<'_>, DatumViewError>;
}

/// Types that can be a reference to an opaque datum allowing comparison for equality and hashing
pub trait MaybeOpaque {
    /// Type used to represent reference to an opaque datum
    type Opaque: Clone + Eq + PartialEq + Hash;
    /// Reference or smart pointer to [`MaybeOpaque::Opaque`] as returned by
    /// [`try_as_opaque`](MaybeOpaque::try_as_opaque)
    type OpaqueRef<'a>: Deref<Target = Self::Opaque>
    where
        Self: 'a,
    = &'a Self::Opaque;
    /// Try to interpret datum as reference to opaque datum
    fn try_as_opaque(&self) -> Result<Self::OpaqueRef<'_>, DatumViewError>;
}

/// Types that can be an UTF-8 text string
pub trait MaybeString<'c>
where
    Self: From<String>,
    Self: From<&'c str>,
    Self: TryInto<String, Error = DatumConversionError<Self>>,
{
    /// Try to convert datum into Unicode string
    fn try_into_string(self) -> Result<String, DatumConversionError<Self>> {
        self.try_into()
    }
    /// Try to interpret datum as Unicode string
    fn try_as_str(&self) -> Result<&str, DatumViewError>;
}

/// Types that can be a binary blob
pub trait MaybeBinary<'c>
where
    Self: From<Vec<u8>>,
    Self: From<&'c [u8]>,
    Self: TryInto<Vec<u8>, Error = DatumConversionError<Self>>,
{
    /// Try to convert datum into binary (8 bit) string
    fn try_into_binary(self) -> Result<Vec<u8>, DatumConversionError<Self>> {
        self.try_into()
    }
    /// Try to interpret datum as binary (8 bit) string
    fn try_as_bin(&self) -> Result<&[u8], DatumViewError>;
}

/// Types that can be a float
pub trait MaybeFloat
where
    Self: From<f64>,
    Self: From<f32>,
{
    /// Try to interpret datum as `f64`
    fn try_as_f64(&self) -> Result<f64, DatumViewError>;
    /// Try to interpret datum as `f64`
    fn try_as_f32(&self) -> Result<f32, DatumViewError> {
        Ok(self.try_as_f64()? as f32)
    }
}

/// Types that can be an integer (at least 32 bits signed integers must be supported)
pub trait MaybeInteger
where
    Self: Sized,
    Self: TryFrom<usize, Error = TryFromIntError>,
    Self: TryFrom<isize, Error = TryFromIntError>,
    Self: TryFrom<u64, Error = TryFromIntError>,
    Self: TryFrom<i64, Error = TryFromIntError>,
    Self: TryFrom<u32, Error = TryFromIntError>,
    Self: From<i32>,
    Self: From<i16>,
    Self: From<u16>,
    Self: From<i8>,
    Self: From<u8>,
{
    /// Try to interpret datum as `i64`
    fn try_as_i64(&self) -> Result<i64, DatumViewError>;
    /// Try to interpret datum as `u64`
    fn try_as_u64(&self) -> Result<u64, DatumViewError> {
        self.try_as_i64().and_then(|x| {
            x.try_into()
                .map_err(|_| DatumViewError::new("integer value does not fit into u64"))
        })
    }
    /// Try to interpret datum as `i32`
    fn try_as_i32(&self) -> Result<i32, DatumViewError> {
        self.try_as_i64().and_then(|x| {
            x.try_into()
                .map_err(|_| DatumViewError::new("integer value does not fit into i32"))
        })
    }
    /// Try to interpret datum as `u32`
    fn try_as_u32(&self) -> Result<u32, DatumViewError> {
        self.try_as_i64().and_then(|x| {
            x.try_into()
                .map_err(|_| DatumViewError::new("integer value does not fit into u32"))
        })
    }
    /// Try to interpret datum as `i16`
    fn try_as_i16(&self) -> Result<i16, DatumViewError> {
        self.try_as_i64().and_then(|x| {
            x.try_into()
                .map_err(|_| DatumViewError::new("integer value does not fit into i16"))
        })
    }
    /// Try to interpret datum as `u16`
    fn try_as_u16(&self) -> Result<u16, DatumViewError> {
        self.try_as_i64().and_then(|x| {
            x.try_into()
                .map_err(|_| DatumViewError::new("integer value does not fit into u16"))
        })
    }
    /// Try to interpret datum as `i8`
    fn try_as_i8(&self) -> Result<i8, DatumViewError> {
        self.try_as_i64().and_then(|x| {
            x.try_into()
                .map_err(|_| DatumViewError::new("integer value does not fit into i8"))
        })
    }
    /// Try to interpret datum as `u8`
    fn try_as_u8(&self) -> Result<u8, DatumViewError> {
        self.try_as_i64().and_then(|x| {
            x.try_into()
                .map_err(|_| DatumViewError::new("integer value does not fit into u8"))
        })
    }
    /// Try to interpret datum as `isize`
    fn try_as_isize(&self) -> Result<usize, DatumViewError> {
        self.try_as_i64().and_then(|x| {
            x.try_into()
                .map_err(|_| DatumViewError::new("integer value does not fit into isize"))
        })
    }
    /// Try to interpret datum as `usize`
    fn try_as_usize(&self) -> Result<usize, DatumViewError> {
        self.try_as_i64().and_then(|x| {
            x.try_into()
                .map_err(|_| DatumViewError::new("integer value does not fit into usize"))
        })
    }
}

/// Iterator over an array-like datum
///
/// Returned by [`MaybeArray::array_to_iter`].
pub struct ArrayIter<'a, D: ?Sized> {
    datum: &'a D,
    len: usize,
    index: usize,
}

impl<'a, D: ?Sized> Iterator for ArrayIter<'a, D>
where
    D: MaybeArray,
{
    type Item = Result<<D as MaybeArray>::Element<'static>, MachineError>;
    fn next(&mut self) -> Option<Result<<D as MaybeArray>::Element<'static>, MachineError>> {
        if self.index >= self.len {
            None
        } else {
            match self.datum.array_get(self.index) {
                Ok(element) => {
                    self.index += 1;
                    Some(Ok(element))
                }
                Err(err) => {
                    self.len = 0;
                    Some(Err(err))
                }
            }
        }
    }
}

/// Types that can be an array
pub trait MaybeArray {
    /// Type of elements
    type Element<'c>;
    /// Return error unless datum is an array-like type
    fn try_array(&self) -> Result<(), DatumViewError>;
    /// Array length
    fn array_len(&self) -> Result<usize, MachineError>;
    /// Retrieve element at index
    ///
    /// NOTE: If index is out of bounds, may either return error or
    /// [null](Nullable::null)
    fn array_get(&self, index: usize) -> Result<Self::Element<'static>, MachineError>;
    /// Set element at index
    ///
    /// NOTE: If index is out of bounds, may either grow array or return error
    fn array_set<'c>(&self, index: usize, element: Self::Element<'c>) -> Result<(), MachineError>;
    /// Push element to array
    fn array_push<'c>(&self, element: Self::Element<'c>) -> Result<(), MachineError>;
    /// Truncate array
    fn array_truncate(&self, len: usize) -> Result<(), MachineError>;
    /// Create [`Iterator`] over entries of array-like datum
    fn array_to_iter<'b>(&'b self) -> Result<ArrayIter<'b, Self>, MachineError> {
        self.try_array()?;
        Ok(ArrayIter {
            datum: self,
            len: self.array_len()?,
            index: 0,
        })
    }
    /// Create [`Vec`] from array-like datum
    fn array_to_vec(&self, maxlen: usize) -> Result<Vec<Self::Element<'static>>, MachineError> {
        self.try_array()?;
        let len = self.array_len()?;
        if len > maxlen {
            Err(MachineError {
                kind: MachineErrorKind::Data,
                message: Some(format!(
                    "array length {} exceeded maximum length {}",
                    len, maxlen
                )),
                ..Default::default()
            })?;
        }
        ArrayIter {
            datum: self,
            len,
            index: 0,
        }
        .collect()
    }
}

/// Types that can be a string map (mapping strings to other datums)
pub trait MaybeStringMap {
    /// Type of values
    type Value<'c>;
    /// Return error unless datum is a string-map-like type
    fn try_string_map(&self) -> Result<(), DatumViewError>;
    /// Get entry from string map
    fn string_map_get(&self, key: &str) -> Result<Self::Value<'static>, MachineError>;
    /// Set entry in string map
    fn string_map_set<'c>(&self, key: &str, value: Self::Value<'c>) -> Result<(), MachineError>;
}

/// [`Machine`]s, which have array-like types
pub trait HasArray<'a>: Machine<'a> {
    /// Create datum that is an empty array
    fn new_empty_array<'b>(&'b self) -> Result<Self::Datum<'b, 'static>, MachineError>;
    /// Create datum that is an array
    fn new_array<'b, 'c, I>(&'b self, elements: I) -> Result<Self::Datum<'b, 'static>, MachineError>
    where
        I: IntoIterator<Item = Self::Datum<'b, 'c>>,
        for<'d> <Self as Machine<'a>>::Datum<'b, 'static>:
            MaybeArray<Element<'d> = <Self as Machine<'a>>::Datum<'b, 'd>>,
    {
        let datum = self.new_empty_array()?;
        for element in elements {
            datum.array_push(element)?;
        }
        Ok(datum)
    }
}

/// [`Machine`]s, which have string-map-like types
pub trait HasStringMap<'a>: Machine<'a> {
    /// Create datum that is an empty string map
    fn new_empty_string_map<'b>(&'b self) -> Result<Self::Datum<'b, 'static>, MachineError>;
    /// Create datum that is a string map
    fn new_string_map<'b, 'c, 'd, I>(
        &'b self,
        entries: I,
    ) -> Result<Self::Datum<'b, 'static>, MachineError>
    where
        I: IntoIterator<Item = (&'d str, Self::Datum<'b, 'c>)>,
        for<'e> <Self as Machine<'a>>::Datum<'b, 'static>:
            MaybeStringMap<Value<'e> = <Self as Machine<'a>>::Datum<'b, 'e>>,
    {
        let datum = self.new_empty_string_map()?;
        for (key, value) in entries {
            datum.string_map_set(key, value)?;
        }
        Ok(datum)
    }
}
