use async_trait::async_trait;
use std::convert::Infallible;
use std::error::Error as StdError;
use std::fmt;

use crate::http::StatusCode;
use crate::{Depot, Request, Response, Writer};

type BoxError = Box<dyn std::error::Error + Send + Sync>;

/// Errors that can happen inside salvo.
pub struct Error {
    inner: BoxError,
}

impl Error {
    #[inline]
    pub fn new<E: Into<BoxError>>(err: E) -> Error {
        Error { inner: err.into() }
    }
}

impl fmt::Debug for Error {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        // Skip showing worthless `Error { .. }` wrapper.
        fmt::Debug::fmt(&self.inner, f)
    }
}

impl fmt::Display for Error {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        fmt::Display::fmt(&self.inner, f)
    }
}

impl StdError for Error {}

impl From<Infallible> for Error {
    fn from(infallible: Infallible) -> Error {
        match infallible {}
    }
}

#[async_trait]
impl Writer for Error {
    #[inline]
    async fn write(mut self, _req: &mut Request, _depot: &mut Depot, res: &mut Response) {
        res.set_status_code(StatusCode::INTERNAL_SERVER_ERROR);
    }
}

#[test]
fn error_size_of() {
    assert_eq!(::std::mem::size_of::<Error>(), ::std::mem::size_of::<usize>() * 2);
}

#[cfg(feature = "anyhow")]
#[async_trait]
impl Writer for ::anyhow::Error {
    async fn write(mut self, _req: &mut Request, _depot: &mut Depot, res: &mut Response) {
        res.set_http_error(
            crate::http::errors::InternalServerError()
                .with_summary("anyhow error message")
                .with_detail(&self.to_string()),
        );
    }
}

#[cfg(test)]
mod tests {
    use crate::http::*;

    use super::*;

    #[tokio::test]
    #[cfg(feature = "anyhow")]
    async fn test_anyhow() {
        let mut request = Request::default();
        let mut response = Response::default();
        let mut depot = Depot::new();

        let err: ::anyhow::Error = Error::new("detail message").into();
        err.write(&mut request, &mut depot, &mut response).await;
        assert_eq!(response.status_code(), Some(StatusCode::INTERNAL_SERVER_ERROR));
    }

    #[tokio::test]
    async fn test_error() {
        let mut request = Request::default();
        let mut response = Response::default();
        let mut depot = Depot::new();

        let err = Error::new("detail message");
        err.write(&mut request, &mut depot, &mut response).await;
        assert_eq!(response.status_code(), Some(StatusCode::INTERNAL_SERVER_ERROR));
    }
}
