#[test]
fn test_empty_hash() {
    let mut empty_hash = [0u8; 64];
    super::hash::sha512(&mut empty_hash, &[]);
    #[rustfmt::skip]
    let expected: [u8; 64] = [
        0xcf, 0x83, 0xe1, 0x35, 0x7e, 0xef, 0xb8, 0xbd,
        0xf1, 0x54, 0x28, 0x50, 0xd6, 0x6d, 0x80, 0x07,
        0xd6, 0x20, 0xe4, 0x05, 0x0b, 0x57, 0x15, 0xdc,
        0x83, 0xf4, 0xa9, 0x21, 0xd3, 0x6c, 0xe9, 0xce,
        0x47, 0xd0, 0xd1, 0x3c, 0x5d, 0x85, 0xf2, 0xb0,
        0xff, 0x83, 0x18, 0xd2, 0x87, 0x7e, 0xec, 0x2f,
        0x63, 0xb9, 0x31, 0xbd, 0x47, 0x41, 0x7a, 0x81,
        0xa5, 0x38, 0x32, 0x7a, 0xf9, 0x27, 0xda, 0x3e,
    ];
    // println!("{:?}", empty_hash[..8]);
    // println!("{:?}", expected[..8]);
    assert_eq!(empty_hash[..16], expected[..16]);
}

#[test]
fn test_empty_hash_components() {
    let mut hash = crate::hash::Hash::new();
    // hash.update(&[]);
    let digest = hash.finalize();
    #[rustfmt::skip]
    let expected: [u8; 64] = [
        0xcf, 0x83, 0xe1, 0x35, 0x7e, 0xef, 0xb8, 0xbd,
        0xf1, 0x54, 0x28, 0x50, 0xd6, 0x6d, 0x80, 0x07,
        0xd6, 0x20, 0xe4, 0x05, 0x0b, 0x57, 0x15, 0xdc,
        0x83, 0xf4, 0xa9, 0x21, 0xd3, 0x6c, 0xe9, 0xce,
        0x47, 0xd0, 0xd1, 0x3c, 0x5d, 0x85, 0xf2, 0xb0,
        0xff, 0x83, 0x18, 0xd2, 0x87, 0x7e, 0xec, 0x2f,
        0x63, 0xb9, 0x31, 0xbd, 0x47, 0x41, 0x7a, 0x81,
        0xa5, 0x38, 0x32, 0x7a, 0xf9, 0x27, 0xda, 0x3e,
    ];
    // println!("{:?}", empty_hash[..8]);
    // println!("{:?}", expected[..8]);
    assert_eq!(digest[..16], expected[..16]);
}

#[test]
fn test_non_empty() {
    let mut digest = [0u8; 64];

    // short example
    super::hash::sha512(&mut digest, &"salty".as_bytes());
    let expected: [u8; 64] = [
        0x34, 0x69, 0x63, 0xb3, 0xd8, 0x46, 0x88, 0x5f, 0x4a, 0x5c, 0x18, 0x60, 0x51, 0xd0,
        0x09, 0x03, 0x8b, 0x82, 0xc7, 0x48, 0xfb, 0xec, 0xc2, 0x8c, 0x2c, 0x79, 0x27, 0xc5,
        0xf8, 0x80, 0xe2, 0xb3, 0x60, 0x1b, 0x0e, 0x83, 0x4c, 0xbf, 0xcf, 0xd6, 0x35, 0x7b,
        0xec, 0x8e, 0x01, 0x82, 0xa8, 0xc4, 0x90, 0x0f, 0xbe, 0xa2, 0x7b, 0x06, 0x0e, 0x5b,
        0xa8, 0xc3, 0x1d, 0x3b, 0xc2, 0xbc, 0xc3, 0x34,
    ];
    assert_eq!(digest[..16], expected[..16]);

    // longer example (>= 122 bytes)
    let example = "saltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysalty";
    super::hash::sha512(&mut digest, &example.as_bytes());
    // use core::array::FixedSizeArray;
    let expected: [u8; 64] = [
        0x57, 0xd3, 0x71, 0x18, 0x15, 0x72, 0x91, 0xbe, 0x02, 0x6b, 0x72, 0x46, 0x81, 0xb4,
        0xcd, 0xb3, 0xb6, 0xc3, 0x18, 0x78, 0x0e, 0x28, 0x95, 0x85, 0xb5, 0xed, 0x69, 0x8f,
        0x35, 0x4d, 0x54, 0xc9, 0x1c, 0xfd, 0x6e, 0xd3, 0xfd, 0xf8, 0xb6, 0x0f, 0x6e, 0x37,
        0x41, 0x16, 0x9a, 0x3b, 0xbc, 0xb9, 0xc1, 0x67, 0x99, 0xf8, 0x45, 0x0c, 0xad, 0x16,
        0x59, 0x18, 0xb9, 0xe9, 0xcb, 0x51, 0x4a, 0x38,
    ];
    assert!(digest.iter().zip(expected.iter()).all(|(a, b)| a == b));
    // assert!(digest.iter().zip(expected.iter()).all_equal());
    // assert!(&digest.as_slice() == &expected.as_slice());
    // assert!(&digest.as_slice() == &expected.as_slice());
    assert_eq!(digest[..16], expected[..16]);
}

#[test]
fn test_non_empty_components() {
    let mut hash = crate::hash::Hash::new();

    // short example
    hash.update(&"salty".as_bytes());
    let expected: [u8; 64] = [
        0x34, 0x69, 0x63, 0xb3, 0xd8, 0x46, 0x88, 0x5f, 0x4a, 0x5c, 0x18, 0x60, 0x51, 0xd0,
        0x09, 0x03, 0x8b, 0x82, 0xc7, 0x48, 0xfb, 0xec, 0xc2, 0x8c, 0x2c, 0x79, 0x27, 0xc5,
        0xf8, 0x80, 0xe2, 0xb3, 0x60, 0x1b, 0x0e, 0x83, 0x4c, 0xbf, 0xcf, 0xd6, 0x35, 0x7b,
        0xec, 0x8e, 0x01, 0x82, 0xa8, 0xc4, 0x90, 0x0f, 0xbe, 0xa2, 0x7b, 0x06, 0x0e, 0x5b,
        0xa8, 0xc3, 0x1d, 0x3b, 0xc2, 0xbc, 0xc3, 0x34,
    ];
    assert_eq!(hash.finalize()[..16], expected[..16]);

    // longer example (>= 122 bytes)
    let example = "saltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysalty";
    let mut hash = crate::hash::Hash::new();
    hash.update(&example.as_bytes());
    let digest = hash.finalize();

    let expected: [u8; 64] = [
        0x57, 0xd3, 0x71, 0x18, 0x15, 0x72, 0x91, 0xbe, 0x02, 0x6b, 0x72, 0x46, 0x81, 0xb4,
        0xcd, 0xb3, 0xb6, 0xc3, 0x18, 0x78, 0x0e, 0x28, 0x95, 0x85, 0xb5, 0xed, 0x69, 0x8f,
        0x35, 0x4d, 0x54, 0xc9, 0x1c, 0xfd, 0x6e, 0xd3, 0xfd, 0xf8, 0xb6, 0x0f, 0x6e, 0x37,
        0x41, 0x16, 0x9a, 0x3b, 0xbc, 0xb9, 0xc1, 0x67, 0x99, 0xf8, 0x45, 0x0c, 0xad, 0x16,
        0x59, 0x18, 0xb9, 0xe9, 0xcb, 0x51, 0x4a, 0x38,
    ];
    assert!(digest.iter().zip(expected.iter()).all(|(a, b)| a == b));
    assert_eq!(digest[..16], expected[..16]);

    // longer example in two update calls
    let example = "saltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysaltysalty".as_bytes();
    let mut hash = crate::hash::Hash::new();

    hash.update(&example[..123]);
    hash.update(&example[123..]);
    let digest = hash.finalize();

    assert!(digest.iter().zip(expected.iter()).all(|(a, b)| a == b));
}

#[test]
fn test_afl_outputs() {
    let mut digest = [0u8; 64];

    let example = [
        0x17u8, 0x0, 0xfd, 0x0, 0x0, 0x1, 0x0, 0xfb, 0x0, 0xf2, 0x0, 0x0, 0x0, 0x0, 0x61, 0xff,
        0xff, 0xff, 0x8, 0x0, 0xeb, 0xff, 0x0, 0xe, 0x0, 0xf2, 0xff, 0xff, 0xff, 0x0, 0x0,
        0xeb, 0xff, 0x0, 0x0, 0x0, 0x0, 0x17, 0x0, 0x0, 0x0, 0x7f, 0xff, 0xf2, 0x0, 0x0, 0x0,
        0x0, 0x61, 0xff, 0xff, 0xff, 0x6e, 0x69, 0x63, 0x6b, 0x72, 0x61, 0x79, 0xa, 0x17, 0x0,
        0xf, 0x0, 0xf2, 0x0, 0x0, 0x0, 0x0, 0x61, 0xff, 0xff, 0xff, 0x0, 0x0, 0xeb, 0xff, 0x0,
        0x0, 0x0, 0x0, 0x17, 0x0, 0x0, 0x16, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
        0xff, 0x0, 0x0, 0x14, 0x0, 0xd, 0x0, 0x0, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a,
        0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x2c, 0x4a, 0xa,
        0x4a, 0x4a, 0x4a, 0x4a, 0x4a, 0x0,
    ];
    super::hash::sha512(&mut digest, &example);
}

#[test]
fn test_bit_fiddling() {
    let b: i64 = 0;
    let mask = !(b - 1);
    assert_eq!(mask, 0);

    let b: i64 = 1;
    let mask = !(b - 1);
    assert_eq!(mask as u64, 0xFFFFFFFFFFFFFFFF);
}

#[test]
fn test_keygen() {
    use crate::sign;

    // test case randomly generated with python-ed25519
    // b"=\rGvx\xdbr\x83*\xb5'n\x0f\x92i\xc7\x9c \x9awG\xe4\x8e\x92f\x8f\xe3\xe4\x17%=\xe8"
    let seed: sign::SeedBuffer = [
        0x3d, 0x0d, 0x47, 0x76, 0x78, 0xdb, 0x72, 0x83,
        0x2a, 0xb5, 0x27, 0x6e, 0x0f, 0x92, 0x69, 0xc7,
        0x9c, 0x20, 0x9a, 0x77, 0x47, 0xe4, 0x8e, 0x92,
        0x66, 0x8f, 0xe3, 0xe4, 0x17, 0x25, 0x3d, 0xe8,
    ];

    let secret_key = sign::generate_key(&sign::Seed(seed));

    // from python-ed25519
    let public_key = sign::PublicKey([
        0x9f, 0xd5, 0x81, 0x0e, 0x49, 0x38, 0x98, 0xde,
        0xe1, 0x09, 0x05, 0x0f, 0x19, 0x69, 0x36, 0x57,
        0xcb, 0xbb, 0x11, 0x18, 0xf9, 0xa2, 0xd9, 0xbb,
        0xfd, 0x0b, 0x74, 0x98, 0xee, 0xae, 0x42, 0xe4,
    ]);

    assert_eq!(
        secret_key.public_key,
        public_key,
    );

    // just for kicks
    assert!(
        secret_key.public_key
        !=
        sign::PublicKey([0u8; 32])
    );

    assert!(
        sign::generate_key(&sign::Seed([0u8; 32])).public_key
        !=
        public_key
    );

}

#[test]
fn test_signature() {
    use crate::sign;
    let seed: sign::SeedBuffer = [
        0x35, 0xb3, 0x07, 0x76, 0x17, 0x9a, 0x78, 0x58,
        0x34, 0xf0, 0x4c, 0x82, 0x88, 0x59, 0x5d, 0xf4,
        0xac, 0xa1, 0x0b, 0x33, 0xaa, 0x12, 0x10, 0xad,
        0xec, 0x3e, 0x82, 0x47, 0x25, 0x3e, 0x6c, 0x65,
    ];

    let secret_key = sign::generate_key(&sign::Seed(seed));

    let data = "salty!".as_bytes();

    let R_expected = [
        0xec, 0x97, 0x27, 0x40, 0x07, 0xe7, 0x08, 0xc6,
        0xd1, 0xee, 0xd6, 0x01, 0x9f, 0x5d, 0x0f, 0xcb,
        0xe1, 0x8a, 0x67, 0x70, 0x8d, 0x17, 0x92, 0x4b,
        0x95, 0xdb, 0x7e, 0x35, 0xcc, 0xaa, 0x06, 0x3a,
    ];

    let S_expected = [
        0xb8, 0x64, 0x8c, 0x9b, 0xf5, 0x48, 0xb0, 0x09,
        0x90, 0x6f, 0xa1, 0x31, 0x09, 0x0f, 0xfe, 0x85,
        0xa1, 0x7e, 0x89, 0x99, 0xb8, 0xc4, 0x2c, 0x97,
        0x32, 0xf9, 0xa6, 0x44, 0x2a, 0x17, 0xbc, 0x09,
    ];

    let (R, S) = sign::sign(&secret_key, &data);

    assert_eq!(R, R_expected);
    assert_eq!(S, S_expected);
}

