use num_enum::{IntoPrimitive, TryFromPrimitive, TryFromPrimitiveError};
use std::{
    collections::hash_map::DefaultHasher,
    hash::{Hash, Hasher},
};

type UnknownOpcodeError = TryFromPrimitiveError<Opcode>;

/// Assembly. Lower level representation.
#[derive(Debug, Copy, Clone, Hash, Eq, PartialEq, IntoPrimitive, TryFromPrimitive)]
#[cfg_attr(feature = "with_serde", derive(serde::Serialize, serde::Deserialize))]
#[repr(u32)]
pub enum Opcode {
    // Primitives:
    Plane = 0,          // vec4
    Sphere = 1,         // center: vec3, radius: f32
    Capsule = 2,        // p0: vec3, p1: vec3, radius: f32
    TaperedCapsule = 3, // p0: vec3, r0: f32, p1: vec3, r0: f32

    Material = 4, // rgb: vec3

    // Combinators:
    Union = 5,
    UnionSmooth = 6,
    Subtract = 7,
    SubtractSmooth = 8,
    Intersect = 9,
    IntersectSmooth = 10,

    // Transforms:
    PushTranslation = 11,
    PushRotation = 12,
    PopTransform = 13,
    PushScale = 14,
    PopScale = 15,

    End = 16,

    RoundedBox = 17,      // half_size: vec3, radius: f32
    BiconvexLens = 18,    // lower_sagitta, upper_sagitta, chord
    RoundedCylinder = 19, // cylinder_radius, half_height, rounding_radius
    Torus = 20,           // big_r, small_r
    TorusSector = 21,     // big_r, small_r, sin_half_angle, cos_half_angle
    Cone = 22,            // radius, height
}

pub fn constants_hash(constants: &[f32]) -> u64 {
    let mut s = DefaultHasher::new();
    for &c in constants {
        let v: u32 = c.to_bits();
        v.hash(&mut s);
    }

    s.finish()
}

pub fn opcodes_hash(opcodes: &[Opcode]) -> u64 {
    let mut s = DefaultHasher::new();
    opcodes.hash(&mut s);
    s.finish()
}

/// Represents a signed distance field function as a program with a constant pool and opcodes.
#[derive(Debug, Clone, Default, PartialEq)]
#[cfg_attr(feature = "with_serde", derive(serde::Serialize, serde::Deserialize))]
pub struct Program {
    pub constants: Vec<f32>,
    pub opcodes: Vec<Opcode>,
}

impl Program {
    #[must_use]
    pub fn with_constants(&self, constants: Vec<f32>) -> Self {
        Self {
            constants,
            opcodes: self.opcodes.clone(),
        }
    }

    pub fn constant_hash(&self) -> u64 {
        constants_hash(&self.constants)
    }

    pub fn program_hash(&self) -> u64 {
        opcodes_hash(&self.opcodes)
    }

    pub fn full_hash(&self) -> u64 {
        self.program_hash() ^ self.constant_hash()
    }

    #[cfg(feature = "with_bincode")]
    pub fn as_bytes(&self) -> Result<Vec<u8>, std::boxed::Box<bincode::ErrorKind>> {
        bincode::serialize(self)
    }

    #[cfg(feature = "with_bincode")]
    pub fn from_bytes(bytes: &[u8]) -> Result<Self, std::boxed::Box<bincode::ErrorKind>> {
        bincode::deserialize(bytes)
    }

    pub fn from_raw(opcodes: &[u32], constants: &[f32]) -> Result<Self, UnknownOpcodeError> {
        // We use collect to convert from a Vec<Result<..>> to a Result<Vec<..>>. Neat!
        let opcodes = opcodes
            .iter()
            .map(|opcode| Opcode::try_from(*opcode))
            .collect::<Result<Vec<Opcode>, _>>()?;
        Ok(Self {
            opcodes,
            constants: constants.to_vec(),
        })
    }

    pub fn as_raw(&self) -> (Vec<u32>, Vec<f32>) {
        let opcodes = self
            .opcodes
            .iter()
            .map(|&opcode| opcode.into())
            .collect::<Vec<u32>>();
        (opcodes, self.constants.clone())
    }

    pub(crate) fn constant_push_vec2(&mut self, v: impl Into<[f32; 2]>) {
        self.constants.extend(&v.into());
    }

    pub(crate) fn constant_push_vec3(&mut self, v: impl Into<[f32; 3]>) {
        self.constants.extend(&v.into());
    }

    pub(crate) fn constant_push_vec4(&mut self, v: impl Into<[f32; 4]>) {
        self.constants.extend(&v.into());
    }

    pub fn disassemble(&self) -> String {
        // Moved it to the compiler file, fits better there.
        crate::compiler::disassemble(&self.opcodes, &self.constants)
            .unwrap_or_else(|e| format!("(failed to disassemble: {:?}", e))
    }
}
