use macaw::{Vec3, Vec4};

#[derive(Debug, Copy, Clone)]
#[cfg_attr(feature = "with_serde", derive(serde::Serialize, serde::Deserialize))]
pub struct Material {
    /// [0-1] linear space
    rgb: Vec3,
}

impl Default for Material {
    fn default() -> Self {
        Self::new(Vec3::ONE)
    }
}

impl From<Vec3> for Material {
    fn from(rgb: Vec3) -> Self {
        Self { rgb }
    }
}

impl Material {
    pub fn new(rgb: Vec3) -> Self {
        Self { rgb }
    }
    pub fn rgb(&self) -> Vec3 {
        self.rgb
    }
}

pub trait SignedDistance: Copy {
    fn infinity() -> Self;
    fn distance(&self) -> f32;
    fn copy_with_distance(&self, distance: f32) -> Self;
    fn multiply_distance_by(&self, factor: f32) -> Self;
    fn material(&self) -> Material;

    fn lerp(&self, b: &Self, t: f32) -> Self;

    fn new_with_distance(material: Material, distance: f32) -> Self;
    fn is_distance_finite(&self) -> bool;
}

impl SignedDistance for f32 {
    fn infinity() -> Self {
        Self::INFINITY
    }

    fn distance(&self) -> f32 {
        *self
    }

    fn material(&self) -> Material {
        Material::new(Vec3::ONE)
    }

    fn copy_with_distance(&self, distance: f32) -> Self {
        distance
    }

    fn multiply_distance_by(&self, factor: f32) -> Self {
        self * factor
    }

    fn new_with_distance(_material: Material, distance: f32) -> Self {
        distance
    }

    fn lerp(&self, b: &Self, t: f32) -> Self {
        //Self(self + (b.0 - self) * t)
        t.mul_add(b - self, *self)
    }

    fn is_distance_finite(&self) -> bool {
        self.is_finite()
    }
}

/// r, g, b, distance
#[derive(Copy, Clone, Debug, PartialEq)]
pub struct RgbWithDistance(Vec4);

impl SignedDistance for RgbWithDistance {
    fn infinity() -> Self {
        Self(Vec4::new(1.0, 1.0, 1.0, std::f32::INFINITY))
    }

    fn distance(&self) -> f32 {
        self.0.w
    }

    fn material(&self) -> Material {
        Material::new(self.0.truncate())
    }

    fn copy_with_distance(&self, distance: f32) -> Self {
        let mut n = *self;
        n.0.w = distance;
        n
    }

    fn multiply_distance_by(&self, factor: f32) -> Self {
        Self(self.0.truncate().extend(self.0.w * factor))
    }

    fn new_with_distance(material: Material, distance: f32) -> Self {
        Self(material.rgb().extend(distance))
    }

    fn lerp(&self, b: &Self, t: f32) -> Self {
        Self(self.0.lerp(b.0, t))
    }

    fn is_distance_finite(&self) -> bool {
        self.0.is_finite()
    }
}
