//! [![crates.io version](https://img.shields.io/crates/v/safina.svg)](https://crates.io/crates/safina)
//! [![license: Apache 2.0](https://gitlab.com/leonhard-llc/safina-rs/-/raw/main/license-apache-2.0.svg)](http://www.apache.org/licenses/LICENSE-2.0)
//! [![unsafe forbidden](https://gitlab.com/leonhard-llc/safina-rs/-/raw/main/unsafe-forbidden-success.svg)](https://github.com/rust-secure-code/safety-dance/)
//! [![pipeline status](https://gitlab.com/leonhard-llc/safina-rs/badges/main/pipeline.svg)](https://gitlab.com/leonhard-llc/safina-rs/-/pipelines)
//!
//! A safe Rust async runtime.
//!
//! # Features
//! - `forbid(unsafe_code)`
//! - Depends only on `std`
//! - Good test coverage (>95%)
//!
//! # Limitations
//! - Building on `stable` requires [`once_cell`](https://crates.io/crates/once_cell)
//!   crate which contains some unsafe code.
//!   This is necessary until
//!   [`std::lazy::OnceCell`](https://doc.rust-lang.org/std/lazy/struct.OnceCell.html)
//!   is stable.
//! - Allocates memory.  You can avoid allocations by using advanced functions, like
//!   [`safina_executor::spawn_unpin`](https://docs.rs/safina-executor/latest/safina_executor/fn.spawn_unpin.html).
//! - Not optimized
//!
//! # Documentation
//! <https://docs.rs/safina>
//!
//! [`safina_async_test`](https://crates.io/crates/safina_async_test)
//! has an `#[async_test]` macro for running `async fn` test functions.
//!
//! # Examples
//! ```rust
//! # fn f() {
//! let executor = safina::executor::Executor::default();
//! let (sender, receiver) = std::sync::mpsc::channel();
//! executor.spawn(async move {
//!     sender.send(()).unwrap();
//! });
//! receiver.recv().unwrap();
//! # }
//! ```
//!
//! ```rust
//! # async fn prepare_request() -> Result<(), std::io::Error> { Ok(()) }
//! # async fn execute_request() -> Result<(), std::io::Error> { Ok(()) }
//! # fn f() -> Result<(), std::io::Error> {
//! let result = safina::executor::block_on(async {
//!     prepare_request().await?;
//!     execute_request().await
//! })?;
//! # Ok(())
//! # }
//! ```
//!
//! # Alternatives
//! - [`smol`](https://crates.io/crates/smol)
//!   - Popular
//!   - Contains generous amounts of `unsafe` code
//! - [`async-std`](https://crates.io/crates/async-std)
//!   - Very popular
//!   - Contains generous amounts of `unsafe` code
//! - [`futures`](https://crates.io/crates/futures)
//!   - Very popular
//!   - Contains generous amounts of `unsafe` code
//! - [`tokio`](https://crates.io/crates/tokio)
//!   - Very popular
//!   - Fast
//!   - Internally extremely complicated
//!   - Full of `unsafe`
//! - [`bastion`](https://crates.io/crates/bastion)
//!   - Generous amounts of `unsafe` code
//! - [`nostd_async`](https://crates.io/crates/nostd_async)
//!
//! # Changelog
//! - v0.3.0
//!   - Move structs into sub-modules.
//!   - Replace `Promise` with `oneshot`, `OneSender`, and `Receiver` that supports async and blocking reads.
//!   - `schedule_blocking` to return new `sync::Receiver`.
//! - v0.2.1 - Update docs.
//! - v0.2.0
//!   - `Executor::new` and `Executor::with_name` to return `Result`.
//!   - `ThreadPool::new` to return `Result`.
//!   - `ThreadPool::try_schedule` to return an error when it fails to restart panicked threads.
//!   - `ThreadPool::schedule` to handle failure starting replacement threads.
//! - v0.1.10 - `block_on` functions to take futures that are not `Send`.
//! - v0.1.9 - Use `once_cell` by default.
//! - v0.1.8 - Support stable with rust 1.51 and `once_cell`.
//! - v0.1.7 - Add [`safina-net`](https://crates.io/crates/safina-net)
//! - v0.1.6 - Use [`safina-executor`](https://crates.io/crates/safina-executor) v0.1.3 API
//! - v0.1.5 - Add [`safina_sync::Mutex`](https://docs.rs/safina-sync/latest/safina_sync/struct.Mutex.html)
//! - v0.1.4 - Upgrade to new safina-executor version which removes need for `Box::pin`.
//! - v0.1.3 - Update docs
//! - v0.1.2 - Renamed `safina` crate to `safina-executor`.
//!   Added new `safina` crate with re-exports, examples, and integration tests.
//! - v0.1.1 - Add badges to readme
//! - v0.1.0 - First published version
//!
//! # TO DO
//! - Add `init` function that makes an executor and starts the timer thread.
//! - Add an `#[async_main]` macro
//!
//! # Release Process
//! 1. Edit `Cargo.toml` and bump version number.
//! 1. Run `./release.sh`
#![forbid(unsafe_code)]

pub mod executor {
    pub use safina_executor::*;
}
pub mod net {
    pub use safina_net::*;
}
pub mod select {
    pub use safina_select::*;
}
pub mod sync {
    pub use safina_sync::*;
}
pub mod timer {
    pub use safina_timer::*;
}
