#![forbid(unsafe_code)]

use crate::{schedule_wake, TimerThreadNotStarted};
use core::future::Future;
use core::pin::Pin;
use core::task::{Context, Poll, Waker};
use std::sync::{Arc, Mutex};
use std::time::Instant;

/// A future that completes after the specified time.
///
/// It is returned by [`sleep_until`] and [`sleep_for`].
#[must_use = "futures stay idle unless you await them"]
pub struct SleepFuture {
    deadline: std::time::Instant,
    waker: Arc<Mutex<Option<Waker>>>,
}
impl SleepFuture {
    pub fn new(deadline: Instant) -> Self {
        Self {
            deadline,
            waker: Arc::new(Mutex::new(None)),
        }
    }
}
impl Future for SleepFuture {
    type Output = Result<(), TimerThreadNotStarted>;

    fn poll(self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<Self::Output> {
        if self.deadline < std::time::Instant::now() {
            return Poll::Ready(Ok(()));
        }
        let old_waker = self.waker.lock().unwrap().replace(cx.waker().clone());
        if old_waker.is_none() {
            schedule_wake(self.deadline, self.waker.clone())?;
        }
        Poll::Pending
    }
}

/// Returns after `deadline`.
///
/// # Panics
/// Panics if [`start_timer_thread()`](fn.start_timer_thread.html) has not been called.
/// If you need to handle this error, use [`SleepFuture::new`](struct.SleepFuture.html).
pub async fn sleep_until(deadline: std::time::Instant) {
    SleepFuture::new(deadline).await.unwrap();
}

/// Returns `duration` time from now.
///
/// # Panics
/// Panics if [`start_timer_thread()`](fn.start_timer_thread.html) has not been called.
/// If you need to handle this error, use [`SleepFuture::new`](struct.SleepFuture.html).
pub async fn sleep_for(duration: core::time::Duration) {
    SleepFuture::new(Instant::now() + duration).await.unwrap();
}
