use crate::{timer_thread, ScheduledWake, TimerThreadNotStarted};
use core::ops::Range;
use core::sync::atomic::AtomicBool;
use core::task::Waker;
use core::time::Duration;
use std::sync::{Arc, Mutex};
use std::time::Instant;

#[derive(Clone)]
pub struct FakeWaker {
    called_flag: Arc<AtomicBool>,
}
impl FakeWaker {
    #[must_use]
    pub fn new(called_flag: &Arc<AtomicBool>) -> Self {
        Self {
            called_flag: called_flag.clone(),
        }
    }
    #[must_use]
    pub fn into_waker(self) -> Waker {
        std::task::Waker::from(Arc::new(self))
    }
}
impl std::task::Wake for FakeWaker {
    fn wake(self: Arc<Self>) {
        if self
            .called_flag
            .fetch_or(true, std::sync::atomic::Ordering::AcqRel)
        {
            panic!("wake already called");
        }
    }
}

/// # Panics
/// Panics if the time elapsed since `before` is outside of `range_ms`.
pub fn expect_elapsed(before: Instant, range_ms: Range<u64>) {
    assert!(!range_ms.is_empty(), "invalid range {:?}", range_ms);
    let elapsed = before.elapsed();
    let duration_range = Duration::from_millis(range_ms.start)..Duration::from_millis(range_ms.end);
    assert!(
        duration_range.contains(&elapsed),
        "{:?} elapsed, out of range {:?}",
        elapsed,
        duration_range
    );
}

fn make_scheduled_wakes() -> (ScheduledWake, ScheduledWake, ScheduledWake, ScheduledWake) {
    let now = Instant::now();
    (
        ScheduledWake {
            instant: now - Duration::from_millis(1),
            waker: Arc::new(Mutex::new(None)),
        },
        ScheduledWake {
            instant: now,
            waker: Arc::new(Mutex::new(None)),
        },
        ScheduledWake {
            instant: now + Duration::from_millis(1),
            waker: Arc::new(Mutex::new(None)),
        },
        ScheduledWake {
            instant: now + Duration::from_millis(1),
            waker: Arc::new(Mutex::new(None)),
        },
    )
}

#[test]
fn test_scheduled() {
    use core::cmp::Ordering;
    let (sw1, sw2, sw3, sw3b) = make_scheduled_wakes();
    assert!(format!("{:?}", sw1).starts_with("ScheduledWake {"));
    assert_eq!(sw3, sw3b);
    assert_ne!(sw3, sw2);
    assert_eq!(Some(Ordering::Less), PartialOrd::partial_cmp(&sw1, &sw2));
    assert_eq!(Some(Ordering::Equal), PartialOrd::partial_cmp(&sw1, &sw1));
    assert_eq!(Some(Ordering::Greater), PartialOrd::partial_cmp(&sw2, &sw1));
    assert_eq!(Ordering::Less, Ord::cmp(&sw1, &sw2));
    assert_eq!(Ordering::Equal, Ord::cmp(&sw1, &sw1));
    assert_eq!(Ordering::Greater, Ord::cmp(&sw2, &sw1));
}

#[test]
fn test_error() {
    let e1 = TimerThreadNotStarted {};
    let e2 = TimerThreadNotStarted {};
    assert_eq!("TimerThreadNotStarted", format!("{:?}", e1));
    assert_eq!(e1, e2);
    assert_eq!("TimerThreadNotStarted", format!("{}", e1));
}

#[test]
#[should_panic]
fn test_disconnected() {
    let (sender, receiver) = std::sync::mpsc::sync_channel(0);
    std::thread::spawn(move || {
        sender
            .send(ScheduledWake {
                instant: Instant::now() + Duration::from_secs(1),
                waker: Arc::new(Mutex::new(None)),
            })
            .unwrap();
    });
    timer_thread(receiver);
}
