# safina-threadpool

[![crates.io version](https://img.shields.io/crates/v/safina-threadpool.svg)](https://crates.io/crates/safina-threadpool)
[![license: Apache 2.0](https://gitlab.com/leonhard-llc/safina-rs/-/raw/main/license-apache-2.0.svg)](http://www.apache.org/licenses/LICENSE-2.0)
[![unsafe forbidden](https://gitlab.com/leonhard-llc/safina-rs/-/raw/main/unsafe-forbidden-success.svg)](https://github.com/rust-secure-code/safety-dance/)
[![pipeline status](https://gitlab.com/leonhard-llc/safina-rs/badges/main/pipeline.svg)](https://gitlab.com/leonhard-llc/safina-rs/-/pipelines)

A threadpool.

You can use it alone or with [`safina`](https://crates.io/crates/safina),
a safe async runtime.

## Features
- Add closures or `FnOnce` to the pool
- One of the pool's threads will execute it
- Automatically restarts panicked threads
- Handles hitting process thread limit.
- `forbid(unsafe_code)`
- Depends only on `std`
- 100% test coverage

## Limitations
- Allocates memory
- Not optimized

## Documentation
<https://docs.rs/safina-threadpool>

## Examples
```rust
let pool =
    safina_threadpool::ThreadPool::new("worker", 2).unwrap();
let receiver = {
    let (sender, receiver) =
        std::sync::mpsc::channel();
    for data in data_source {
        let sender_clone = sender.clone();
        pool.schedule(
            move || process_data(data, sender_clone));
    }
    receiver
};
let results: Vec<ProcessResult> =
    receiver.iter().collect();
// ...
```

## Alternatives
- [`blocking`](https://crates.io/crates/blocking)
  - Popular
  - A little `unsafe` code
- [`threadpool`](https://crates.io/crates/threadpool)
  - Popular
  - Well maintained
  - Dependencies have `unsafe` code
- [`futures-executor`](https://crates.io/crates/futures-executor)
  - Very popular
  - Full of `unsafe`
- [`scoped_threadpool`](https://crates.io/crates/scoped_threadpool)
  - Popular
  - Contains `unsafe` code
- [`scheduled-thread-pool`](https://crates.io/crates/scheduled-thread-pool)
  - Used by a popular connection pool library
  - Dependencies have `unsafe` code
- [`workerpool`](https://crates.io/crates/workerpool)
  - Dependencies have `unsafe` code
- [`threads_pool`](https://crates.io/crates/threads_pool)
  - Full of `unsafe`
- [`thread-pool`](https://crates.io/crates/thread-pool)
  - Old
  - Dependencies have `unsafe` code
- [`tasque`](https://crates.io/crates/tasque)
  - Dependencies have `unsafe` code
- [`fast-threadpool`](https://crates.io/crates/fast-threadpool)
  - Dependencies have `unsafe` code
- [`blocking-permit`](https://crates.io/crates/blocking-permit)
  - Full of `unsafe`
- [`rayon-core`](https://crates.io/crates/rayon-core)
  - Full of `unsafe`

## Changelog
- v0.2.1 - Improve test coverage.
- v0.2.0
  - `ThreadPool::new` to return `Result`.
  - `ThreadPool::try_schedule` to return an error when it fails to restart panicked threads.
  - `ThreadPool::schedule` to handle failure starting replacement threads.
- v0.1.4 - Stop threads on drop.
- v0.1.3 - Support stable Rust!  Needs 1.51+.
- v0.1.2 - Add another example
- v0.1.1 - Simplified internals and improved documentation.
- v0.1.0 - First release

## TO DO
- Log a warning when all threads panicked.
- Update test coverage.
- Add a public `respawn_threads` function.
- Add a stress test
- Add a benchmark.  See benchmarks in <https://crates.io/crates/executors>
- Add a way for a job to schedule another job on the same thread, with stealing.

## Release Process
1. Edit `Cargo.toml` and bump version number.
1. Run `./release.sh`

License: Apache-2.0
