// These tests don't work on macOS because
// macOS doesn't enforce hard limits on threads/processes.
#![cfg(linux)]

use errno::errno;
use libc::{rlim_t, rlimit, setrlimit};
use permit::Permit;
use safina_threadpool::{NewThreadPoolError, ThreadPool, TryScheduleError};
use std::time::{Duration, Instant};

use safe_lock::SafeLock;
static LOCK: SafeLock = SafeLock::new();

const MAX_THREADS: usize = 10;
const POOL_SIZE: usize = 5;

fn assert_elapsed(before: Instant, range_ms: Range<u64>) {
    assert!(!range_ms.is_empty(), "invalid range {:?}", range_ms);
    let elapsed = before.elapsed();
    let duration_range = Duration::from_millis(range_ms.start)..Duration::from_millis(range_ms.end);
    assert!(
        duration_range.contains(&elapsed),
        "{:?} elapsed, out of range {:?}",
        elapsed,
        duration_range
    );
}

fn sleep_ms(ms: u64) {
    std::thread::sleep(Duration::from_millis(ms));
}

fn set_nproc_limit() -> Result<(), String> {
    let values = rlimit {
        rlim_cur: MAX_THREADS as rlim_t,
        rlim_max: MAX_THREADS as rlim_t,
    };
    if unsafe { setrlimit(libc::RLIMIT_NPROC, &values) } == 0 {
        Ok(())
    } else {
        Err(format!("{}", errno()));
    }
}

fn panic_threads(pool: &ThreadPool, num: usize) {
    let panic_permit = Permit::new();
    for _ in 0..num {
        let sub = panic_permit.new_sub();
        pool.try_schedule(move || {
            while !sub.is_revoked() {
                sleep_ms(10);
            }
            panic!("ignore this panic")
        })
        .unwrap();
    }
    sleep_ms(50);
    drop(panic_permit);
}

fn start_threads(permit: Permit) {
    std::thread::spawn(move || {
        while !permit.is_revoked() {
            let sub = permit.new_sub();
            let _result = std::thread::Builder::new().spawn(move || {
                while !sub.is_revoked() {
                    sleep_ms(10);
                }
            });
            sleep_ms(1);
        }
    });
}

#[test]
fn new_thread_pool_error_spawn() {
    let _guard = LOCK.lock().unwrap();
    set_nproc_limit();
    let err = ThreadPool::new("pool1", MAX_THREADS + 5).unwrap_err();
    assert!(matches!(err, NewThreadPoolError::Spawn(_)));
}

#[test]
fn schedule_retries_thread_start() {
    let _guard = LOCK.lock().unwrap();
    set_nproc_limit();
    let pool = ThreadPool::new("pool1", POOL_SIZE).unwrap();
    panic_threads(&pool, POOL_SIZE);
    let permit = Permit::new();
    let threads_permit = permit.new_sub();
    let before = Instant::now();
    std::thread::spawn(move || {
        sleep_ms(100);
        drop(permit);
    });
    start_threads(threads_permit);
    let (sender, receiver) = std::sync::mpsc::channel();
    pool.schedule(move || {
        sender.send(()).unwrap();
    });
    receiver.recv_timeout(Duration::from_millis(500)).unwrap();
    assert_elapsed(before, 100..200);
}

#[test]
fn try_schedule_error_no_threads() {
    let _guard = LOCK.lock().unwrap();
    set_nproc_limit();
    let pool = ThreadPool::new("pool1", POOL_SIZE).unwrap();
    panic_threads(&pool, POOL_SIZE);
    let permit = Permit::new();
    start_threads(permit.new_sub());
    let result = pool.try_schedule(move || sleep_ms(100));
    assert!(
        matches!(result, Err(TryScheduleError::NoThreads(_))),
        "{:?}",
        result
    );
}

#[test]
fn try_schedule_error_respawn() {
    let _guard = LOCK.lock().unwrap();
    set_nproc_limit();
    let pool = ThreadPool::new("pool1", POOL_SIZE).unwrap();
    panic_threads(&pool, 1);
    let permit = Permit::new();
    start_threads(permit.new_sub());
    let result = pool.try_schedule(move || sleep_ms(100));
    assert!(
        matches!(result, Err(TryScheduleError::Respawn(_))),
        "{:?}",
        result
    );
}
